//
// 04/07/03  <gerstl>	Updated to use signals for wires (SpecC 2.0)

#include "cnst.sh"
#include "typedef.sh"

import "c_handshake";

import "reset";

import "bus";

import "pre_process";
import "cod_12k2";
import "post_process";


/*------------------ I/O instructions of DSP56600 (MOVEM) ---------------- */

#define WS 2       // wait states

interface IDspIO
{
  bit[15:0] read_word(bit[15:0] addr);
  void write_word(bit[15:0] addr, bit[15:0] data);
};

channel DspIO(out   signal bit[15:0] A,        // address
	      inout signal bit[23:0] D,        // data
	      out   signal bit[1]    MCS,      // chip select
	      out   signal bit[1]    nRD,      // control lines
	      out   signal bit[1]    nWR)
  implements IDspIO
{
  bit[15:0] read_word(bit[15:0] addr)
  {
    bit[15:0] data;
    
    waitfor(5 * DSP_CLOCK_PERIOD / 2);
    A = addr;
    MCS = 1;
    waitfor(DSP_CLOCK_PERIOD / 2);          	// 0.5T-4.0 = (4.3ns,-)
    nRD = 0;
    waitfor((2*WS + 1) * DSP_CLOCK_PERIOD / 2);	// (WS+0.5)T-8.5 = (16.5ns,-)
    data = D;
    waitfor(DSP_CLOCK_PERIOD / 2);  
    nRD = 1;        				// nRD pulse width: (WS+0.25)T-3.8 = (17ns,-)
    waitfor(WS * DSP_CLOCK_PERIOD);
    MCS = 0;
    
    return data;
  }
  
  void write_word(bit[15:0] addr, bit[15:0] data)
  {
    waitfor(5 * DSP_CLOCK_PERIOD / 2);
    A = addr;                             
    MCS = 1;
    waitfor(DSP_CLOCK_PERIOD / 4);         	// 0.25T-3.7 = (0.5ns,-)
    nWR = 0;
    waitfor(3 * DSP_CLOCK_PERIOD / 4);		// 0.75T-3.7 = (8.8ns,-)
    D = data;
    waitfor((WS + 1) * DSP_CLOCK_PERIOD);
    nWR = 1;        				// nWR pulse width: 1.5T-5.7 = (19.3ns,-)
    waitfor(WS * DSP_CLOCK_PERIOD / 2);
    MCS = 0;
  }
};


/*------------------ DSP RTOS (application layer) ----------------- */

channel DspBusDriver(out   signal bit[15:0] A,        // address
		     inout signal bit[23:0] D,        // data
		     out   signal bit[1]    MCS,      // chip select
		     out   signal bit[1]    nRD,      // control lines
		     out   signal bit[1]    nWR,
		     i_receive intC)
  implements IBus
{
  DspIO io(A, D, MCS, nRD, nWR);

  
  void sendWord(Word16 data, int addr)
  {
    intC.receive();
    waitfor( 5 * DSP_CLOCK_PERIOD );
    io.write_word(addr, data);
  }
  
  void sendWordP(Word16* data, int len, int addr)
  {
    int i;
    
    for (i = 0; i < len; i++) {
      sendWord(data[i], addr);
    }
  }

  void recvWord(Word16* data, int addr)
  {
    intC.receive();
    waitfor( 5 * DSP_CLOCK_PERIOD );
    *data = io.read_word(addr);
  }
  
  void recvWordP(Word16* data, int len, int addr)
  {
    int i;
    
    for (i = 0; i < len; i++) {
      recvWord(&data[i], addr);
    }
  }  
};


behavior IntHandler(i_send ev)
{
  void main(void)
  {
    waitfor( 20 * DSP_CLOCK_PERIOD );
    ev.send();
  }
};


/*------------------ DSP main() program ------------------ */


behavior DspMain (
	      in  bit[SAMPLE_WIDTH-1:0] speech_samples[L_FRAME],
	      in  Flag dtx_mode,
	      in  event new_frame,
	      out unsigned bit[BITS_PER_FRAME-1:0] serial,
	      out DTXctrl txdtx_ctrl,
	      out event serialbits_ready,
	      IBus bus
	      )
  implements Ireset

{
  Word16 speech_frame[L_FRAME];    /* Buffer for preprocessed speech        */
  Word16 prm[PRM_SIZE];            /* Analysis parameters.                  */

  DTXctrl txdtx_ctrl_val;
  
  Flag reset_flag_1;
  Flag reset_flag_2;

  Word16 syn[L_FRAME];             /* Buffer for synthesis speech (debug)   */

  Pre_Process pre_process(speech_samples, speech_frame, reset_flag_1, 
			  reset_flag_2, new_frame);
  Coder_12k2 coder_12k2(speech_frame, prm, syn, dtx_mode, txdtx_ctrl_val,
			reset_flag_1, bus);
  Post_Process post_process(prm, txdtx_ctrl_val, reset_flag_2, serial,
			    txdtx_ctrl, serialbits_ready);


  void reset(void)
    {
      pre_process.reset();
      coder_12k2.reset();
      post_process.reset();
    }
  


  void main(void)
  {
    while (true)
    {
      /* filter + downscaling      */
      pre_process.main();
	  
	  
      /* Find speech parameters    */
      coder_12k2.main();
	  
	
      /* insert comfort noise and convert parameters to serial bits */
      post_process.main();
    }
  }
};



/*------------------ Processor model ---------------- */


behavior Dsp (  
	      in  bit[SAMPLE_WIDTH-1:0] speech_samples[L_FRAME],
	      in  Flag dtx_mode,
	      in  event new_frame,
	      out unsigned bit[BITS_PER_FRAME-1:0] serial,
	      out DTXctrl txdtx_ctrl,
	      out event serialbits_ready,
	      out   signal bit[15:0] A,        // address
	      inout signal bit[23:0] D,        // data
	      out   signal bit[1]    MCS,      // chip select
	      out   signal bit[1]    nRD,      // control lines
	      out   signal bit[1]    nWR,
	      in    signal bit[1]    intC      // interrupt
	      )
  implements Ireset

{
  c_handshake intCflag;
  
  IntHandler   intChandler(intCflag);
  DspBusDriver busdriver(A, D, MCS, nRD, nWR, intCflag);
  
  DspMain dsp(speech_samples, dtx_mode, new_frame, serial, txdtx_ctrl, 
	      serialbits_ready, busdriver);
  

  void reset(void)
  {
    dsp.reset();
  }
  

  void main(void)
  {
    try{
	dsp.main();
    }
    interrupt(rising intC) {
      intChandler.main();
    }
  }
};
