/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: stereo.sc,v 1.4 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: stereo.sc,v $
 * Revision 1.4  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

# include "global.sh"

# include "frame.sh"
# include "stream.sh"
# include "layer3.sh"

import "huffman";


/*
 * coefficients for intensity stereo processing
 * derived from section 2.4.3.4.9.3 of ISO/IEC 11172-3
 *
 * is_ratio[i] = tan(i * (PI / 12))
 * is_table[i] = is_ratio[i] / (1 + is_ratio[i])
 */
static
mad_fixed_t const is_table[7] = {
  MAD_F(0x00000000) /* 0.000000000 */,
    MAD_F(0x0361962f) /* 0.211324865 */,
    MAD_F(0x05db3d74) /* 0.366025404 */,
    MAD_F(0x08000000) /* 0.500000000 */,
    MAD_F(0x0a24c28c) /* 0.633974596 */,
    MAD_F(0x0c9e69d1) /* 0.788675135 */,
    MAD_F(0x10000000) /* 1.000000000 */
};

/*
 * coefficients for LSF intensity stereo processing
 * derived from section 2.4.3.2 of ISO/IEC 13818-3
 *
 * is_lsf_table[0][i] = (1 / sqrt(sqrt(2)))^(i + 1)
 * is_lsf_table[1][i] = (1 /      sqrt(2)) ^(i + 1)
 */
static
mad_fixed_t const is_lsf_table[2][15] = {
  {
    MAD_F(0x0d744fcd) /* 0.840896415 */,
      MAD_F(0x0b504f33) /* 0.707106781 */,
      MAD_F(0x09837f05) /* 0.594603558 */,
      MAD_F(0x08000000) /* 0.500000000 */,
      MAD_F(0x06ba27e6) /* 0.420448208 */,
      MAD_F(0x05a8279a) /* 0.353553391 */,
      MAD_F(0x04c1bf83) /* 0.297301779 */,
      MAD_F(0x04000000) /* 0.250000000 */,
      MAD_F(0x035d13f3) /* 0.210224104 */,
      MAD_F(0x02d413cd) /* 0.176776695 */,
      MAD_F(0x0260dfc1) /* 0.148650889 */,
      MAD_F(0x02000000) /* 0.125000000 */,
      MAD_F(0x01ae89fa) /* 0.105112052 */,
      MAD_F(0x016a09e6) /* 0.088388348 */,
      MAD_F(0x01306fe1) /* 0.074325445 */
  }, {
    MAD_F(0x0b504f33) /* 0.707106781 */,
      MAD_F(0x08000000) /* 0.500000000 */,
      MAD_F(0x05a8279a) /* 0.353553391 */,
      MAD_F(0x04000000) /* 0.250000000 */,
      MAD_F(0x02d413cd) /* 0.176776695 */,
      MAD_F(0x02000000) /* 0.125000000 */,
      MAD_F(0x016a09e6) /* 0.088388348 */,
      MAD_F(0x01000000) /* 0.062500000 */,
      MAD_F(0x00b504f3) /* 0.044194174 */,
      MAD_F(0x00800000) /* 0.031250000 */,
      MAD_F(0x005a827a) /* 0.022097087 */,
      MAD_F(0x00400000) /* 0.015625000 */,
      MAD_F(0x002d413d) /* 0.011048543 */,
      MAD_F(0x00200000) /* 0.007812500 */,
      MAD_F(0x0016a09e) /* 0.005524272 */
  }
};


behavior Stereo(in unsigned int gr,
                struct mad_header header,
                in struct sideinfo_channel si_channel0[2],
                in struct sideinfo_channel si_channel1[2],
                in unsigned char scalefacs[2][2][39],
                mad_fixed_t xr0[576],
                mad_fixed_t xr1[576],
                unsigned char sfbwidth0[39],
                out enum mad_error error)
{
  /*
   * NAMEi:	III_stereo()
   * DESCRIPTION:	perform joint stereo processing on a granule
   */
  enum mad_error III_stereo(unsigned char left_ch_flags,
                            unsigned char left_ch_block_type,
                            unsigned short right_ch_scalefac_compress,
                            unsigned char right_ch_flags,
                            unsigned char right_ch_block_type,
                            unsigned char *right_ch_scalefac,
                            mad_fixed_t xr0[576],
                            mad_fixed_t xr1[576],
                            unsigned char const *sfbwidth)
  {
    short modes[39];
    unsigned int sfbi, l, n, i;
    
    if (left_ch_block_type !=
        right_ch_block_type ||
        (left_ch_flags & mixed_block_flag) !=
        (right_ch_flags & mixed_block_flag))
      return MAD_ERROR_BADSTEREO;
    
    for (i = 0; i < 39; ++i)
      modes[i] = header.mode_extension;
    
    /* intensity stereo */
    
    if (header.mode_extension & I_STEREO) {
      mad_fixed_t *right_xr;
      unsigned int is_pos;
      
      right_xr = xr1;
      
      header.flags |= MAD_FLAG_I_STEREO;
      
      /* first determine which scalefactor bands are to be processed */
      
      if (right_ch_block_type == 2) {
        unsigned int lower, start, max, bound[3], w;
        
        lower = start = max = bound[0] = bound[1] = bound[2] = 0;
        
        sfbi = l = 0;
        
        if (right_ch_flags & mixed_block_flag) {
          while (l < 36) {
            n = sfbwidth[sfbi++];
            
            for (i = 0; i < n; ++i) {
              if (right_xr[i]) {
                lower = sfbi;
                break;
              }
            }
            
            right_xr += n;
            l += n;
          }
          
          start = sfbi;
        }
        
        w = 0;
        while (l < 576) {
          n = sfbwidth[sfbi++];
          
          for (i = 0; i < n; ++i) {
            if (right_xr[i]) {
              max = bound[w] = sfbi;
              break;
            }
          }
          
          right_xr += n;
          l += n;
          w = (w + 1) % 3;
        }
        
        if (max)
          lower = start;
        
        /* long blocks */
        
        for (i = 0; i < lower; ++i)
          modes[i] = header.mode_extension & ~I_STEREO;
        
        /* short blocks */
        
        w = 0;
        for (i = start; i < max; ++i) {
          if (i < bound[w])
            modes[i] = header.mode_extension & ~I_STEREO;
          
          w = (w + 1) % 3;
        }
      }
      else {  /* right_ch->block_type != 2 */
        unsigned int bound;
        
        bound = 0;
        for (sfbi = l = 0; l < 576; l += n) {
          n = sfbwidth[sfbi++];
          
          for (i = 0; i < n; ++i) {
            if (right_xr[i]) {
              bound = sfbi;
              break;
            }
          }
          
          right_xr += n;
        }
        
        for (i = 0; i < bound; ++i)
          modes[i] = header.mode_extension & ~I_STEREO;
      }
      
      /* now do the actual processing */
      
      if (header.flags & MAD_FLAG_LSF_EXT) {
        unsigned char const *illegal_pos;
        mad_fixed_t const *lsf_scale;
        
        illegal_pos = scalefacs[1][1];
        
        /* intensity_scale */
        lsf_scale = is_lsf_table[right_ch_scalefac_compress & 0x1];
        
        for (sfbi = l = 0; l < 576; ++sfbi, l += n) {
          n = sfbwidth[sfbi];
          
          if (!(modes[sfbi] & I_STEREO))
            continue;
          
          if (illegal_pos[sfbi]) {
            modes[sfbi] &= ~I_STEREO;
            continue;
          }
          
          is_pos = right_ch_scalefac[sfbi];
          
          for (i = 0; i < n; ++i) {
            register mad_fixed_t left;
            
            left = xr0[l + i];
            
            if (is_pos == 0)
              xr1[l + i] = left;
            else {
              register mad_fixed_t opposite;
              
              opposite = mad_f_mul(left, lsf_scale[(is_pos - 1) / 2]);
              
              if (is_pos & 1) {
                xr0[l + i] = opposite;
                xr1[l + i] = left;
              }
              else
                xr1[l + i] = opposite;
            }
          }
        }
      }
      else {  /* !(header->flags & MAD_FLAG_LSF_EXT) */
        for (sfbi = l = 0; l < 576; ++sfbi, l += n) {
          n = sfbwidth[sfbi];
          
          if (!(modes[sfbi] & I_STEREO))
            continue;
          
          is_pos = right_ch_scalefac[sfbi];
          
          if (is_pos >= 7) {  /* illegal intensity position */
            modes[sfbi] &= ~I_STEREO;
            continue;
          }
          
          for (i = 0; i < n; ++i) {
            register mad_fixed_t left;
            
            left = xr0[l + i];
            
            xr0[l + i] = mad_f_mul(left, is_table[    is_pos]);
            xr1[l + i] = mad_f_mul(left, is_table[6 - is_pos]);
          }
        }
      }
    }
    
    /* middle/side stereo */
    
    if (header.mode_extension & MS_STEREO) {
      register mad_fixed_t invsqrt2;
      
      header.flags |= MAD_FLAG_MS_STEREO;
      
      invsqrt2 = root_table[3 + -2];
      
      for (sfbi = l = 0; l < 576; ++sfbi, l += n) {
        n = sfbwidth[sfbi];
        
        if (modes[sfbi] != MS_STEREO)
          continue;
        
        for (i = 0; i < n; ++i) {
          register mad_fixed_t m, s;
          
          m = xr0[l + i];
          s = xr1[l + i];
          
          xr0[l + i] = mad_f_mul(m + s, invsqrt2);  /* l = (m + s) / sqrt(2) */
          xr1[l + i] = mad_f_mul(m - s, invsqrt2);  /* r = (m - s) / sqrt(2) */
        }
      }
    }
    
    return MAD_ERROR_NONE;
  }
  
  
  void main(void)
  {
    error = III_stereo(si_channel0[gr].flags, si_channel0[gr].block_type,
                       si_channel1[gr].scalefac_compress, 
                       si_channel1[gr].flags, si_channel1[gr].block_type, 
                       scalefacs[gr][1],
                       xr0, xr1, sfbwidth0);
  }
};
