/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: scalefac.sc,v 1.5 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: scalefac.sc,v $
 * Revision 1.5  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

# include "global.sh"

import "global";

# include "fixed.sh"
# include "frame.sh"
# include "layer3.sh"

import "bit";

static struct {
  unsigned char const l[39];
  unsigned char const s[39];
  unsigned char const m[39];
} sfbwidth_table[9] = {
  /*
   * MPEG-1 scalefactor band widths
   * derived from Table B.8 of ISO/IEC 11172-3
   */        
  { /*sfb_48000_long*/ { 4,  4,  4,  4,  4,  4,  6,  6,  6,   8,  10,
      12, 16, 18, 22, 28, 34, 40, 46, 54,  54, 192, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
    /*sfb_48000_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  6,
        6,  6,  6,  6,  6, 10, 10, 10, 12, 12, 12, 14, 14,
        14, 16, 16, 16, 20, 20, 20, 26, 26, 26, 66, 66, 66 },
    /*sfb_48000_mixed*/ { /* long */   4,  4,  4,  4,  4,  4,  6,  6,
        /* short */  4,  4,  4,  6,  6,  6,  6,  6,  6, 10,
        10, 10, 12, 12, 12, 14, 14, 14, 16, 16,
        16, 20, 20, 20, 26, 26, 26, 66, 66, 66 }
  },
  { /*sfb_44100_long*/ { 4,  4,  4,  4,  4,  4,  6,  6,  8,   8,  10,
      12, 16, 20, 24, 28, 34, 42, 50, 54,  76, 158, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
    /*sfb_44100_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  6,
        6,  6,  8,  8,  8, 10, 10, 10, 12, 12, 12, 14, 14,
        14, 18, 18, 18, 22, 22, 22, 30, 30, 30, 56, 56, 56 },
    /*sfb_44100_mixed*/ { /* long */   4,  4,  4,  4,  4,  4,  6,  6,
        /* short */  4,  4,  4,  6,  6,  6,  8,  8,  8, 10,
        10, 10, 12, 12, 12, 14, 14, 14, 18, 18,
        18, 22, 22, 22, 30, 30, 30, 56, 56, 56 }
  },
  { /*sfb_32000_long*/ { 4,  4,  4,  4,  4,  4,  6,  6,  8,  10,  12,
      16, 20, 24, 30, 38, 46, 56, 68, 84, 102,  26, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 
    /*sfb_32000_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  6,
        6,  6,  8,  8,  8, 12, 12, 12, 16, 16, 16, 20, 20,
        20, 26, 26, 26, 34, 34, 34, 42, 42, 42, 12, 12, 12 },
    /*sfb_32000_mixed*/ { /* long */   4,  4,  4,  4,  4,  4,  6,  6,
        /* short */  4,  4,  4,  6,  6,  6,  8,  8,  8, 12,
        12, 12, 16, 16, 16, 20, 20, 20, 26, 26,
        26, 34, 34, 34, 42, 42, 42, 12, 12, 12 }
  },
  /*
   * MPEG-2 scalefactor band widths
   * derived from Table B.2 of ISO/IEC 13818-3
   */
  { /*sfb_24000_long*/ { 6,  6,  6,  6,  6,  6,  8, 10, 12,  14,  16,
      18, 22, 26, 32, 38, 46, 54, 62, 70,  76,  36, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 
    /*sfb_24000_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  6,  6,  6,  8,
        8,  8, 10, 10, 10, 12, 12, 12, 14, 14, 14, 18, 18,
        18, 24, 24, 24, 32, 32, 32, 44, 44, 44, 12, 12, 12 },
    /*sfb_24000_mixed*/ { /* long */   6,  6,  6,  6,  6,  6,
        /* short */  6,  6,  6,  8,  8,  8, 10, 10, 10, 12,
        12, 12, 14, 14, 14, 18, 18, 18, 24, 24,
        24, 32, 32, 32, 44, 44, 44, 12, 12, 12 }
  },
  { /*sfb_22050_long*/ { 6,  6,  6,  6,  6,  6,  8, 10, 12,  14,  16,
      20, 24, 28, 32, 38, 46, 52, 60, 68,  58,  54, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 
    /*sfb_22050_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  6,  6,  6,  6,
        6,  6,  8,  8,  8, 10, 10, 10, 14, 14, 14, 18, 18,
        18, 26, 26, 26, 32, 32, 32, 42, 42, 42, 18, 18, 18 },
    /*sfb_22050_mixed*/ { /* long */   6,  6,  6,  6,  6,  6,
        /* short */  6,  6,  6,  6,  6,  6,  8,  8,  8, 10,
        10, 10, 14, 14, 14, 18, 18, 18, 26, 26,
        26, 32, 32, 32, 42, 42, 42, 18, 18, 18 }
  },
  { /*sfb_16000_long aka sfb_22050_long*/ { 6,  6,  6,  6,  6,  6,  8, 10, 12,  14,  16,
      20, 24, 28, 32, 38, 46, 52, 60, 68,  58,  54, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 
    /*sfb_16000_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  6,  6,  6,  8,
        8,  8, 10, 10, 10, 12, 12, 12, 14, 14, 14, 18, 18,
        18, 24, 24, 24, 30, 30, 30, 40, 40, 40, 18, 18, 18 },
    /*sfb_16000_mixed*/ { /* long */   6,  6,  6,  6,  6,  6,
        /* short */  6,  6,  6,  8,  8,  8, 10, 10, 10, 12,
        12, 12, 14, 14, 14, 18, 18, 18, 24, 24,
        24, 30, 30, 30, 40, 40, 40, 18, 18, 18 }
  },
  /*
   * MPEG 2.5 scalefactor band widths
   * derived from public sources
   */
  { /*sfb_12000_long aka sfb_16000_long aka sfb_22050_long*/ { 6,  6,  6,  6,  6,  6,  8, 10, 12,  14,  16,
      20, 24, 28, 32, 38, 46, 52, 60, 68,  58,  54, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 
    /*sfb_12000_short aka sfb_16000_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  6,  6,  6,  8,
        8,  8, 10, 10, 10, 12, 12, 12, 14, 14, 14, 18, 18,
        18, 24, 24, 24, 30, 30, 30, 40, 40, 40, 18, 18, 18 },
    /*sfb_12000_mixed aka sfb_16000_mixed*/ { /* long */   6,  6,  6,  6,  6,  6,
        /* short */  6,  6,  6,  8,  8,  8, 10, 10, 10, 12,
        12, 12, 14, 14, 14, 18, 18, 18, 24, 24,
        24, 30, 30, 30, 40, 40, 40, 18, 18, 18 }
  },
  { /*sfb_11025_long aka sfb_12000_long aka sfb_16000_long aka sfb_22050_long*/ { 6,  6,  6,  6,  6,  6,  8, 10, 12,  14,  16,
      20, 24, 28, 32, 38, 46, 52, 60, 68,  58,  54, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, 
    /*sfb_11025_short aka sfb_12000_short aka sfb_16000_short*/ { 4,  4,  4,  4,  4,  4,  4,  4,  4,  6,  6,  6,  8,
        8,  8, 10, 10, 10, 12, 12, 12, 14, 14, 14, 18, 18,
        18, 24, 24, 24, 30, 30, 30, 40, 40, 40, 18, 18, 18 }, 
    /*sfb_11025_mixed aka sfb_12000_mixed aka sfb_16000_mixed*/ { /* long */   6,  6,  6,  6,  6,  6,
        /* short */  6,  6,  6,  8,  8,  8, 10, 10, 10, 12,
        12, 12, 14, 14, 14, 18, 18, 18, 24, 24,
        24, 30, 30, 30, 40, 40, 40, 18, 18, 18 }
  },
  { /*sfb_8000_long*/ { 12, 12, 12, 12, 12, 12, 16, 20, 24,  28,  32,
      40, 48, 56, 64, 76, 90,  2,  2,  2,   2,   2, /*padded with zeros*/
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },  
    /*sfb_8000_short*/ { 8,  8,  8,  8,  8,  8,  8,  8,  8, 12, 12, 12, 16,
        16, 16, 20, 20, 20, 24, 24, 24, 28, 28, 28, 36, 36,
        36,  2,  2,  2,  2,  2,  2,  2,  2,  2, 26, 26, 26 },
    /*sfb_8000_mixed*/ { /* long */  12, 12, 12,
        /* short */  4,  4,  4,  8,  8,  8, 12, 12, 12, 16, 16, 16,
        20, 20, 20, 24, 24, 24, 28, 28, 28, 36, 36, 36,
        2,  2,  2,  2,  2,  2,  2,  2,  2, 26, 26, 26 }
  }
};


/*
 * scalefactor bit lengths
 * derived from section 2.4.2.7 of ISO/IEC 11172-3
 */

static struct {
  unsigned char const slen1;
  unsigned char const slen2;
} sflen_table[16] = {
  { 0, 0 }, { 0, 1 }, { 0, 2 }, { 0, 3 },
  { 3, 0 }, { 1, 1 }, { 1, 2 }, { 1, 3 },
  { 2, 1 }, { 2, 2 }, { 2, 3 }, { 3, 1 },
  { 3, 2 }, { 3, 3 }, { 4, 2 }, { 4, 3 }
};

/*
 * number of LSF scalefactor band values
 * derived from section 2.4.3.2 of ISO/IEC 13818-3
 */
static unsigned char const nsfb_table[6][3][4] = {
  { {  6,  5,  5, 5 },
    {  9,  9,  9, 9 },
    {  6,  9,  9, 9 } },
  
  { {  6,  5,  7, 3 },
    {  9,  9, 12, 6 },
    {  6,  9, 12, 6 } },
  
  { { 11, 10,  0, 0 },
    { 18, 18,  0, 0 },
    { 15, 18,  0, 0 } },
  
  { {  7,  7,  7, 0 },
    { 12, 12, 12, 0 },
    {  6, 15, 12, 0 } },
  
  { {  6,  6,  6, 3 },
    { 12,  9,  9, 6 },
    {  6, 12,  9, 6 } },
  
  { {  8,  8,  5, 0 },
    { 15, 12,  9, 0 },
    {  6, 18,  9, 0 } }
};



behavior Scalefactors(struct mad_bitptr bitptr,
		      in struct mad_header header,
		      in unsigned int gr,
		      in unsigned int ch,
		      in unsigned int sfreqi,
		      in unsigned char si_scfsi[2],
                      struct sideinfo_channel si_channel[2],
		      out unsigned char scalefac[2][2][39],
		      out unsigned char sfbwidth[39],
                      out unsigned int part2_length)
{  
  /*
   * NAME:	III_scalefactors_lsf()
   * DESCRIPTION:	decode channel scalefactors for LSF from a bitstream
   */
  unsigned int III_scalefactors_lsf(struct mad_bitptr *ptr,
				    unsigned char *gr1ch_scalefac,
				    int mode_extension)
  {
    struct mad_bitptr start;
    unsigned int scalefac_compress, index, slen[4], part, n, i;
    unsigned char const *nsfb;
    
    //start = *ptr;
    mymemcpy(&start, ptr, sizeof(struct mad_bitptr));
    
    scalefac_compress = si_channel[gr].scalefac_compress;
    index = (si_channel[gr].block_type == 2) ?
      ((si_channel[gr].flags & mixed_block_flag) ? 2 : 1) : 0;
    
    if (!((mode_extension & I_STEREO) && gr1ch_scalefac)) {
      if (scalefac_compress < 400) {
        slen[0] = (scalefac_compress >> 4) / 5;
        slen[1] = (scalefac_compress >> 4) % 5;
        slen[2] = (scalefac_compress % 16) >> 2;
        slen[3] =  scalefac_compress %  4;
        
        nsfb = nsfb_table[0][index];
      }
      else if (scalefac_compress < 500) {
        scalefac_compress -= 400;
        
        slen[0] = (scalefac_compress >> 2) / 5;
        slen[1] = (scalefac_compress >> 2) % 5;
        slen[2] =  scalefac_compress %  4;
        slen[3] = 0;
        
        nsfb = nsfb_table[1][index];
      }
      else {
        scalefac_compress -= 500;
        
        slen[0] = scalefac_compress / 3;
        slen[1] = scalefac_compress % 3;
        slen[2] = 0;
        slen[3] = 0;
        
        si_channel[gr].flags |= preflag;
        
      nsfb = nsfb_table[2][index];
      }
      
      n = 0;
      for (part = 0; part < 4; ++part) {
        for (i = 0; i < nsfb[part]; ++i)
          scalefac[gr][ch][n++] = mad_bit_read(ptr, slen[part]);
      }

      while (n < 39)
        scalefac[gr][ch][n++] = 0;
    }
    else {  /* (mode_extension & I_STEREO) && gr1ch (i.e. ch == 1) */
      scalefac_compress >>= 1;
      
      if (scalefac_compress < 180) {
        slen[0] =  scalefac_compress / 36;
        slen[1] = (scalefac_compress % 36) / 6;
        slen[2] = (scalefac_compress % 36) % 6;
        slen[3] = 0;
        
        nsfb = nsfb_table[3][index];
      }
      else if (scalefac_compress < 244) {
        scalefac_compress -= 180;
        
        slen[0] = (scalefac_compress % 64) >> 4;
        slen[1] = (scalefac_compress % 16) >> 2;
        slen[2] =  scalefac_compress %  4;
        slen[3] = 0;
        
        nsfb = nsfb_table[4][index];
      }
      else {
        scalefac_compress -= 244;
        
        slen[0] = scalefac_compress / 3;
        slen[1] = scalefac_compress % 3;
        slen[2] = 0;
        slen[3] = 0;
        
        nsfb = nsfb_table[5][index];
      }
      
      n = 0;
      for (part = 0; part < 4; ++part) {
        unsigned int max, is_pos;
        
        max = (1 << slen[part]) - 1;
        
        for (i = 0; i < nsfb[part]; ++i) {
          is_pos = mad_bit_read(ptr, slen[part]);
          
          scalefac[gr][ch][n] = is_pos;
          gr1ch_scalefac[n++] = (is_pos == max);
        }
      }
      
      while (n < 39) {
        scalefac[gr][ch][n] = 0;
        gr1ch_scalefac[n++] = 0;  /* apparently not illegal */
      }
    }
    
    return mad_bit_length(&start, ptr);
  }
  
  /*
   * NAME:	III_scalefactors()
   * DESCRIPTION:	decode channel scalefactors of one granule from a bitstream
   */
  unsigned int III_scalefactors(struct mad_bitptr *ptr,
				unsigned char *gr0ch_scalefac,
				unsigned int scfsi)
  {
    struct mad_bitptr start;
    unsigned int slen1, slen2, sfbi;
    
    //start = *ptr;
    mymemcpy(&start, ptr, sizeof(struct mad_bitptr));
    
    slen1 = sflen_table[si_channel[gr].scalefac_compress].slen1;
    slen2 = sflen_table[si_channel[gr].scalefac_compress].slen2;
    
    if (si_channel[gr].block_type == 2) {
      unsigned int nsfb;
      
      sfbi = 0;
      
      nsfb = (si_channel[gr].flags & mixed_block_flag) ? 8 + 3 * 3 : 6 * 3;
      while (nsfb--)
        scalefac[gr][ch][sfbi++] = mad_bit_read(ptr, slen1);
      
      nsfb = 6 * 3;
      while (nsfb--)
        scalefac[gr][ch][sfbi++] = mad_bit_read(ptr, slen2);
      
      nsfb = 1 * 3;
      while (nsfb--)
        scalefac[gr][ch][sfbi++] = 0;
    }
    else {  /* channel->block_type != 2 */
      if (scfsi & 0x8) {
        for (sfbi = 0; sfbi < 6; ++sfbi)
          scalefac[gr][ch][sfbi] = gr0ch_scalefac[sfbi];
      }
      else {
        for (sfbi = 0; sfbi < 6; ++sfbi)
          scalefac[gr][ch][sfbi] = mad_bit_read(ptr, slen1);
      }
      
      if (scfsi & 0x4) {
        for (sfbi = 6; sfbi < 11; ++sfbi)
          scalefac[gr][ch][sfbi] = gr0ch_scalefac[sfbi];
      }
      else {
        for (sfbi = 6; sfbi < 11; ++sfbi)
          scalefac[gr][ch][sfbi] = mad_bit_read(ptr, slen1);
      }
      
      if (scfsi & 0x2) {
        for (sfbi = 11; sfbi < 16; ++sfbi)
          scalefac[gr][ch][sfbi] = gr0ch_scalefac[sfbi];
      }
      else {
        for (sfbi = 11; sfbi < 16; ++sfbi)
          scalefac[gr][ch][sfbi] = mad_bit_read(ptr, slen2);
      }
      
      if (scfsi & 0x1) {
        for (sfbi = 16; sfbi < 21; ++sfbi)
          scalefac[gr][ch][sfbi] = gr0ch_scalefac[sfbi];
      }
      else {
        for (sfbi = 16; sfbi < 21; ++sfbi)
          scalefac[gr][ch][sfbi] = mad_bit_read(ptr, slen2);
      }
      
      scalefac[gr][ch][21] = 0;
    }
    
    return mad_bit_length(&start, ptr);
  }
  
  
  void main(void)
  {
    unsigned char* grch_scalefac = 0;
    
    sfbwidth = sfbwidth_table[sfreqi].l;

#if 0
    if (si_channel[gr].block_type == 2) {
	sfbwidth = (si_channel[gr].flags & mixed_block_flag) ?
	  sfbwidth_table[sfreqi].m : sfbwidth_table[sfreqi].s;
    }
#else
    if (si_channel[gr].block_type == 2) {
        if (si_channel[gr].flags & mixed_block_flag) 
	        sfbwidth = sfbwidth_table[sfreqi].m;
        else
                sfbwidth = sfbwidth_table[sfreqi].s;
    }
#endif

    if (header.flags & MAD_FLAG_LSF_EXT) {
      if (ch != 0) grch_scalefac = &(scalefac[1][1][0]);
      part2_length = III_scalefactors_lsf(&bitptr, grch_scalefac,
					  header.mode_extension);
    }
    else {
      part2_length = III_scalefactors(&bitptr,
				      ch == 0 ? &(scalefac[0][0][0]) : &(scalefac[0][1][0]),
				      gr == 0 ? 0 : si_scfsi[ch]);
    }
  }
};
