/*
 * SpecC MPEG 3 audio decoder
 * Copyright (C) 2006-2007 CECS, UC Irvine
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: mp3monitor.sc,v 1.3 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: mp3monitor.sc,v $
 * Revision 1.3  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sim.sh>


# include "global.sh"
# include "stream.sh"

#ifdef DBG_FRAME_INFO
import "i_receive";
#endif

import "io";

/* Stream assumptions: */
/* 44.1 stereo:
   2 granules * 576 samples per granule * 2 channels (stereo) * 2 bytes per sample
   1152 * 4 = 4608 */
#define BYTES_PER_SAMPLE 2*2

/* 1000[ms/s] * (1152[samples/frame] / 44100 [samples/sec]) ms */ 
#define STEREO_DEADLINE 26.12245
#define SAMPLES_PER_FRAME 1152
#define SAMPLING_FREQ   44100

#define NFRAMES2AVG 1


import "i_receiver";
#ifdef EXTERN_CONTROL
import "i_send";
#endif


interface IMad_Monitor {
  int open_file (const char* pcmfile_name);
};

#ifdef DBG_FRAME_INFO
behavior Mad_Monitor_file(
#else
behavior Mad_Monitor(
#endif
                 i_receiver pcm_out, in unsigned long nframes
#ifdef  EXTERN_CONTROL
                 ,i_send mp3Finish
#endif
                 )
  implements IMad_Monitor
{
  FILE *file_handle;            

  int open_file (const char* pcmfile_name)
  {
    if ((file_handle = fopen (pcmfile_name, "w")) == NULL)
    {
      fprintf (stderr, "Error opening output bitstream file %s!\n",
               pcmfile_name);
      return 0;
    }
    fprintf (stderr, "Output bitstream file:\t%s\n", pcmfile_name);
    return 1;
  }

  void main(void)
  {
    sim_time lastTime = 0, nowTime, deltaTime;
    size_t len;
    char buf[BYTES_PER_SAMPLE];
    int blockNr = 0, nsamples = 0;
    long long deadline, deadline_per_sample;

    deadline_per_sample = (long long)((1.0/(double)(SAMPLING_FREQ))
                                      * 1000000000.0 * 1000ull);
    //this is the deadline per sample, we need to find the deadline for
    //NFRAMES2AVG frames
    deadline = deadline_per_sample * NFRAMES2AVG * SAMPLES_PER_FRAME;

    while(blockNr < nframes)
    {
      pcm_out.receive(buf, BYTES_PER_SAMPLE);

#ifndef DBG_FRAME_INFO
      if(lastTime == 0 && nsamples == 0 && blockNr == 0)
        fprintf(stderr, "Initial Latency = %2.2f ms\n",
                (double)now() / (double)(1000000.0 * 1000.0));
#endif      
      if(++nsamples == SAMPLES_PER_FRAME)
      {
        blockNr++;
        nsamples = 0;
        if((blockNr % NFRAMES2AVG) == 0)
        {
          nowTime = now();
          if ( lastTime > 0 ) {
            deltaTime = nowTime - lastTime;
#ifndef DBG_FRAME_INFO
            /* did we miss the deadline ? */
            if ((double) deltaTime > (double) deadline) {
              fprintf(stderr, "Missed deadline for frame %d: %2.2f (> %2.2f)!\n",
                      blockNr,
                      (double)deltaTime / (double)(1000000.0 * 1000.0 * NFRAMES2AVG),
                      ((double)deadline)/(double)(1000000.0 * 1000.0 * NFRAMES2AVG));
            }
            else
            {
              //printing every 10 frames was too often and 
              //NFRAMES2AVG might
              //change in future so this extra check
              // if((blockNr % 100) == 0) 
                fprintf(stderr, "Decode time per frame at frame "
                                "%d = %2.2f ms < deadline %2.2f ms\n", 
                        blockNr, (double) deltaTime / (1000000.0 * 1000.0 * NFRAMES2AVG),
                        (double)deadline/(double)(1000000.0 * 1000.0 * NFRAMES2AVG)); 
            }
#endif
          }
          lastTime = nowTime;
        }
      }

      len = fwrite( buf, sizeof(char), BYTES_PER_SAMPLE,
                    file_handle);
#ifdef THROTTLE_OUTPUT
      // wait for sample time to simulate a correct 44.1KHz output
      waitfor(deadline_per_sample);
#endif
    }

#ifdef EXTERN_CONTROL
    // if external controlled do not exit the application but 
    // signal to controller that we are done 
    mp3Finish.send();
#else
    fclose(file_handle);
    exit(0);
#endif
  }
};


#ifdef DBG_FRAME_INFO
behavior Mad_Monitor_print(i_receive newFrame ) {
  unsigned int blockNr; 

  void main(void) {
    sim_time frameStart, frameEnd;
    sim_time lastFrameStart, lastFrameEnd;
    sim_time totalTransfer, totalComputation;
    event incDeltaCycle;


    long long deadline, deadline_per_sample;

    deadline_per_sample = (long long)((1.0/(double)(SAMPLING_FREQ))
                                      * 1000000000.0 * 1000ull);
    //this is the deadline per sample, we need to find the deadline for
    //NFRAMES2AVG frames
    deadline = deadline_per_sample * NFRAMES2AVG * SAMPLES_PER_FRAME;




    blockNr = 0;

    /// reset counter for frame averaging
    totalTransfer = 0ull;
    totalComputation = 0ull;

    while(1) {
      // begin of the frame
      newFrame.receive();
      lastFrameStart = frameStart;
      frameStart = now();
      //printf("Monitor_print: got start %Lu\n",
      //       (unsigned long long) now() );
      
      // need at least a delta cycle delay otherwise 
      // it will self trigger forever in BFM 
      notify incDeltaCycle;
      wait incDeltaCycle;
      
      // finished frame transmission
      newFrame.receive();
      lastFrameEnd = frameEnd;
      frameEnd   = now();
      //printf("Monitor_print: got finish %Lu\n",
      //       (unsigned long long) now() );
      
      

      totalTransfer         += frameEnd - frameStart;
      totalComputation      += frameStart - lastFrameEnd;
      
      if((blockNr % NFRAMES2AVG) == 0){ 
        if(blockNr == 0) {
          fprintf(stderr, "Initial Latency = %2.3f ms, %2.3f ms tx\n",
                  // initial delay 
                  (double)frameStart / (double)(1000000.0 * 1000.0),
                  // delay due to transfer to monitor 
                  (double) totalTransfer/ (double)(1000000.0 * 1000.0)
                  );
        } else {
          fprintf(stderr, "Decode time per frame at frame "
                  "%d = %2.3f ms,  %2.3f ms tx (deadline %2.2f ms)\n", 
                  blockNr + 1, 
                  (double) totalComputation / (1000000.0 * 1000.0 * NFRAMES2AVG),
                  (double) totalTransfer    / (1000000.0 * 1000.0 * NFRAMES2AVG),
                  (double)deadline/(double)(1000000.0 * 1000.0 * NFRAMES2AVG)); 
        }
        // reset totals
        totalTransfer = 0ull;
        totalComputation = 0ull;
      }

      // need at least a delta cycle delay otherwise 
      // it will self trigger forever in BFM 
      notify incDeltaCycle;
      wait incDeltaCycle;
      

      blockNr++;
    }
  }
};


behavior Mad_Monitor( 
                 i_receiver pcm_out, in unsigned long nframes,
                 i_receive newFrame
#ifdef  EXTERN_CONTROL
                 ,i_send mp3Finish
#endif
                 ) 
implements IMad_Monitor 
{
  
  Mad_Monitor_print monPrint(newFrame);
  Mad_Monitor_file monFile(pcm_out, nframes
#ifdef  EXTERN_CONTROL
                          ,mp3Finish
#endif
                          );
  
  int open_file (const char* pcmfile_name) {
    monFile.open_file(pcmfile_name);
  }

  void main(void) {
    par{
      monFile.main();
      monPrint.main();
    }
  }
};
#endif




#ifdef SIGNAL_TYPE_ERROR
behavior Mad_Error(in signal enum mad_error error)
#else
behavior Mad_Error(i_mad_error_receiver error)
#endif
{
  char const *mad_stream_errorstr(enum mad_error inError)
  {
  switch (inError) {
  case MAD_ERROR_NONE:		 return "no error";
  case MAD_ERROR_INTERNAL:	 return "internal error";

  case MAD_ERROR_BUFLEN:	 return "input buffer too small (or EOF)";
  case MAD_ERROR_BUFPTR:	 return "invalid (null) buffer pointer";

  case MAD_ERROR_NOMEM:		 return "not enough memory";

  case MAD_ERROR_LOSTSYNC:	 return "lost synchronization";
  case MAD_ERROR_BADLAYER:	 return "reserved or unsupported header layer value";
  case MAD_ERROR_BADBITRATE:	 return "forbidden bitrate value";
  case MAD_ERROR_BADSAMPLERATE:	 return "reserved sample frequency value";
  case MAD_ERROR_BADEMPHASIS:	 return "reserved emphasis value";

  case MAD_ERROR_BADCRC:	 return "CRC check failed";
  case MAD_ERROR_BADBITALLOC:	 return "forbidden bit allocation value";
  case MAD_ERROR_BADSCALEFACTOR: return "bad scalefactor index";
  case MAD_ERROR_BADMODE:	 return "bad bitrate/mode combination";
  case MAD_ERROR_BADFRAMELEN:	 return "bad frame length";
  case MAD_ERROR_BADBIGVALUES:	 return "bad big_values count";
  case MAD_ERROR_BADBLOCKTYPE:	 return "reserved block_type";
  case MAD_ERROR_BADSCFSI:	 return "bad scalefactor selection info";
  case MAD_ERROR_BADDATAPTR:	 return "bad main_data_begin pointer";
  case MAD_ERROR_BADPART3LEN:	 return "bad audio data length";
  case MAD_ERROR_BADHUFFTABLE:	 return "bad Huffman table select";
  case MAD_ERROR_BADHUFFDATA:	 return "Huffman data overrun";
  case MAD_ERROR_BADSTEREO:	 return "incompatible block_type for JS";
  }
  return "internal!";
  }
  
  void main(void)
  {
    enum mad_error err;
    while(1)
    {
#ifdef SIGNAL_TYPE_ERROR
      wait (error);
      err = error;
#else
      int v;
      error.receive(&v);
      err = (enum mad_error)v;
#endif
      fprintf(stderr, "Decoding error 0x%04x (%s)\n",
              err, mad_stream_errorstr(err));
    }
  }  
};
