/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: mp3decoder.sc,v 1.3 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: mp3decoder.sc,v $
 * Revision 1.3  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

# include "global.sh"

# include "stream.sh"
# include "frame.sh"

import "io";
import "header";
import "frame";
import "synth";

#ifdef DBG_FRAME_INFO
import "i_send";
#endif


behavior Mad_decoder_init(out struct mad_stream stream,
			  out struct mad_header header,
			  out enum mad_flow flow,
			  out mad_fixed_t synth_filter0[2][2][16][8],
			  out mad_fixed_t synth_filter1[2][2][16][8],
			  out unsigned int synth_phase)
{
  /*
   * NAME:	synth->mute()
   * DESCRIPTION:	zero all polyphase filterbank values, resetting synthesis
   */
  void mad_synth_mute(void)
  {
    unsigned int s, v;

    for (s = 0; s < 16; ++s) {
      for (v = 0; v < 8; ++v) {
	synth_filter0[0][0][s][v] = 0;
        synth_filter0[0][1][s][v] = 0;
	synth_filter0[1][0][s][v] = 0;
        synth_filter0[1][1][s][v] = 0;
	synth_filter1[0][0][s][v] = 0;
        synth_filter1[0][1][s][v] = 0;
	synth_filter1[1][0][s][v] = 0;
        synth_filter1[1][1][s][v] = 0;
      }
    }
  }

  /*
   * NAME:	stream->init()
   * DESCRIPTION:	initialize stream struct
   */
  void mad_stream_init(struct mad_stream *pStream)
  {
    pStream->bufend     = pStream->buffer;
    pStream->skiplen    = 0;
    
    pStream->sync       = 1;
    pStream->freerate   = 0;
    
    pStream->this_frame = pStream->buffer;
    pStream->next_frame = pStream->buffer;
    mad_bit_init(&pStream->ptr, pStream->buffer);

    mad_bit_init(&pStream->anc_ptr, 0);
    pStream->anc_bitlen = 0;
    
    pStream->md_len     = 0;
    
    pStream->options    = 0;
    pStream->error      = MAD_ERROR_NONE;
  }
  
  /*
   * NAME:	header->init()
   * DESCRIPTION:	initialize header struct
   */
  void mad_header_init(struct mad_header *pHeader)
  {
    pHeader->layer          = MAD_LAYER_III;
    pHeader->mode           = MAD_MODE_STEREO;
    pHeader->mode_extension = 0;
    pHeader->emphasis       = MAD_EMPHASIS_NONE;
    
    pHeader->bitrate        = 0;
    pHeader->samplerate     = 0;
    
    pHeader->crc_check      = 0;
    pHeader->crc_target     = 0;
    
    pHeader->flags          = 0;
    pHeader->private_bits   = 0;
    
    //PC: to avoid memcopy
    //pHeader->duration       = mad_timer_zero;
    pHeader->duration.fraction = mad_timer_zero.fraction;
    pHeader->duration.seconds = mad_timer_zero.seconds;
  }
  
  void main(void)
  {
    mad_stream_init(&stream);
    mad_header_init(&header);
    
    mad_synth_mute();

    synth_phase = 0;
    
    // anyway set in the synth filter before the output
    //synth_pcm_samplerate = 0;
    //synth_pcm_channels   = 0;
    //synth_pcm_length     = 0;
    
    stream.options = 0;
    
    flow = MAD_FLOW_CONTINUE;
  }
};


behavior Mad_decoder_recover(struct mad_stream stream,
			     enum mad_flow flow,
                 int result,
#ifdef SIGNAL_TYPE_ERROR
			     out signal enum mad_error decoder_error)
#else
			     i_mad_error_sender decoder_error)
#endif
{
  int bad_last_frame = 0;  
      
  void main(void)
  {
    if (flow == MAD_FLOW_IGNORE) {
      flow = MAD_FLOW_CONTINUE;      
      stream.error = MAD_ERROR_NONE;
      return;
    }
         
    flow = MAD_FLOW_CONTINUE;
    
    if (result)
    {      
#ifdef SIGNAL_TYPE_ERROR
      decoder_error = stream.error;
#else
      decoder_error.send(stream.error);
#endif
      
      if (stream.error == MAD_ERROR_BADCRC) {
	flow = MAD_FLOW_IGNORE;
	if (bad_last_frame) return;
	bad_last_frame = 1;
      }

      result = 0;
      stream.error = MAD_ERROR_NONE;
    } 
    else {
      bad_last_frame = 0;
    }
  }
};


behavior Mad_decoder_fail(struct mad_stream stream,
#ifdef SIGNAL_TYPE_ERROR
			     out signal enum mad_error decoder_error)
#else
			     i_mad_error_sender decoder_error)
#endif
{
  void main(void)
  {
    enum mad_error err;
    if (stream.error)
      err = stream.error;
    else
      err = MAD_ERROR_INTERNAL;

#ifdef SIGNAL_TYPE_ERROR
      decoder_error = err;
#else
      decoder_error.send(stream.error);
#endif
  }
};


behavior Mad_decoder_finish(struct mad_stream stream)
{
  void main(void)
  {
    mad_bit_finish(&stream.anc_ptr);
    mad_bit_finish(&stream.ptr);
  }
};



behavior Mad_decoder(i_receiver stream_in,
		     i_sender   pcm_out,
#ifdef SIGNAL_TYPE_ERROR
		     out signal enum mad_error error
#else
		     i_mad_error_sender error
#endif
#ifdef DBG_FRAME_INFO
                     ,i_send newFrame
#endif                     
#ifdef EXTERN_CONTROL
                     , i_receive mp3Continue, in unsigned int mp3Pause
#endif
                     )
{
  enum mad_flow flow;
  int result = 0;
  
  struct mad_stream stream;
  struct mad_header header;
  
  mad_fixed_t frame_overlap[32][18];
  mad_fixed_t frame_sbsample[36][32];
  mad_fixed_t frame_overlap2[32][18];
  mad_fixed_t frame_sbsample2[36][32];
  
  mad_fixed_t synth_filter0[2][2][16][8];
  mad_fixed_t synth_filter1[2][2][16][8];
  unsigned int synth_phase;
  
  
  Mad_decoder_init init(stream, header, flow,
			synth_filter0, synth_filter1,
			synth_phase);

  
  Mad_frame_mute mute(frame_overlap, frame_sbsample,
		      frame_overlap2, frame_sbsample2);

  Mad_input input(stream_in, result, flow, stream);

  Mad_header_decode decode_header(header, stream, result);
    
  Mad_frame_decode decode_frame(header, stream,
				frame_overlap, frame_sbsample,
				frame_overlap2, frame_sbsample2,
				result);
  
  Mad_synth_frame synth(header,
			synth_filter0, synth_filter1,
                        frame_sbsample, frame_sbsample2,
			synth_phase, pcm_out
#ifdef DBG_FRAME_INFO
                        ,newFrame
#endif
#ifdef EXTERN_CONTROL
                        , mp3Continue, mp3Pause
#endif
                        );

  Mad_decoder_recover recover(stream, flow, result, error);
  
  Mad_decoder_fail    fail(stream, error);

  Mad_decoder_finish  finish(stream);

  
  void main(void)
  {
    fsm {

      init:
      
      mute: {
	if (flow == MAD_FLOW_IGNORE) goto synth;
      }
      
      input: {
	if (flow == MAD_FLOW_STOP) goto finish;
	if (flow == MAD_FLOW_BREAK) goto fail;
	if (header.flags & MAD_FLAG_INCOMPLETE) goto decode_frame;
      }
      
      decode_header: {
	if (result == 0) goto decode_frame;
	if (stream.error == MAD_ERROR_BUFLEN) goto input;
	if (!MAD_RECOVERABLE(stream.error)) goto fail;
	goto recover;
      }
      
      decode_frame: {
	if (result == 0) goto synth;
	if (stream.error == MAD_ERROR_LOSTSYNC) goto decode_header;
	if (!MAD_RECOVERABLE(stream.error)) goto fail;
	goto recover;
      }
  
      synth: {
	if (flow == MAD_FLOW_STOP) goto finish;
	if (flow == MAD_FLOW_BREAK) goto fail;
      }

      recover: {
	if (flow == MAD_FLOW_STOP) goto finish;
	if (flow == MAD_FLOW_BREAK) goto fail;
	if ((flow == MAD_FLOW_IGNORE) && (stream.error)) goto mute;
	if (flow == MAD_FLOW_IGNORE) goto synth;
	goto decode_header;
      }
            
      fail:
      
      finish:
    }
  }  
};
