/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: layer3.sc,v 1.3 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: layer3.sc,v $
 * Revision 1.3  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

#include <assert.h>

# include "global.sh"

# include "bit.sh"
# include "stream.sh"
# include "frame.sh"


import "bit";

import "sideinfo";
import "decode3";


/* --- Layer III ----------------------------------------------------------- */



behavior Mad_layer_III_init(struct mad_stream stream,
                            struct mad_header header,
                            in int frame_options,
                            unsigned int nch,
                            out int result)
{
  void main(void)
  {
    unsigned int si_len;
    
    nch = MAD_NCHANNELS(header);
    si_len = (header.flags & MAD_FLAG_LSF_EXT) ?
      (nch == 1 ? 9 : 17) : (nch == 1 ? 17 : 32);
    
    /* check frame sanity */
    
    if (stream.next_frame - mad_bit_nextbyte(&stream.ptr) <
        (signed int) si_len) {
      stream.error = MAD_ERROR_BADFRAMELEN;
      stream.md_len = 0;
      result = -2;
      return;
    }
    
    /* check CRC word */
    
    if (header.flags & MAD_FLAG_PROTECTION) {
      header.crc_check =
        mad_bit_crc(& stream.ptr, si_len * CHAR_BIT, header.crc_check);
      
      if (header.crc_check != header.crc_target &&
          !(frame_options & MAD_OPTION_IGNORECRC)) {
        stream.error = MAD_ERROR_BADCRC;
        result = -1;
      }
    }    
  }
};


behavior Mad_layer_III_main_data(struct mad_stream stream,
                                 struct mad_header header,
                                 in  enum mad_error error,
                                 in  unsigned int priv_bitlen,
                                 in  unsigned int si_private_bits,
                                 in  unsigned int si_main_data_begin,
                                     unsigned int md_len,
                                     unsigned int next_md_begin,
                                 out unsigned int frame_free,
                                 out struct mad_bitptr ptr,
                                     int result)
{
  void main(void)
  {
    unsigned int frame_space, frame_used;
    
    if (error && result == 0) {
      stream.error = error;
      result = -1;
    }
    
    header.flags        |= priv_bitlen;
    header.private_bits |= si_private_bits;
    
    /* find main_data of next frame */
    
    {
      struct mad_bitptr peek;
      uln header;
      
      mad_bit_init(&peek, stream.next_frame);
      
      header = mad_bit_read(&peek, 32);
      if ((header & 0xffe60000L) /* syncword | layer */ == 0xffe20000L) {
        if (!(header & 0x00010000L))  /* protection_bit */
          mad_bit_skip(&peek, 16);  /* crc_check */
        
        next_md_begin =
          mad_bit_read(&peek, (header & 0x00080000L) /* ID */ ? 9 : 8);
      }
      
      mad_bit_finish(&peek);
    }
    
    /* find main_data of this frame */
    
    frame_space = stream.next_frame - mad_bit_nextbyte(&stream.ptr);
    
    if (next_md_begin > si_main_data_begin + frame_space)
      next_md_begin = 0;
    
    md_len = si_main_data_begin + frame_space - next_md_begin;
    
    frame_used = 0;
    
    if (si_main_data_begin == 0) {
      //ptr = stream->ptr;
      mymemcpy(&ptr, &stream.ptr, sizeof(struct mad_bitptr));
      
      stream.md_len = 0;
      
      frame_used = md_len;
    }
    else {
      if (si_main_data_begin > stream.md_len) {
        if (result == 0) {
          stream.error = MAD_ERROR_BADDATAPTR;
          result = -1;
        }
      }
      else {
        mad_bit_init(&ptr,
                     stream.main_data + stream.md_len - si_main_data_begin);
        
        if (md_len > si_main_data_begin) {
          assert(stream.md_len + md_len -
                 si_main_data_begin <= MAD_BUFFER_MDLEN);
          mymemcpy(stream.main_data + stream.md_len,
                   mad_bit_nextbyte(&stream.ptr),
                   frame_used = md_len - si_main_data_begin);
          
          stream.md_len += frame_used;
        }
      }
    }
  
    frame_free = frame_space - frame_used;
  }
};


behavior Mad_layer_III_anc(struct mad_stream stream,
                           in struct mad_bitptr ptr,
                           in unsigned int data_bitlen,
                           in unsigned int md_len,
			   enum mad_error error,
                           out int result)
{
  void main(void)
  {
    if (error) {
      stream.error = error;
      result = -1;
    }

    /* designate ancillary bits */

    //stream->anc_ptr    = ptr;
	mymemcpy(&stream.anc_ptr, &ptr, sizeof(struct mad_bitptr));

    stream.anc_bitlen = md_len * CHAR_BIT - data_bitlen;
  }
};
  

behavior Mad_layer_III_preload(struct mad_stream stream,
                               in unsigned int si_main_data_begin,
                               in unsigned int data_bitlen,
                               in unsigned int md_len,
                               in unsigned int next_md_begin,
                               in unsigned int frame_free)
{
  void main(void)
  {
    /* preload main_data buffer with up to 511 bytes for next frame(s) */
    
    if (frame_free >= next_md_begin) {
      mymemcpy(stream.main_data,
               stream.next_frame - next_md_begin, next_md_begin);
      stream.md_len = next_md_begin;
    }
    else {
      if (md_len < si_main_data_begin) {
        unsigned int extra;
        
        extra = si_main_data_begin - md_len;
        if (extra + frame_free > next_md_begin)
          extra = next_md_begin - frame_free;
        
        if (extra < stream.md_len) {
          mymemmove(stream.main_data,
                    stream.main_data + stream.md_len - extra, extra);
          stream.md_len = extra;
        }
      }
      else
        stream.md_len = 0;
      mymemcpy(stream.main_data + stream.md_len,
               stream.next_frame - frame_free, frame_free);
      stream.md_len += frame_free;
    }
  }
};
  
  

behavior Mad_layer_III(struct mad_stream stream,
                       struct mad_header header,
                       in int frame_options,
		           mad_fixed_t frame_overlap[32][18],
		       out mad_fixed_t frame_sbsample[36][32],
		           mad_fixed_t frame_overlap2[32][18],
		       out mad_fixed_t frame_sbsample2[36][32],
                       int result)
{
  unsigned int nch;
  unsigned int data_bitlen, priv_bitlen;
  unsigned int md_len, next_md_begin, frame_free;
  enum mad_error error;

  unsigned int si_main_data_begin, si_private_bits;
  unsigned char si_scfsi[2];
  struct sideinfo_channel si_ch0[2], si_ch1[2];
  struct mad_bitptr ptr;
  
  Mad_layer_III_init init(stream, header, frame_options, nch, result);

  Sideinfo sideinfo(stream, header, nch, 
                    si_main_data_begin, si_private_bits, si_scfsi,
                    si_ch0, si_ch1, data_bitlen, priv_bitlen, error);

  Mad_layer_III_main_data main_data(stream, header, error, priv_bitlen, 
                                    si_private_bits, si_main_data_begin,
                                    md_len, next_md_begin, frame_free, 
                                    ptr, result);

  III_decode decode(ptr, nch, header, si_scfsi, si_ch0, si_ch1, 
		    frame_overlap, frame_sbsample, 
		    frame_overlap2, frame_sbsample2,
		    error);
  
  Mad_layer_III_anc anc(stream, ptr, data_bitlen, md_len, error, result);
  
  Mad_layer_III_preload preload(stream, si_main_data_begin,
                                data_bitlen, md_len, next_md_begin,
                                frame_free);
  
  void main(void)
  {
    fsm {
      init: {
        if(result < -1) break;
      }
      
      sideinfo:
      
      main_data: {
        if(result != 0) goto preload;
      }

      decode:
      
      anc:
      
      preload:      
    }
  }
};


