/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: imdct.sc,v 1.4 2007/09/27 18:17:56 gerstl Exp $
 *
 * $Log: imdct.sc,v $
 * Revision 1.4  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

# include "global.sh"

# include "fixed.sh"
# include "layer3.sh"


/* --- IMDCT ----------------------------------------------------------- */

/*
 * IMDCT coefficients for short blocks
 * derived from section 2.4.3.4.10.2 of ISO/IEC 11172-3
 *
 * imdct_s[i/even][k] = cos((PI / 24) * (2 *       (i / 2) + 7) * (2 * k + 1))
 * imdct_s[i /odd][k] = cos((PI / 24) * (2 * (6 + (i-1)/2) + 7) * (2 * k + 1))
 */
static
mad_fixed_t const imdct_s[6][6] = {
# include "imdct_s.dat"
};

/*
 * windowing coefficients for long blocks
 * derived from section 2.4.3.4.10.3 of ISO/IEC 11172-3
 *
 * window_l[i] = sin((PI / 36) * (i + 1/2))
 */
static
mad_fixed_t const window_l[36] = {
  MAD_F(0x00b2aa3e) /* 0.043619387 */, MAD_F(0x0216a2a2) /* 0.130526192 */,
    MAD_F(0x03768962) /* 0.216439614 */, MAD_F(0x04cfb0e2) /* 0.300705800 */,
    MAD_F(0x061f78aa) /* 0.382683432 */, MAD_F(0x07635284) /* 0.461748613 */,
    MAD_F(0x0898c779) /* 0.537299608 */, MAD_F(0x09bd7ca0) /* 0.608761429 */,
    MAD_F(0x0acf37ad) /* 0.675590208 */, MAD_F(0x0bcbe352) /* 0.737277337 */,
    MAD_F(0x0cb19346) /* 0.793353340 */, MAD_F(0x0d7e8807) /* 0.843391446 */,
    
    MAD_F(0x0e313245) /* 0.887010833 */, MAD_F(0x0ec835e8) /* 0.923879533 */,
    MAD_F(0x0f426cb5) /* 0.953716951 */, MAD_F(0x0f9ee890) /* 0.976296007 */,
    MAD_F(0x0fdcf549) /* 0.991444861 */, MAD_F(0x0ffc19fd) /* 0.999048222 */,
    MAD_F(0x0ffc19fd) /* 0.999048222 */, MAD_F(0x0fdcf549) /* 0.991444861 */,
    MAD_F(0x0f9ee890) /* 0.976296007 */, MAD_F(0x0f426cb5) /* 0.953716951 */,
    MAD_F(0x0ec835e8) /* 0.923879533 */, MAD_F(0x0e313245) /* 0.887010833 */,
    
    MAD_F(0x0d7e8807) /* 0.843391446 */, MAD_F(0x0cb19346) /* 0.793353340 */,
    MAD_F(0x0bcbe352) /* 0.737277337 */, MAD_F(0x0acf37ad) /* 0.675590208 */,
    MAD_F(0x09bd7ca0) /* 0.608761429 */, MAD_F(0x0898c779) /* 0.537299608 */,
    MAD_F(0x07635284) /* 0.461748613 */, MAD_F(0x061f78aa) /* 0.382683432 */,
    MAD_F(0x04cfb0e2) /* 0.300705800 */, MAD_F(0x03768962) /* 0.216439614 */,
    MAD_F(0x0216a2a2) /* 0.130526192 */, MAD_F(0x00b2aa3e) /* 0.043619387 */,
};

/*
 * windowing coefficients for short blocks
 * derived from section 2.4.3.4.10.3 of ISO/IEC 11172-3
 *
 * window_s[i] = sin((PI / 12) * (i + 1/2))
 */
static
mad_fixed_t const window_s[12] = {
  MAD_F(0x0216a2a2) /* 0.130526192 */, MAD_F(0x061f78aa) /* 0.382683432 */,
    MAD_F(0x09bd7ca0) /* 0.608761429 */, MAD_F(0x0cb19346) /* 0.793353340 */,
    MAD_F(0x0ec835e8) /* 0.923879533 */, MAD_F(0x0fdcf549) /* 0.991444861 */,
    MAD_F(0x0fdcf549) /* 0.991444861 */, MAD_F(0x0ec835e8) /* 0.923879533 */,
    MAD_F(0x0cb19346) /* 0.793353340 */, MAD_F(0x09bd7ca0) /* 0.608761429 */,
    MAD_F(0x061f78aa) /* 0.382683432 */, MAD_F(0x0216a2a2) /* 0.130526192 */,
};


behavior Imdct(in struct sideinfo_channel si_channel[2],
               in unsigned int gr,
               in mad_fixed_t xr[576],
               mad_fixed_t frame_overlap[32][18],
               out mad_fixed_t frame_sbsample[36][32])
{
  void fastsdct(mad_fixed_t const x[9], mad_fixed_t y[18])
  {
    mad_fixed_t a0,  a1,  a2,  a3,  a4,  a5,  a6,  a7,  a8,  a9,  a10, a11, a12;
    mad_fixed_t a13, a14, a15, a16, a17, a18, a19, a20, a21, a22, a23, a24, a25;
    mad_fixed_t m0,  m1,  m2,  m3,  m4,  m5,  m6,  m7;
    
    enum {
      c0 =  MAD_F(0x1f838b8d),  /* 2 * cos( 1 * PI / 18) */
      c1 =  MAD_F(0x1bb67ae8),  /* 2 * cos( 3 * PI / 18) */
      c2 =  MAD_F(0x18836fa3),  /* 2 * cos( 4 * PI / 18) */
      c3 =  MAD_F(0x1491b752),  /* 2 * cos( 5 * PI / 18) */
      c4 =  MAD_F(0x0af1d43a),  /* 2 * cos( 7 * PI / 18) */
      c5 =  MAD_F(0x058e86a0),  /* 2 * cos( 8 * PI / 18) */
      c6 = -MAD_F(0x1e11f642)   /* 2 * cos(16 * PI / 18) */
    };
    
    a0 = x[3] + x[5];
    a1 = x[3] - x[5];
    a2 = x[6] + x[2];
    a3 = x[6] - x[2];
    a4 = x[1] + x[7];
    a5 = x[1] - x[7];
    a6 = x[8] + x[0];
    a7 = x[8] - x[0];
    
    a8  = a0  + a2;
    a9  = a0  - a2;
    a10 = a0  - a6;
    a11 = a2  - a6;
    a12 = a8  + a6;
    a13 = a1  - a3;
    a14 = a13 + a7;
    a15 = a3  + a7;
    a16 = a1  - a7;
    a17 = a1  + a3;
    
    m0 = mad_f_mul(a17, -c3);
    m1 = mad_f_mul(a16, -c0);
    m2 = mad_f_mul(a15, -c4);
    m3 = mad_f_mul(a14, -c1);
    m4 = mad_f_mul(a5,  -c1);
    m5 = mad_f_mul(a11, -c6);
    m6 = mad_f_mul(a10, -c5);
    m7 = mad_f_mul(a9,  -c2);
    
    a18 =     x[4] + a4;
    a19 = 2 * x[4] - a4;
    a20 = a19 + m5;
    a21 = a19 - m5;
    a22 = a19 + m6;
    a23 = m4  + m2;
    a24 = m4  - m2;
    a25 = m4  + m1;
    
    /* output to every other slot for convenience */
    
    y[ 0] = a18 + a12;
    y[ 2] = m0  - a25;
    y[ 4] = m7  - a20;
    y[ 6] = m3;
    y[ 8] = a21 - m6;
    y[10] = a24 - m1;
    y[12] = a12 - 2 * a18;
    y[14] = a23 + m0;
    y[16] = a22 + m7;
  }
  
  void sdctII(mad_fixed_t const x[18], mad_fixed_t X[18])
  {
    mad_fixed_t tmp[9];
    int i;
    
    /* scale[i] = 2 * cos(PI * (2 * i + 1) / (2 * 18)) */
    static mad_fixed_t const scale[9] = {
      MAD_F(0x1fe0d3b4), MAD_F(0x1ee8dd47), MAD_F(0x1d007930),
      MAD_F(0x1a367e59), MAD_F(0x16a09e66), MAD_F(0x125abcf8),
      MAD_F(0x0d8616bc), MAD_F(0x08483ee1), MAD_F(0x02c9fad7)
    };
    
    /* divide the 18-point SDCT-II into two 9-point SDCT-IIs */

    /* even input butterfly */

    for (i = 0; i < 9; i += 3) {
      tmp[i + 0] = x[i + 0] + x[18 - (i + 0) - 1];
      tmp[i + 1] = x[i + 1] + x[18 - (i + 1) - 1];
      tmp[i + 2] = x[i + 2] + x[18 - (i + 2) - 1];
    }

    fastsdct(tmp, &X[0]);

    /* odd input butterfly and scaling */

    for (i = 0; i < 9; i += 3) {
      tmp[i + 0] = mad_f_mul(x[i + 0] - x[18 - (i + 0) - 1], scale[i + 0]);
      tmp[i + 1] = mad_f_mul(x[i + 1] - x[18 - (i + 1) - 1], scale[i + 1]);
      tmp[i + 2] = mad_f_mul(x[i + 2] - x[18 - (i + 2) - 1], scale[i + 2]);
    }
    
    fastsdct(tmp, &X[1]);
    
    /* output accumulation */
    
    for (i = 3; i < 18; i += 8) {
      X[i + 0] -= X[(i + 0) - 2];
      X[i + 2] -= X[(i + 2) - 2];
      X[i + 4] -= X[(i + 4) - 2];
      X[i + 6] -= X[(i + 6) - 2];
    }
  }
  
  void dctIV(mad_fixed_t const y[18], mad_fixed_t X[18])
  {
    mad_fixed_t tmp[18];
    int i;
    
    /* scale[i] = 2 * cos(PI * (2 * i + 1) / (4 * 18)) */
    static mad_fixed_t const scale[18] = {
      MAD_F(0x1ff833fa), MAD_F(0x1fb9ea93), MAD_F(0x1f3dd120),
      MAD_F(0x1e84d969), MAD_F(0x1d906bcf), MAD_F(0x1c62648b),
      MAD_F(0x1afd100f), MAD_F(0x1963268b), MAD_F(0x1797c6a4),
      MAD_F(0x159e6f5b), MAD_F(0x137af940), MAD_F(0x11318ef3),
      MAD_F(0x0ec6a507), MAD_F(0x0c3ef153), MAD_F(0x099f61c5),
      MAD_F(0x06ed12c5), MAD_F(0x042d4544), MAD_F(0x0165547c)
    };
    
    /* scaling */
    
    for (i = 0; i < 18; i += 3) {
      tmp[i + 0] = mad_f_mul(y[i + 0], scale[i + 0]);
      tmp[i + 1] = mad_f_mul(y[i + 1], scale[i + 1]);
      tmp[i + 2] = mad_f_mul(y[i + 2], scale[i + 2]);
    }

    /* SDCT-II */
    
    sdctII(tmp, X);
    
    /* scale reduction and output accumulation */
    
    X[0] /= 2;
    for (i = 1; i < 17; i += 4) {
      X[i + 0] = X[i + 0] / 2 - X[(i + 0) - 1];
      X[i + 1] = X[i + 1] / 2 - X[(i + 1) - 1];
      X[i + 2] = X[i + 2] / 2 - X[(i + 2) - 1];
      X[i + 3] = X[i + 3] / 2 - X[(i + 3) - 1];
    }
    X[17] = X[17] / 2 - X[16];
  }
  
  /*
   * NAME:	imdct36
   * DESCRIPTION:	perform X[18]->x[36] IMDCT using Szu-Wei Lee's fast algorithm
   */
  void imdct36(mad_fixed_t const x[18], mad_fixed_t y[36])
  {
    mad_fixed_t tmp[18];
    int i;
    
    /* DCT-IV */
    
    dctIV(x, tmp);
    
    /* convert 18-point DCT-IV to 36-point IMDCT */
    
    for (i =  0; i <  9; i += 3) {
      y[i + 0] =  tmp[9 + (i + 0)];
      y[i + 1] =  tmp[9 + (i + 1)];
      y[i + 2] =  tmp[9 + (i + 2)];
    }
    for (i =  9; i < 27; i += 3) {
      y[i + 0] = -tmp[36 - (9 + (i + 0)) - 1];
      y[i + 1] = -tmp[36 - (9 + (i + 1)) - 1];
      y[i + 2] = -tmp[36 - (9 + (i + 2)) - 1];
    }
    for (i = 27; i < 36; i += 3) {
      y[i + 0] = -tmp[(i + 0) - 27];
      y[i + 1] = -tmp[(i + 1) - 27];
      y[i + 2] = -tmp[(i + 2) - 27];
    }
  }
  
  /*
   * NAME:	III_imdct_l()
   * DESCRIPTION:	perform IMDCT and windowing for long blocks
   */
  void III_imdct_l(mad_fixed_t const X[18], mad_fixed_t z[36],
                   unsigned int block_type)
  {
    unsigned int i;
    
    /* IMDCT */
    
    
    /* windowing */
    
    switch (block_type) {
      case 0:  /* normal window */
# if defined(ASO_INTERLEAVE1)
      {
        register mad_fixed_t tmp1, tmp2;
        
        tmp1 = window_l[0];
        tmp2 = window_l[1];
        
        for (i = 0; i < 34; i += 2) {
          z[i + 0] = mad_f_mul(z[i + 0], tmp1);
          tmp1 = window_l[i + 2];
          z[i + 1] = mad_f_mul(z[i + 1], tmp2);
          tmp2 = window_l[i + 3];
        }
        
        z[34] = mad_f_mul(z[34], tmp1);
        z[35] = mad_f_mul(z[35], tmp2);
      }
# elif defined(ASO_INTERLEAVE2)
      {
        register mad_fixed_t tmp1, tmp2;
        
        tmp1 = z[0];
        tmp2 = window_l[0];
        
        for (i = 0; i < 35; ++i) {
          z[i] = mad_f_mul(tmp1, tmp2);
          tmp1 = z[i + 1];
          tmp2 = window_l[i + 1];
        }
        
        z[35] = mad_f_mul(tmp1, tmp2);
      }
# elif 1
      for (i = 0; i < 36; i += 4) {
        z[i + 0] = mad_f_mul(z[i + 0], window_l[i + 0]);
        z[i + 1] = mad_f_mul(z[i + 1], window_l[i + 1]);
        z[i + 2] = mad_f_mul(z[i + 2], window_l[i + 2]);
        z[i + 3] = mad_f_mul(z[i + 3], window_l[i + 3]);
      }
# else
      for (i =  0; i < 36; ++i) z[i] = mad_f_mul(z[i], window_l[i]);
# endif
      break;
      
      case 1:  /* start block */
      for (i =  0; i < 18; i += 3) {
        z[i + 0] = mad_f_mul(z[i + 0], window_l[i + 0]);
        z[i + 1] = mad_f_mul(z[i + 1], window_l[i + 1]);
        z[i + 2] = mad_f_mul(z[i + 2], window_l[i + 2]);
      }
      /*  (i = 18; i < 24; ++i) z[i] unchanged */
      for (i = 24; i < 30; ++i) z[i] = mad_f_mul(z[i], window_s[i - 18]);
      for (i = 30; i < 36; ++i) z[i] = 0;
      break;
      
      case 3:  /* stop block */
      for (i =  0; i <  6; ++i) z[i] = 0;
      for (i =  6; i < 12; ++i) z[i] = mad_f_mul(z[i], window_s[i - 6]);
      /*  (i = 12; i < 18; ++i) z[i] unchanged */
      for (i = 18; i < 36; i += 3) {
        z[i + 0] = mad_f_mul(z[i + 0], window_l[i + 0]);
        z[i + 1] = mad_f_mul(z[i + 1], window_l[i + 1]);
        z[i + 2] = mad_f_mul(z[i + 2], window_l[i + 2]);
      }
      break;
    }
  }
  
  /*
   * NAME:	III_imdct_s()
   * DESCRIPTION:	perform IMDCT and windowing for short blocks
   */
  void III_imdct_s(mad_fixed_t const X[18], mad_fixed_t z[36])
  {
    mad_fixed_t y[36], *yptr;
    mad_fixed_t const *wptr, *x;
    int w, i;
    register mad_fixed64hi_t hi;
    register mad_fixed64lo_t lo;
    
    /* IMDCT */
    
    yptr = &y[0];
    x = &X[0];
    
    for (w = 0; w < 3; ++w) {
      register mad_fixed_t const (*s)[6];
      
      s = imdct_s;
      
      for (i = 0; i < 3; ++i) {
        MAD_F_ML0(hi, lo, x[0], (*s)[0]);
        MAD_F_MLA(hi, lo, x[1], (*s)[1]);
        MAD_F_MLA(hi, lo, x[2], (*s)[2]);
        MAD_F_MLA(hi, lo, x[3], (*s)[3]);
        MAD_F_MLA(hi, lo, x[4], (*s)[4]);
        MAD_F_MLA(hi, lo, x[5], (*s)[5]);
        
        yptr[i + 0] = MAD_F_MLZ(hi, lo);
        yptr[5 - i] = -yptr[i + 0];
        
        ++s;
        
        MAD_F_ML0(hi, lo, x[0], (*s)[0]);
        MAD_F_MLA(hi, lo, x[1], (*s)[1]);
        MAD_F_MLA(hi, lo, x[2], (*s)[2]);
        MAD_F_MLA(hi, lo, x[3], (*s)[3]);
        MAD_F_MLA(hi, lo, x[4], (*s)[4]);
        MAD_F_MLA(hi, lo, x[5], (*s)[5]);
        
        yptr[ i + 6] = MAD_F_MLZ(hi, lo);
        yptr[11 - i] = yptr[i + 6];
        
        ++s;
      }

      yptr += 12;
      x    += 6;
    }
    
    /* windowing, overlapping and concatenation */
    
    yptr = &y[0];
    wptr = &window_s[0];
    
    for (i = 0; i < 6; ++i) {
      z[i +  0] = 0;
      z[i +  6] = mad_f_mul(yptr[ 0 + 0], wptr[0]);
      
      MAD_F_ML0(hi, lo, yptr[ 0 + 6], wptr[6]);
      MAD_F_MLA(hi, lo, yptr[12 + 0], wptr[0]);
      
      z[i + 12] = MAD_F_MLZ(hi, lo);
      
      MAD_F_ML0(hi, lo, yptr[12 + 6], wptr[6]);
      MAD_F_MLA(hi, lo, yptr[24 + 0], wptr[0]);
      
      z[i + 18] = MAD_F_MLZ(hi, lo);
      
      z[i + 24] = mad_f_mul(yptr[24 + 6], wptr[6]);
      z[i + 30] = 0;
      
      ++yptr;
      ++wptr;
    }
  }
  
  /*
   * NAME:	III_overlap()
   * DESCRIPTION:	perform overlap-add of windowed IMDCT outputs
   */
  void III_overlap(mad_fixed_t const output[36], mad_fixed_t overlap[18],
                   mad_fixed_t sample[18][32], unsigned int sb)
  {
    unsigned int i;

# if defined(ASO_INTERLEAVE2)
    {
      register mad_fixed_t tmp1, tmp2;

      tmp1 = overlap[0];
      tmp2 = overlap[1];
      
      for (i = 0; i < 16; i += 2) {
        sample[i + 0][sb] = output[i + 0 +  0] + tmp1;
        overlap[i + 0]    = output[i + 0 + 18];
        tmp1 = overlap[i + 2];

        sample[i + 1][sb] = output[i + 1 +  0] + tmp2;
        overlap[i + 1]    = output[i + 1 + 18];
        tmp2 = overlap[i + 3];
      }

      sample[16][sb] = output[16 +  0] + tmp1;
      overlap[16]    = output[16 + 18];
      sample[17][sb] = output[17 +  0] + tmp2;
      overlap[17]    = output[17 + 18];
    }
# elif 0
    for (i = 0; i < 18; i += 2) {
      sample[i + 0][sb] = output[i + 0 +  0] + overlap[i + 0];
      overlap[i + 0]    = output[i + 0 + 18];
      
      sample[i + 1][sb] = output[i + 1 +  0] + overlap[i + 1];
      overlap[i + 1]    = output[i + 1 + 18];
    }
# else
    for (i = 0; i < 18; ++i) {
      sample[i][sb] = output[i +  0] + overlap[i];
      overlap[i]    = output[i + 18];
    }
# endif
  }
  
  /*
   * NAME:	III_overlap_z()
   * DESCRIPTION:	perform "overlap-add" of zero IMDCT outputs
   */
  void III_overlap_z(mad_fixed_t overlap[18],
                     mad_fixed_t sample[18][32], unsigned int sb)
  {
    unsigned int i;
    
# if defined(ASO_INTERLEAVE2)
    {
      register mad_fixed_t tmp1, tmp2;
      
      tmp1 = overlap[0];
      tmp2 = overlap[1];
      
      for (i = 0; i < 16; i += 2) {
        sample[i + 0][sb] = tmp1;
        overlap[i + 0]    = 0;
        tmp1 = overlap[i + 2];
        
        sample[i + 1][sb] = tmp2;
        overlap[i + 1]    = 0;
        tmp2 = overlap[i + 3];
      }
      
      sample[16][sb] = tmp1;
      overlap[16]    = 0;
      sample[17][sb] = tmp2;
      overlap[17]    = 0;
    }
# else
    for (i = 0; i < 18; ++i) {
      sample[i][sb] = overlap[i];
      overlap[i]    = 0;
    }
# endif
  }
  
  /*
   * NAME:	III_freqinver()
   * DESCRIPTION:	perform subband frequency inversion for odd sample lines
   */
  void III_freqinver(mad_fixed_t sample[18][32], unsigned int sb)
  {
    unsigned int i;
    
# if 1 || defined(ASO_INTERLEAVE1) || defined(ASO_INTERLEAVE2)
    {
      register mad_fixed_t tmp1, tmp2;
      
      tmp1 = sample[1][sb];
      tmp2 = sample[3][sb];
      
      for (i = 1; i < 13; i += 4) {
        sample[i + 0][sb] = -tmp1;
        tmp1 = sample[i + 4][sb];
        sample[i + 2][sb] = -tmp2;
        tmp2 = sample[i + 6][sb];
      }
      
      sample[13][sb] = -tmp1;
      tmp1 = sample[17][sb];
      sample[15][sb] = -tmp2;
      sample[17][sb] = -tmp1;
    }
# else
    for (i = 1; i < 18; i += 2)
      sample[i][sb] = -sample[i][sb];
# endif
  }
  
  
  void main(void)
  {
      unsigned int sb, l, i, sblimit;
      unsigned int block_type;
      mad_fixed_t output[36];
      mad_fixed_t (*sample)[32];
    
      sample = &frame_sbsample[18 * gr];
      l = 0;

      /* subbands 0-1 */
      block_type = si_channel[gr].block_type;
      if (si_channel[gr].flags & mixed_block_flag)
	block_type = 0;

      for (sb = 0; sb < 2; ++sb, l += 18) {
	if (si_channel[gr].block_type != 2 || (si_channel[gr].flags & mixed_block_flag)) {
	  /* long blocks */
	  imdct36(&xr[l], output); /*** send */

	  III_imdct_l(&xr[l], output, block_type); /*** receive */
	}
	else {
	  /* short blocks */
	  III_imdct_s(&xr[l], output);
	}
	III_overlap(output, (frame_overlap)[sb], sample, sb);
	if (sb & 1)
	    III_freqinver(sample, sb);
      }

      /* (nonzero) subbands 2-31 */
      i = 576;
      while (i > 36 && xr[i - 1] == 0)
	--i;
      sblimit = 32 - (576 - i) / 18;

      for (sb = 2; sb < sblimit; ++sb, l += 18) {
	if (si_channel[gr].block_type != 2) {
	  /* long blocks */
          imdct36(&xr[l], output);  /*** send */

          III_imdct_l(&xr[l], output, si_channel[gr].block_type); /*** receive */
	}
	else {
	  /* short blocks */
          III_imdct_s(&xr[l], output);
	}
        III_overlap(output, (frame_overlap)[sb], sample, sb);
	    
        if (sb & 1)
          III_freqinver(sample, sb);
      }

      /* remaining (zero) subbands */
      for (sb = sblimit; sb < 32; ++sb) {
	III_overlap_z((frame_overlap)[sb], sample, sb);

	if (sb & 1)
	  III_freqinver(sample, sb);
      }
  }
};
  
