/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: huffman.sc,v 1.5 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: huffman.sc,v $
 * Revision 1.5  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

#include <assert.h>

# include "global.sh"

import "global";

# include "fixed.sh"
# include "stream.sh"
# include "layer3.sh"

# include "bit.sh"

import "bit";

#include "huffman.h"


/*
 * fractional powers of two
 * used for requantization and joint stereo decoding
 *
 * root_table[3 + x] = 2^(x/4)
 */
mad_fixed_t const root_table[7] = {
  MAD_F(0x09837f05) /* 2^(-3/4) == 0.59460355750136 */,
    MAD_F(0x0b504f33) /* 2^(-2/4) == 0.70710678118655 */,
    MAD_F(0x0d744fcd) /* 2^(-1/4) == 0.84089641525371 */,
    MAD_F(0x10000000) /* 2^( 0/4) == 1.00000000000000 */,
    MAD_F(0x1306fe0a) /* 2^(+1/4) == 1.18920711500272 */,
    MAD_F(0x16a09e66) /* 2^(+2/4) == 1.41421356237310 */,
    MAD_F(0x1ae89f99) /* 2^(+3/4) == 1.68179283050743 */
};  


/*
 * scalefactor band preemphasis (used only when preflag is set)
 * derived from Table B.6 of ISO/IEC 11172-3
 */
static
unsigned char const pretab[22] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 3, 3, 3, 2, 0
};

/*
 * table for requantization
 *
 * rq_table[x].mantissa * 2^(rq_table[x].exponent) = x^(4/3)
 */
static
struct fixedfloat {
  uln mantissa  : 27;
  unsigned exponent :  5;
} const rq_table[8207] = {
# include "rq_table.dat"
};



behavior Huffman(struct mad_bitptr bitptr,
                 in unsigned int gr,
                 in unsigned int ch,
                 in struct sideinfo_channel si_channel[2],
                 in unsigned char scalefac[2][2][39],
		 in unsigned char sfbwidth[39],
		 in unsigned int part2_length,
                 out mad_fixed_t xr[576],
                 out enum mad_error error)
{
  /*
   * The Layer III formula for requantization and scaling is defined by
   * section 2.4.3.4.7.1 of ISO/IEC 11172-3, as follows:
   *
   *   long blocks:
   *   xr[i] = sign(is[i]) * abs(is[i])^(4/3) *
   *           2^((1/4) * (global_gain - 210)) *
   *           2^-(scalefac_multiplier *
   *               (scalefac_l[sfb] + preflag * pretab[sfb]))
   *
   *   short blocks:
   *   xr[i] = sign(is[i]) * abs(is[i])^(4/3) *
   *           2^((1/4) * (global_gain - 210 - 8 * subblock_gain[w])) *
   *           2^-(scalefac_multiplier * scalefac_s[sfb][w])
   *
   *   where:
   *   scalefac_multiplier = (scalefac_scale + 1) / 2
   *
   * The routines III_exponents() and III_requantize() facilitate this
   * calculation.
   */
  
  /*
   * NAME:	III_exponents()
   * DESCRIPTION:	calculate scalefactor exponents
   */
  void III_exponents(unsigned char const *sfbwidth, signed int exponents[39])
  {
    signed int gain;
    unsigned int scalefac_multiplier, sfbi;
    
    gain = (signed int) si_channel[gr].global_gain - 210;
    scalefac_multiplier = (si_channel[gr].flags & scalefac_scale) ? 2 : 1;
    
    if (si_channel[gr].block_type == 2) {
      unsigned int l;
      signed int gain0, gain1, gain2;
      
      sfbi = l = 0;
      
      if (si_channel[gr].flags & mixed_block_flag) {
        unsigned int premask;
        
        premask = (si_channel[gr].flags & preflag) ? ~0 : 0;
        
        /* long block subbands 0-1 */
        
        while (l < 36) {
          exponents[sfbi] = gain -
            (signed int) ((scalefac[gr][ch][sfbi] + (pretab[sfbi] & premask)) <<
                          scalefac_multiplier);
          
          l += sfbwidth[sfbi++];
        }
      }
      
      /* this is probably wrong for 8000 Hz short/mixed blocks */
      
      gain0 = gain - 8 * (signed int) si_channel[gr].subblock_gain[0];
      gain1 = gain - 8 * (signed int) si_channel[gr].subblock_gain[1];
      gain2 = gain - 8 * (signed int) si_channel[gr].subblock_gain[2];
      
      while (l < 576) {
        exponents[sfbi + 0] = gain0 -
          (signed int) (scalefac[gr][ch][sfbi + 0] << scalefac_multiplier);
        exponents[sfbi + 1] = gain1 -
          (signed int) (scalefac[gr][ch][sfbi + 1] << scalefac_multiplier);
        exponents[sfbi + 2] = gain2 -
          (signed int) (scalefac[gr][ch][sfbi + 2] << scalefac_multiplier);
        
        l    += 3 * sfbwidth[sfbi];
        sfbi += 3;
      }
    }
    else {  /* channel->block_type != 2 */
      if (si_channel[gr].flags & preflag) {
        for (sfbi = 0; sfbi < 22; ++sfbi) {
          exponents[sfbi] = gain -
            (signed int) ((scalefac[gr][ch][sfbi] + pretab[sfbi]) <<
                          scalefac_multiplier);
        }
      }
      else {
        for (sfbi = 0; sfbi < 22; ++sfbi) {
          exponents[sfbi] = gain -
            (signed int) (scalefac[gr][ch][sfbi] << scalefac_multiplier);
        }
      }
    }
  }
  
  /*
   * NAME:	III_requantize()
   * DESCRIPTION:	requantize one (positive) value
   */
  mad_fixed_t III_requantize(unsigned int value, signed int exp)
  {
    mad_fixed_t requantized;
    signed int frac;
    struct fixedfloat const *power;
    
    frac = exp % 4;  /* assumes sign(frac) == sign(exp) */
    exp /= 4;
    
    power = &rq_table[value];
    requantized = power->mantissa;
    exp += power->exponent;
    
    if (exp < 0) {
      //PC:if (-exp >= sizeof(mad_fixed_t) * CHAR_BIT) {
      if (-exp >= SIZEOF_INT * CHAR_BIT) {
        /* underflow */
        requantized = 0;
      }
      else {
        requantized += 1L << (-exp - 1);
        requantized >>= -exp;
      }
    }
    else {
      if (exp >= 5) {
        /* overflow */
# if defined(DEBUG)
        fprintf(stderr, "requantize overflow (%f * 2^%d)\n",
                mad_f_todouble(requantized), exp);
# endif
        requantized = MAD_F_MAX;
      }
      else
        requantized <<= exp;
    }
    
    return frac ? mad_f_mul(requantized, root_table[3 + frac]) : requantized;
  }
  
/* we must take care that sz >= bits and sz < sizeof(long) lest bits == 0 */
# define MASK(cache, sz, bits)	\
    (((cache) >> ((sz) - (bits))) & ((1 << (bits)) - 1))
# define MASK1BIT(cache, sz)  \
    ((cache) & (1 << ((sz) - 1)))

  /*
   * NAME:	III_huffdecode()
   * DESCRIPTION:	decode Huffman code words of one channel of one granule
   */
  enum mad_error III_huffdecode(struct mad_bitptr *ptr, mad_fixed_t xr[576],
                                unsigned char const *sfbwidth,
                                unsigned int part2_length)
  {
    signed int exponents[39], exp;
    signed int const *expptr;
    struct mad_bitptr peek;
    signed int bits_left, cachesz;
    register mad_fixed_t *xrptr;
    mad_fixed_t const *sfbound;
    register uln bitcache;
    
    bits_left = (signed) si_channel[gr].part2_3_length - (signed) part2_length;
    if (bits_left < 0)
      return MAD_ERROR_BADPART3LEN;
    
    III_exponents(sfbwidth, exponents);
    
    //peek = *ptr;
    mymemcpy(&peek, ptr, sizeof(struct mad_bitptr));
    
    
    mad_bit_skip(ptr, bits_left);
    
    /* align bit reads to byte boundaries */
    cachesz  = mad_bit_bitsleft(&peek);
    cachesz += ((32 - 1 - 24) + (24 - cachesz)) & ~7;
    
    bitcache   = mad_bit_read(&peek, cachesz);
    bits_left -= cachesz;
    
    xrptr = &xr[0];
    
    /* big_values */
    {
      unsigned int region, rcount;
      struct hufftable const *entry;
      union huffpair const *table;
      unsigned int linbits, startbits, big_values, reqhits;
      mad_fixed_t reqcache[16];
      
      sfbound = xrptr + *sfbwidth++;
      rcount  = si_channel[gr].region0_count + 1;
      
      entry     = &mad_huff_pair_table[si_channel[gr].table_select[region = 0]];
      table     = entry->table;
      linbits   = entry->linbits;
      startbits = entry->startbits;
      
      if (table == 0)
        return MAD_ERROR_BADHUFFTABLE;
      
      expptr  = &exponents[0];
      exp     = *expptr++;
      reqhits = 0;
      
      big_values = si_channel[gr].big_values;
      
      while (big_values-- && cachesz + bits_left > 0) {
        union huffpair const *pair;
        unsigned int clumpsz, value;
        register mad_fixed_t requantized;
        
        if (xrptr == sfbound) {
          sfbound += *sfbwidth++;
          
          /* change table if region boundary */
          
          if (--rcount == 0) {
            if (region == 0)
              rcount = si_channel[gr].region1_count + 1;
            else
              rcount = 0;  /* all remaining */
            
            entry     = &mad_huff_pair_table[si_channel[gr].table_select[++region]];
            table     = entry->table;
            linbits   = entry->linbits;
            startbits = entry->startbits;
            
            if (table == 0)
              return MAD_ERROR_BADHUFFTABLE;
          }
          
          if (exp != *expptr) {
            exp = *expptr;
            reqhits = 0;
          }
          
          ++expptr;
        }
        
        if (cachesz < 21) {
          unsigned int bits;
          
          bits       = ((32 - 1 - 21) + (21 - cachesz)) & ~7;
          bitcache   = (bitcache << bits) | mad_bit_read(&peek, bits);
          cachesz   += bits;
          bits_left -= bits;
        }
        
        /* hcod (0..19) */
        
        clumpsz = startbits;
        pair    = &table[MASK(bitcache, cachesz, clumpsz)];
        
        while (!pair->final) {
          cachesz -= clumpsz;
          
          clumpsz = pair->ptr.bits;
          pair    = &table[pair->ptr.offset + MASK(bitcache, cachesz, clumpsz)];
        }
        
        cachesz -= pair->value.hlen;
        
        if (linbits) {
          /* x (0..14) */
          
          value = pair->value.x;
          
          switch (value) {
            case 0:
            xrptr[0] = 0;
            break;
            
            case 15:
            if (cachesz < linbits + 2) {
              bitcache   = (bitcache << 16) | mad_bit_read(&peek, 16);
              cachesz   += 16;
              bits_left -= 16;
            }
            
            value += MASK(bitcache, cachesz, linbits);
            cachesz -= linbits;
            
            requantized = III_requantize(value, exp);
            goto x_final;
            
            default:
            if (reqhits & (1 << value))
              requantized = reqcache[value];
            else {
              reqhits |= (1 << value);
              requantized = reqcache[value] = III_requantize(value, exp);
            }
            
            x_final:
            xrptr[0] = MASK1BIT(bitcache, cachesz--) ?
              -requantized : requantized;
          }
          
          /* y (0..14) */
          
          value = pair->value.y;
          
          switch (value) {
            case 0:
            xrptr[1] = 0;
            break;
            
            case 15:
            if (cachesz < linbits + 1) {
              bitcache   = (bitcache << 16) | mad_bit_read(&peek, 16);
              cachesz   += 16;
              bits_left -= 16;
            }
            
            value += MASK(bitcache, cachesz, linbits);
            cachesz -= linbits;
            
            requantized = III_requantize(value, exp);
            goto y_final;
            
            default:
            if (reqhits & (1 << value))
              requantized = reqcache[value];
            else {
              reqhits |= (1 << value);
              requantized = reqcache[value] = III_requantize(value, exp);
            }
            
            y_final:
            xrptr[1] = MASK1BIT(bitcache, cachesz--) ?
              -requantized : requantized;
          }
        }
        else {
          /* x (0..1) */
          
          value = pair->value.x;
          
          if (value == 0)
            xrptr[0] = 0;
          else {
            if (reqhits & (1 << value))
              requantized = reqcache[value];
            else {
              reqhits |= (1 << value);
              requantized = reqcache[value] = III_requantize(value, exp);
            }
            
            xrptr[0] = MASK1BIT(bitcache, cachesz--) ?
              -requantized : requantized;
          }
          
          /* y (0..1) */
          
          value = pair->value.y;
          
          if (value == 0)
            xrptr[1] = 0;
          else {
            if (reqhits & (1 << value))
              requantized = reqcache[value];
            else {
              reqhits |= (1 << value);
              requantized = reqcache[value] = III_requantize(value, exp);
            }
            
            xrptr[1] = MASK1BIT(bitcache, cachesz--) ?
              -requantized : requantized;
          }
        }
        
        xrptr += 2;
      }
    }
    
    if (cachesz + bits_left < 0)
      return MAD_ERROR_BADHUFFDATA;  /* big_values overrun */
    
    /* count1 */
    {
      union huffquad const *table;
      register mad_fixed_t requantized;
      
      table = mad_huff_quad_table[si_channel[gr].flags & count1table_select];
      
      requantized = III_requantize(1, exp);
      
      while (cachesz + bits_left > 0 && xrptr <= &xr[572]) {
        union huffquad const *quad;
        
        /* hcod (1..6) */
        
        if (cachesz < 10) {
          bitcache   = (bitcache << 16) | mad_bit_read(&peek, 16);
          cachesz   += 16;
          bits_left -= 16;
        }
        
        quad = &table[MASK(bitcache, cachesz, 4)];
        
        /* quad tables guaranteed to have at most one extra lookup */
        if (!quad->final) {
          cachesz -= 4;
          
          quad = &table[quad->ptr.offset +
                        MASK(bitcache, cachesz, quad->ptr.bits)];
        }
        
        cachesz -= quad->value.hlen;
        
        if (xrptr == sfbound) {
          sfbound += *sfbwidth++;
          
          if (exp != *expptr) {
            exp = *expptr;
            requantized = III_requantize(1, exp);
          }
          
          ++expptr;
        }
        
        /* v (0..1) */
        
        xrptr[0] = quad->value.v ?
          (MASK1BIT(bitcache, cachesz--) ? -requantized : requantized) : 0;
        
        /* w (0..1) */
        
        xrptr[1] = quad->value.w ?
          (MASK1BIT(bitcache, cachesz--) ? -requantized : requantized) : 0;
        
        xrptr += 2;
        
        if (xrptr == sfbound) {
          sfbound += *sfbwidth++;
          
          if (exp != *expptr) {
            exp = *expptr;
            requantized = III_requantize(1, exp);
          }
          
          ++expptr;
        }
        
        /* x (0..1) */
        
        xrptr[0] = quad->value.x ?
          (MASK1BIT(bitcache, cachesz--) ? -requantized : requantized) : 0;
        
        /* y (0..1) */
        
        xrptr[1] = quad->value.y ?
          (MASK1BIT(bitcache, cachesz--) ? -requantized : requantized) : 0;
        
        xrptr += 2;
      }
      
      if (cachesz + bits_left < 0) {
# if 0 && defined(DEBUG)
        fprintf(stderr, "huffman count1 overrun (%d bits)\n",
                -(cachesz + bits_left));
# endif
        
        /* technically the bitstream is misformatted, but apparently
	 some encoders are just a bit sloppy with stuffing bits */
        
        xrptr -= 4;
      }
    }
    
    assert(-bits_left <= MAD_BUFFER_GUARD * CHAR_BIT);
    
# if 0 && defined(DEBUG)
    if (bits_left < 0)
      fprintf(stderr, "read %d bits too many\n", -bits_left);
    else if (cachesz + bits_left > 0)
      fprintf(stderr, "%d stuffing bits\n", cachesz + bits_left);
# endif
    
    /* rzero */
    while (xrptr < &xr[576]) {
      xrptr[0] = 0;
      xrptr[1] = 0;
      
      xrptr += 2;
    }
    
    return MAD_ERROR_NONE;
  }
  
# undef MASK
# undef MASK1BIT
  
  void main(void)
  {
    error = III_huffdecode(&bitptr, &xr[0], sfbwidth, part2_length);
  }
};
