/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: header.sc,v 1.5 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: header.sc,v $
 * Revision 1.5  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

# include "global.sh"

import "global";

# include "bit.sh"
# include "stream.sh"
# include "frame.sh"

import "bit";
import "timer";


/*
 * NAME:	header->decode()
 * DESCRIPTION:	read the next frame header from the stream
 */
behavior Mad_header_decode(struct mad_header header, 
                           struct mad_stream stream,
                           out int more_input)
{
  /*
   * NAME:	stream->sync()
   * DESCRIPTION:	locate the next stream sync word
   */
  int mad_stream_sync(struct mad_stream *pStream)
  {
    register unsigned char const *ptr, *end;
    
    ptr = mad_bit_nextbyte(&pStream->ptr);
    end = pStream->bufend;
    
    while (ptr < end - 1 &&
	   !(ptr[0] == 0xff && (ptr[1] & 0xe0) == 0xe0))
      ++ptr;
    
    if (end - ptr < MAD_BUFFER_GUARD)
      return -1;
    
    mad_bit_init(&pStream->ptr, ptr);
    
    return 0;
  }
  
  
// uln const bitrate_table[5][15] = { // by dongwans
  uln bitrate_table[5][15] = {
    /* MPEG-1 */
    { 0,  32000,  64000,  96000, 128000, 160000, 192000, 224000,  /* Layer I   */
        256000, 288000, 320000, 352000, 384000, 416000, 448000 },
    { 0,  32000,  48000,  56000,  64000,  80000,  96000, 112000,  /* Layer II  */
        128000, 160000, 192000, 224000, 256000, 320000, 384000 },
    { 0,  32000,  40000,  48000,  56000,  64000,  80000,  96000,  /* Layer III */
        112000, 128000, 160000, 192000, 224000, 256000, 320000 },
    
    /* MPEG-2 LSF */
    { 0,  32000,  48000,  56000,  64000,  80000,  96000, 112000,  /* Layer I   */
        128000, 144000, 160000, 176000, 192000, 224000, 256000 },
    { 0,   8000,  16000,  24000,  32000,  40000,  48000,  56000,  /* Layers    */
        64000,  80000,  96000, 112000, 128000, 144000, 160000 } /* II & III  */
  };
  
//  unsigned int const samplerate_table[3] = { 44100, 48000, 32000 };   // by dongwans
  unsigned int samplerate_table[3] = { 44100, 48000, 32000 };
  
  /*
   * NAME:	decode_header()
   * DESCRIPTION:	read header data and following CRC word
   */
  int decode_header(struct mad_header *pHeader, struct mad_stream *pStream)
  {
    unsigned int index;
    
    pHeader->flags        = 0;
    pHeader->private_bits = 0;
    
    /* header() */
    
    /* syncword */
    mad_bit_skip(&pStream->ptr, 11);
    
    /* MPEG 2.5 indicator (really part of syncword) */
    if (mad_bit_read(&pStream->ptr, 1) == 0)
      pHeader->flags |= MAD_FLAG_MPEG_2_5_EXT;
    
    /* ID */
    if (mad_bit_read(&pStream->ptr, 1) == 0)
      pHeader->flags |= MAD_FLAG_LSF_EXT;
    else if (pHeader->flags & MAD_FLAG_MPEG_2_5_EXT) {
      pStream->error = MAD_ERROR_LOSTSYNC;
      return -1;
    }
    
    /* layer */
    pHeader->layer = (enum mad_layer)(4 - mad_bit_read(&pStream->ptr, 2));
    
    if ((int)pHeader->layer == 4) {
      pStream->error = MAD_ERROR_BADLAYER;
      return -1;
    }
    
    /* protection_bit */
    if (mad_bit_read(&pStream->ptr, 1) == 0) {
      pHeader->flags    |= MAD_FLAG_PROTECTION;
      pHeader->crc_check = mad_bit_crc(& pStream->ptr, 16, 0xffff);
    }
    
    /* bitrate_index */
    index = mad_bit_read(&pStream->ptr, 4);
    
    if (index == 15) {
      pStream->error = MAD_ERROR_BADBITRATE;
      return -1;
    }
    
    if (pHeader->flags & MAD_FLAG_LSF_EXT)
      pHeader->bitrate = bitrate_table[3 + (pHeader->layer >> 1)][index];
    else
      pHeader->bitrate = bitrate_table[pHeader->layer - 1][index];
    
    /* sampling_frequency */
    index = mad_bit_read(&pStream->ptr, 2);
    
    if (index == 3) {
      pStream->error = MAD_ERROR_BADSAMPLERATE;
      return -1;
    }
    
    pHeader->samplerate = samplerate_table[index];
    
    if (pHeader->flags & MAD_FLAG_LSF_EXT) {
      pHeader->samplerate /= 2;
      
      if (pHeader->flags & MAD_FLAG_MPEG_2_5_EXT)
        pHeader->samplerate /= 2;
    }
    
    /* padding_bit */
    if (mad_bit_read(&pStream->ptr, 1))
      pHeader->flags |= MAD_FLAG_PADDING;
    
    /* private_bit */
    if (mad_bit_read(&pStream->ptr, 1))
      pHeader->private_bits |= MAD_PRIVATE_HEADER;
    
    /* mode */
    pHeader->mode = (enum mad_mode)(3 - mad_bit_read(&pStream->ptr, 2));
    
    /* mode_extension */
    pHeader->mode_extension = mad_bit_read(&pStream->ptr, 2);
    
    /* copyright */
    if (mad_bit_read(&pStream->ptr, 1))
      pHeader->flags |= MAD_FLAG_COPYRIGHT;
    
    /* original/copy */
    if (mad_bit_read(&pStream->ptr, 1))
      pHeader->flags |= MAD_FLAG_ORIGINAL;
    
    /* emphasis */
    pHeader->emphasis = (enum mad_emphasis)(mad_bit_read(&pStream->ptr, 2));
    
# if defined(OPT_STRICT)
    /*
     * ISO/IEC 11172-3 says this is a reserved emphasis value, but
     * streams exist which use it anyway. Since the value is not important
     * to the decoder proper, we allow it unless OPT_STRICT is defined.
     */
    if (pHeader->emphasis == MAD_EMPHASIS_RESERVED) {
      pStream->error = MAD_ERROR_BADEMPHASIS;
      return -1;
    }
# endif
    
    /* error_check() */
    
    /* crc_check */
    if (pHeader->flags & MAD_FLAG_PROTECTION)
      pHeader->crc_target = mad_bit_read(&pStream->ptr, 16);
    
    return 0;
  }
  
  /*
   * NAME:	free_bitrate()
   * DESCRIPTION:	attempt to discover the bitstream's free bitrate
   */
  int free_bitrate(struct mad_stream *pStream, struct mad_header const *pHeader)
  {
    struct mad_bitptr keep_ptr;
    uln rate = 0;
    unsigned int pad_slot, slots_per_frame;
    unsigned char const *ptr = 0;
    
    //keep_ptr = pStream->ptr;
    mymemcpy(&keep_ptr, & pStream->ptr, sizeof(struct mad_bitptr));
    //keep_ptr.byte = pStream->ptr.byte;
    //keep_ptr.cache = pStream->ptr.cache;
    //keep_ptr.left = pStream->ptr.left;
    
    pad_slot = (pHeader->flags & MAD_FLAG_PADDING) ? 1 : 0;
    slots_per_frame = (pHeader->layer == MAD_LAYER_III &&
                       (pHeader->flags & MAD_FLAG_LSF_EXT)) ? 72 : 144;
    
    while (mad_stream_sync(pStream) == 0) {
      struct mad_stream peek_stream;
      struct mad_header peek_header;
      
      //peek_stream = *stream;
      mymemcpy(&peek_stream, pStream, sizeof(struct mad_stream));
      //peek_header = *header;
      mymemcpy(&peek_header, pHeader, sizeof(struct mad_stream));
      
      if (decode_header(&peek_header, &peek_stream) == 0 &&
          peek_header.layer == pHeader->layer &&
          peek_header.samplerate == pHeader->samplerate) {
        unsigned int N;
        
        ptr = mad_bit_nextbyte(&pStream->ptr);
        
        N = ptr - pStream->this_frame;
        
        if (pHeader->layer == MAD_LAYER_I) {
          rate = (uln) pHeader->samplerate *
            (N - 4 * pad_slot + 4) / 48 / 1000;
        }
        else {
          rate = (uln) pHeader->samplerate *
            (N - pad_slot + 1) / slots_per_frame / 1000;
        }
        
        if (rate >= 8)
          break;
      }
      
      mad_bit_skip(&pStream->ptr, 8);
    }
    
    //  pStream->ptr = keep_ptr;
    mymemcpy(& pStream->ptr, &keep_ptr, sizeof(struct mad_bitptr));
    
    if (rate < 8 || (pHeader->layer == MAD_LAYER_III && rate > 640)) {
      pStream->error = MAD_ERROR_LOSTSYNC;
      return -1;
    }
    
    pStream->freerate = rate * 1000;
    
    return 0;
  }    

  
  void main(void)
  {  
    register unsigned char const *ptr, *end;
    unsigned int pad_slot, N;

    more_input = -1;
    
    ptr = stream.next_frame;
    end = stream.bufend;
    
    if (ptr == 0) {
      stream.error = MAD_ERROR_BUFPTR;
      stream.sync = 0;
      return;
    }
    
    sync:
    /* synchronize */
    if (stream.sync) {
      if (end - ptr < MAD_BUFFER_GUARD) {
        stream.next_frame = ptr;
        
        stream.error = MAD_ERROR_BUFLEN;        
        more_input = MAD_BUFFER_GUARD;
        return;
      }
      else if (!(ptr[0] == 0xff && (ptr[1] & 0xe0) == 0xe0)) {
        /* mark point where frame sync word was expected */
        stream.this_frame = ptr;
        stream.next_frame = ptr + 1;
        
        stream.error = MAD_ERROR_LOSTSYNC;
        stream.sync = 0;
        return;
      }
    }
    else {
      mad_bit_init(&stream.ptr, ptr);
      
      if (mad_stream_sync(&stream) == -1) {
        if (end - stream.next_frame >= MAD_BUFFER_GUARD)
          stream.next_frame = end - MAD_BUFFER_GUARD;
        
        stream.error = MAD_ERROR_BUFLEN;
        more_input = MAD_BUFFER_GUARD;
        return;
      }
      
      ptr = mad_bit_nextbyte(&stream.ptr);
    }
    
    /* begin processing */
    stream.this_frame = ptr;
    stream.next_frame = ptr + 1;  /* possibly bogus sync word */
    
    mad_bit_init(&stream.ptr, stream.this_frame);
    
    if (decode_header(&header, &stream) == -1) {
      stream.sync = 0;
      return;
    }
    
    /* calculate frame duration */
    mad_timer_set(&header.duration, 0,
                  32 * MAD_NSBSAMPLES(header), header.samplerate);
    
    /* calculate free bit rate */
    if (header.bitrate == 0) {
      if ((stream.freerate == 0 || !stream.sync ||
           (header.layer == MAD_LAYER_III && stream.freerate > 640000)) &&
          free_bitrate(&stream, &header) == -1) {
        stream.sync = 0;
        return;
      }
      
      header.bitrate = stream.freerate;
      header.flags  |= MAD_FLAG_FREEFORMAT;
    }
    
    /* calculate beginning of next frame */
    pad_slot = (header.flags & MAD_FLAG_PADDING) ? 1 : 0;
    
    if (header.layer == MAD_LAYER_I)
      N = ((12 * header.bitrate / header.samplerate) + pad_slot) * 4;
    else {
      unsigned int slots_per_frame;
      
      slots_per_frame = (header.layer == MAD_LAYER_III &&
                         (header.flags & MAD_FLAG_LSF_EXT)) ? 72 : 144;
      
      N = (slots_per_frame * header.bitrate / header.samplerate) + pad_slot;
    }
    
    stream.next_frame = stream.this_frame + N;
    
    header.flags |= MAD_FLAG_INCOMPLETE;  /* header is decoded */
    
    /* verify there is enough data left in buffer to decode this frame */
    if (N + MAD_BUFFER_GUARD > end - stream.this_frame) {
      stream.error = MAD_ERROR_BUFLEN;
      more_input = MAD_BUFFER_GUARD;
      return;
    }
  }
};
    
