/*
 * libmad - MPEG audio decoder library 
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: decode3.sc,v 1.3 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: decode3.sc,v $
 * Revision 1.3  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

# include "global.sh"

# include "fixed.sh"
# include "bit.sh"
# include "layer3.sh"

import "bit";

import "scalefac";
import "huffman";
import "stereo";
import "aliasreduc";
import "imdct";


/* --- Layer III Decode --------------------------------------------------- */


behavior III_decode_getchannel(struct mad_bitptr bitptr,
                               in unsigned int gr,
                               in unsigned int ch,
                               struct mad_header header,
                               in unsigned int sfreqi,
                               in unsigned char si_scfsi[2],
                               struct sideinfo_channel si_channel[2],
                               unsigned char scalefacs[2][2][39],
                               unsigned char sfbwidth[39],
                               out mad_fixed_t xr[576],
                               out enum mad_error error)
{
  unsigned int part2_length;
  
  Scalefactors scalefac(bitptr, header, gr, ch, sfreqi, si_scfsi, si_channel,
                        scalefacs, sfbwidth, part2_length);
                            
  Huffman huff(bitptr, gr, ch, si_channel, scalefacs,
               sfbwidth, part2_length, xr, error);
                            
  void main(void)
  {
    scalefac.main();
    huff.main();
  }
};


behavior III_decode_getdata(struct mad_bitptr bitptr,
                            in unsigned int gr,
                            in unsigned int nch,
                            struct mad_header header,
                            in unsigned int sfreqi,
                            in unsigned char si_scfsi[2],
                            struct sideinfo_channel si_channel0[2],
                            struct sideinfo_channel si_channel1[2],
                            unsigned char scalefacs[2][2][39],
                            unsigned char sfbwidth0[39],
                            unsigned char sfbwidth1[39],
                            out mad_fixed_t xr0[576],
                            out mad_fixed_t xr1[576],
                            enum mad_error error)
{
  III_decode_getchannel left(bitptr, gr, 0u, header, sfreqi, si_scfsi,
                             si_channel0, scalefacs,
                             sfbwidth0, xr0, error);
  III_decode_getchannel right(bitptr, gr, 1u, header, sfreqi, si_scfsi,
                              si_channel1, scalefacs,
                              sfbwidth1, xr1, error);

  void main(void)
  {
    fsm {
      left: {
        if (error) break;
        if (nch == 1) break;
      }
      
      right: {
        if (error) break;
      }
    }    
  }  
};

  

behavior III_decode_channel(in unsigned int gr,
                            in struct sideinfo_channel si_channel[2],
                            mad_fixed_t xr[576],
                            in unsigned char sfbwidth[39],
                            mad_fixed_t frame_overlap[32][18],
                            out mad_fixed_t frame_sbsample[36][32])
{
  Aliasreduce aliasreduce(gr, si_channel, sfbwidth, xr);
  Imdct imdct(si_channel, gr, xr, frame_overlap, frame_sbsample);
  
  void main(void)
  {
    aliasreduce.main();
    imdct.main();
  }
};


behavior III_decode_channels(in unsigned int gr,
                             in struct sideinfo_channel si_channel0[2],
                             in unsigned char sfbwidth0[39],
                             mad_fixed_t xr0[576],
                             mad_fixed_t frame_overlap0[32][18],
                             out mad_fixed_t frame_sbsample0[36][32],
                             in struct sideinfo_channel si_channel1[2],
                             in unsigned char sfbwidth1[39],
                             mad_fixed_t xr1[576],
                             mad_fixed_t frame_overlap1[32][18],
                             out mad_fixed_t frame_sbsample1[36][32])
{                             
  III_decode_channel left(gr, si_channel0, xr0, sfbwidth0, 
                          frame_overlap0, frame_sbsample0);
  III_decode_channel right(gr, si_channel1, xr1, sfbwidth1,
                           frame_overlap1, frame_sbsample1);
  
  void main(void)
  {
    par {
      left.main();
      right.main();
    }
  }
};


  
behavior III_decode_granule(struct mad_bitptr bitptr,
                            in unsigned int gr,
                            in unsigned int nch,
                            struct mad_header header,
                            in unsigned int sfreqi,
                            in unsigned char si_scfsi[2],
                            struct sideinfo_channel si_channel0[2],
                            struct sideinfo_channel si_channel1[2],
                            unsigned char scalefacs[2][2][39],
                            mad_fixed_t frame_overlap0[32][18],
                            out mad_fixed_t frame_sbsample0[36][32],
                            mad_fixed_t frame_overlap1[32][18],
                            out mad_fixed_t frame_sbsample1[36][32],
                            enum mad_error error)
{
  unsigned char sfbwidth0[39], sfbwidth1[39];
  mad_fixed_t xr0[576], xr1[576];
  
  /* scalefactors, Huffman decoding, requantization */
  III_decode_getdata getdata(bitptr, gr, nch, header, sfreqi, si_scfsi,
                             si_channel0, si_channel1, scalefacs,
                             sfbwidth0, sfbwidth1, xr0, xr1, error);

  /* joint stereo processing */
  Stereo stereo(gr, header, si_channel0, si_channel1, scalefacs,
                xr0, xr1, sfbwidth0, error);
  
  /* reordering, alias reduction, IMDCT, overlap-add, frequency inversion */
  III_decode_channels channels(gr, si_channel0, sfbwidth0, xr0, 
                               frame_overlap0, frame_sbsample0,
                               si_channel1, sfbwidth1, xr1,
                               frame_overlap1, frame_sbsample1);

  void main(void)
  {
    fsm {
      getdata: {
        if (error) break;
        if (header.mode == MAD_MODE_JOINT_STEREO && header.mode_extension)
          goto stereo;
        goto channels;
      }
      
      stereo: {
        if (error) break;
      }
      
      channels:
    }
  }
};

  
  
behavior III_decode_init(struct mad_header header,
                         out unsigned int ngr,
                         out unsigned int gr,
                         out unsigned int sfreqi)
{
  void main(void)
  {
    {
      unsigned int sfreq;
      
      sfreq = header.samplerate;
      if (header.flags & MAD_FLAG_MPEG_2_5_EXT)
        sfreq *= 2;
      
    /* 48000 => 0, 44100 => 1, 32000 => 2,
     24000 => 3, 22050 => 4, 16000 => 5 */
      sfreq = ((sfreq >>  7) & 0x000f) +
             ((sfreq >> 15) & 0x0001) - 8;

      if (header.flags & MAD_FLAG_MPEG_2_5_EXT)
        sfreq += 3;
      
      sfreqi = sfreq;
    }
    
    ngr = (header.flags & MAD_FLAG_LSF_EXT) ? 1 : 2;
    
    gr = 0;
  }
};
  
  
behavior III_decode(struct mad_bitptr bitptr,
                    in unsigned int nch,
                    struct mad_header header,
                    in unsigned char si_scfsi[2],
                    struct sideinfo_channel si_ch0[2],
                    struct sideinfo_channel si_ch1[2],
                    mad_fixed_t frame_overlap0[32][18],
                    out mad_fixed_t frame_sbsample0[36][32],
                    mad_fixed_t frame_overlap1[32][18],
                    out mad_fixed_t frame_sbsample1[36][32],
                    enum mad_error error)
{    
  unsigned int sfreqi, ngr, gr;

  unsigned char scalefacs[2][2][39];

  III_decode_init init(header, ngr, gr, sfreqi);
  
  III_decode_granule granule(bitptr, gr, nch, header, sfreqi, si_scfsi,
                              si_ch0, si_ch1, scalefacs,
                              frame_overlap0, frame_sbsample0,
                              frame_overlap1, frame_sbsample1, error);  
  
  void main(void)
  {
    fsm {
      init:
      
      granule: {
        if (error) break;                
        if (++gr >= ngr) break;
        goto granule;
      }
    }
  }  
};
