/*
 * libmad - MPEG audio decoder library
 * Copyright (C) 2006-2007 CECS, UC Irvine
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: aliasreduc.sc,v 1.4 2007/09/27 18:17:56 gerstl Exp $
 * 
 * $Log: aliasreduc.sc,v $
 * Revision 1.4  2007/09/27 18:17:56  gerstl
 * Public release of SpecC model (based on libmad MP3 library).
 *
 */

# include "global.sh"

import "global";

# include "fixed.sh"
# include "layer3.sh"


/* --- Aliasreduce--------------------------------------------------------- */

/*
 * coefficients for aliasing reduction
 * derived from Table B.9 of ISO/IEC 11172-3
 *
 *  c[]  = { -0.6, -0.535, -0.33, -0.185, -0.095, -0.041, -0.0142, -0.0037 }
 * cs[i] =    1 / sqrt(1 + c[i]^2)
 * ca[i] = c[i] / sqrt(1 + c[i]^2)
 */
static
mad_fixed_t const cs[8] = {
  +MAD_F(0x0db84a81) /* +0.857492926 */, +MAD_F(0x0e1b9d7f) /* +0.881741997 */,
    +MAD_F(0x0f31adcf) /* +0.949628649 */, +MAD_F(0x0fbba815) /* +0.983314592 */,
    +MAD_F(0x0feda417) /* +0.995517816 */, +MAD_F(0x0ffc8fc8) /* +0.999160558 */,
    +MAD_F(0x0fff964c) /* +0.999899195 */, +MAD_F(0x0ffff8d3) /* +0.999993155 */
};

static
mad_fixed_t const ca[8] = {
  -MAD_F(0x083b5fe7) /* -0.514495755 */, -MAD_F(0x078c36d2) /* -0.471731969 */,
    -MAD_F(0x05039814) /* -0.313377454 */, -MAD_F(0x02e91dd1) /* -0.181913200 */,
    -MAD_F(0x0183603a) /* -0.094574193 */, -MAD_F(0x00a7cb87) /* -0.040965583 */,
    -MAD_F(0x003a2847) /* -0.014198569 */, -MAD_F(0x000f27b4) /* -0.003699975 */
};

  

behavior Aliasreduce(in unsigned int gr,
                     in struct sideinfo_channel si_channel[2],
                     in unsigned char sfbwidth[39],
                     mad_fixed_t xr[576])
{
  /*
   * NAME:	III_reorder()
   * DESCRIPTION:	reorder frequency lines of a short block into subband order
   */
  void III_reorder(mad_fixed_t xr[576], unsigned char channel_flags,
                   unsigned char const sfbwidth[39])
  {
    mad_fixed_t tmp[32][3][6];
    unsigned int sb, l, f, w, sbw[3], sw[3];

    /* this is probably wrong for 8000 Hz mixed blocks */

    sb = 0;
    if (channel_flags & mixed_block_flag) {
      sb = 2;
      
      l = 0;
      while (l < 36)
        l += *sfbwidth++;
    }
    
    for (w = 0; w < 3; ++w) {
      sbw[w] = sb;
      sw[w]  = 0;
    }
    
    f = *sfbwidth++;
    w = 0;
    
    for (l = 18 * sb; l < 576; ++l) {
      if (f-- == 0) {
        f = *sfbwidth++ - 1;
        w = (w + 1) % 3;
      }

      tmp[sbw[w]][w][sw[w]++] = xr[l];
      
      if (sw[w] == 6) {
        sw[w] = 0;
        ++sbw[w];
      }
    }
    
    //PC:memcpy(&xr[18 * sb], &tmp[sb], (576 - 18 * sb) * sizeof(mad_fixed_t));
    mymemcpy(&xr[18 * sb], &tmp[sb], (576 - 18 * sb) * SIZEOF_INT);
  }

  /*
   * NAME:	III_aliasreduce()
   * DESCRIPTION:	perform frequency line alias reduction
   */
  void III_aliasreduce(mad_fixed_t *xr, int lines)
  {
    mad_fixed_t const *bound;
    int i;
    
    bound = &xr[lines];
    for (xr += 18; xr < bound; xr += 18) {
      for (i = 0; i < 8; ++i) {
        register mad_fixed_t a, b;
        register mad_fixed64hi_t hi;
        register mad_fixed64lo_t lo;
        
        a = xr[-1 - i];
        b = xr[     i];
        
# if defined(ASO_ZEROCHECK)
        if (a | b) {
# endif
          MAD_F_ML0(hi, lo,  a, cs[i]);
          MAD_F_MLA(hi, lo, -b, ca[i]);
          
          xr[-1 - i] = MAD_F_MLZ(hi, lo);
          
          MAD_F_ML0(hi, lo,  b, cs[i]);
          MAD_F_MLA(hi, lo,  a, ca[i]);

          xr[     i] = MAD_F_MLZ(hi, lo);
# if defined(ASO_ZEROCHECK)
        }
# endif
      }
    }
  }
  
    
  void main(void)
  {
      if (si_channel[gr].block_type == 2) {
	III_reorder(xr, si_channel[gr].flags, sfbwidth);

# if !defined(OPT_STRICT)
	/*
	 * According to ISO/IEC 11172-3, "Alias reduction is not applied for
	 * granules with block_type == 2 (short block)." However, other
	 * sources suggest alias reduction should indeed be performed on the
	 * lower two subbands of mixed blocks. Most other implementations do
	 * this, so by default we will too.
	 */
	if (si_channel[gr].flags & mixed_block_flag)
	  III_aliasreduce(xr, 36);
# endif
      }
      else
	III_aliasreduce(xr, 576);    
  }
};
  
