#include "systemc.h"

#define SZ_ROW          64
#define SZ_COL          128
#define NUM_ROW_BLOCKS  (SZ_ROW / 8)
#define NUM_COL_BLOCKS  (SZ_COL / 8)
#define CLK_CYCLE       10

const unsigned char QuantShiftTable[64] = 
                     { 0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 1, 1, 1, 1,
                       1, 1, 1, 1, 1, 1, 2, 2,
                       2, 2, 2, 2, 2, 2, 2, 3,
                       3, 3, 3, 3, 3, 3, 3, 4,
                       4, 4, 4, 4, 4, 4, 5, 5,
                       5, 5, 5, 5, 6, 6, 6, 6,
                       6, 7, 7, 7, 7, 8, 8, 8 };

SC_MODULE(cntrl) {
   // CNTRL Ports
   sc_fifo_in<bool>  doneCapture;
   sc_fifo_in<char>  ccdppPixelPop;
   sc_fifo_in<short> codecPixelPop;

   sc_fifo_out<bool> startCapture;
   sc_fifo_out<char> uartPixel;
   sc_fifo_out<short>   codecPixelPush;

   // CNTRL Vars
   short buffer[SZ_ROW][SZ_COL];
   short temp;
   int i, j, k, l;
   double	sim_time;

   void main( void )
   {
	  sim_time = sc_simulation_time();
      // CNTRL Capture
      startCapture.write( true );
      doneCapture.read();
      for(i=0; i<SZ_ROW; i++)
      {
         for(j=0; j<SZ_COL; j++)
         {
            buffer[i][j] = ccdppPixelPop.read();
			wait(2*CLK_CYCLE, SC_NS);
         }
      }

      // CNTRL Compress
      for(i=0; i<NUM_ROW_BLOCKS; i++) {

         for(j=0; j<NUM_COL_BLOCKS; j++) {

            // Push the block and perform FDCT
            for(k=0; k<8; k++)
            {
               for(l=0; l<8; l++)
               {
                  codecPixelPush.write( (char)buffer[i * 8 + k][j * 8 + l] );
				  wait(2*CLK_CYCLE, SC_NS);
               }
            }

            // Pop the block and store it in buffer
            for(k=0; k<8; k++)
            {
               for(l=0; l<8; l++)
               {
                  buffer[i * 8 + k][j * 8 + l] = codecPixelPop.read();
				  wait(2*CLK_CYCLE, SC_NS);
               }
            }
         }
      }

      // CNTRL Quantization 
      for(i=0; i<NUM_ROW_BLOCKS; i++)
      {
         for(j=0; j<NUM_COL_BLOCKS; j++)
         {
            // Quantize the block in place
            for(k=0; k<8; k++)
            {
               for(l=0; l<8; l++)
               {
                  buffer[i * 8 + k][j * 8 + l] >>= QuantShiftTable[k * 8 + l];
				  wait(4*CLK_CYCLE, SC_NS);
               }
            }
         }
      }

      // CNTRL Send Image
      for(i=0; i<SZ_ROW; i++) 
      {
         for(j=0; j<SZ_COL; j++) 
         {
            temp = buffer[i][j];

            uartPixel.write( ((char*)&temp)[0] );   // Send Upper Byte
			wait(2*CLK_CYCLE, SC_NS);
            uartPixel.write( ((char*)&temp)[1] );   // Send Lower Byte
			wait(2*CLK_CYCLE, SC_NS);
         }
      }
	  
	  cout << "CNTRL\tdone at " 
         << (sc_simulation_time()/1000) << " us\t"
         << "execution time = " 
         << (sc_simulation_time() - sim_time)/1000 << " us" << endl;
      // Need to stop the simulation manually now that we use a TIMED simulation
	  sc_stop();
      return;
   }


   // Module Constructor
   SC_CTOR( cntrl ) {
      SC_THREAD( main );
   }
};