function idx = subbandindexesn(scale, subband, varargin)
%SUBBANDINDEXESN - find indexes for specific subband and specific scale of
%  n-dimensional wavelet coefficients.  idx  = SUBBANDINDEXESN(scale,
%  subband, ...) returns the indexes for the requested N-dimensional subband of 
%  the N-dimensional signal at 'scale'.  subband identifies the subband and is 
%  a length(N) string of 'L' and 'H'.  For example, for a 2-D signal, the
%  course approximation subband is 'LL' and the three detail subbands are
%  'HL', 'LH' and 'HH' respectively.  Thus, the dimension of the signal, N, is 
%  determined by the number of entries in subband.
%
%  Note: If most coarse subband is selected and the scale is NOT the most
%        coarse it will actually contain detail subbands as well.
%
%  The optional parameter pair 'origsize', sz allows for the original size
%  of the data to be specified.  This is useful if the data was zero-padded
%  prior to the multiscale expansion.  Supplying the original size of the
%  data will return subband indexes that are proply sized (note that due to
%  non-integer rounding edge effects of upto 1 pixel may be present).
%
%  Optional Parameters
%  -------------------
%    'originalsize',sz - original size of the data.
%    'overcomplete',oc - transform was an overcomplete expansion
%
%  See Also: TRANSFORM, SUBBANDINDEXES
%
%Ian Atkinson - 2004

%==========================================================================
%== Revision History ======================================================
%==========================================================================
% Date     | Author | Description 
%--------------------------------------------------------------------------
% 06.18.04 | IA     | Original write
%--------------------------------------------------------------------------
% 04.07.06 | IA     | Fixed problem with indexes when original data is 
%          |        | smaller than wavelet data (floor vs ceil).
%--------------------------------------------------------------------------
% 05.12.06 | IA     | Updated to work for non-uniform dimensonal data with
%          |        | arbitrary decomposition levels in each dimension
%--------------------------------------------------------------------------


% parse input params
params = parse_inputs([], varargin{:});
params = validate_inputs(scale, params);

% the number of dimensions we need to get the indexes for
N = length(subband);

if(isscalar(scale))
    scale = repmat(scale, 1, N);
end

if(isscalar(params.originalsize))
    params.originalsize = repmat(params.originalsize, 1, N);
end

if(isscalar(params.originalScale))
    params.originalScale = repmat(params.originalScale, 1, N);
end

if(N>1)
    for n=1:N
        idx{n} = subbandindexesn(scale(n), subband(n), 'originalsize', params.originalsize(n), 'overcomplete', params.overcomplete, 'originalScale', params.originalScale(n));
    end    
else
        
    % the size of the subband at the requested scale    
    if(~params.overcomplete)
        sbSize = 2^scale;   % size of subband at this scale
        resSize = sbSize;   % size of residual (Low pass) at this scale
    else
        % for an overcomplete expansion there is no subsampling so all
        % subbands are the same size as the original signal
        sbSize  = 2^nextpow2(params.originalsize);
        resSize = scale*sbSize;
    end
    
    switch(upper(subband))
        case 'L'
            idx = 1:sbSize;
        case 'H'
            idx = (resSize+1):(resSize+sbSize);
        otherwise 
            error(sprintf('Unknown subband type %s', subband))
    end
           
    % For transforms that have been padded/extended in order to have an integer
    % power of two length we can determine (approximately) which
    % coefficients actually comprise the subband.  
    
    if(~params.overcomplete)
        K = ceil(params.originalsize/(2^(params.originalScale-scale)));
    else
        K = params.originalsize;
    end
    
    idx = idx(1:K);
    
end


%-------------------------------------------------------------------------------
% input checking routine
%-------------------------------------------------------------------------------
function params = validate_inputs(scale, params)

    params.error         = field_default(params, 'error', false);
    params.overcomplete  = field_default(params, 'overcomplete', false);
    params.originalsize  = field_default(params, 'originalsize', 2.^scale);
    params.originalScale = field_default(params, 'originalScale', ceil(log2(params.originalsize)));        
