function  dispactivationmap(map, type, varargin)
%DISPACTIVATIONMAP - displays an activation map
%  DISPACTIVATIONMAP(map[,type, ...]) will display the map as 
%  an activation map.
%
%  type can be any value for which map.type exists.  This is typically one
%  of the following (generated by ACTIVATIONMAP):
%    
%    'active'    - simple activation (binary) map
%    'threshold' - threshold applied to each voxel
%    'pvalue'    - type I error (false positive probability)
%
%  Optional parameters are specifed in two-ples of 'name', value.  The 
%  supported parameters and values are:
%
%    'additionalAlpha', alpha - additional alpha channel modifier.  alpha
%                               must be the same size as the map and have 
%                               values in [0,1].
%
%    'overlayRange',[mn mx]   - range of overlay values display.
%  
%
%    'logOverlay', v          - v can be true or false (default).  This 
%                               specifies that the overlay should be
%                               displayed in the log-domain.
%
%    'useAlpha',v             - v specifies the name of the alpha channel
%                               to use.  Valid values of v are the same as
%                               type.
%
%    'tileX',v                - number of images to tile in the X direction
%
%    'tileY',v                - number of images to tile in the Y direction
%
%    'trueActivation',v       - v is a matix with 1 where there is known
%                               activation.  Passing this option when the
%                               type is 'active' will color false positive
%                               and false negatives appropriately.
%
%  Note: additionalAlpha can be used to add an addition alpha mask to the
%  overlay.  This is useful to remove activations that result in overly
%  high error.  additionalAlpha should be the same dimensions as the
%  overlay (e.g., sizeof(map.active)) and have values in [0,1].
%
%  Note: if both tileX and tileY are specified, the value for tileX will be
%  used and tileY will be ignored.
%
%See Also: ACTIVATIONMAP
%
%Ian Atkinson - 2005

%==========================================================================
%== Revision History ======================================================
%==========================================================================
% Date     | Author | Description 
%--------------------------------------------------------------------------
% 01.12.05 | IA     | Original write 
%--------------------------------------------------------------------------
% 01.15.05 | IA     | Added additional alpha parameter 
%--------------------------------------------------------------------------
% 04.13.05 | IA     | Added overlayRange parameter 
%--------------------------------------------------------------------------
% 04.25.05 | IA     | changed default colormap to go from red->yellow
%--------------------------------------------------------------------------
% 10.13.05 | IA     | added tileX and tileY options
%--------------------------------------------------------------------------

if(~exist('type', 'var'))
    type = 'active';
end

params = parse_inputs([],varargin{:});
params = validate_params(type, map, params);   

type = lower(type);

% special type handling
%-- CNR ---
switch(type)
    case 'cnr'
        params.useAlpha = 'contrast';   % use the contrast alpha
    
        otype = 'contrast';
        utype = 'noise';
    case 'bnr'
        params.useAlpha = 'baseline'; % use the baseline alpha

        otype = 'baseline';
        utype = 'noise';

    otherwise
        % default the overlay to the requested type and no underlay
        otype = type;
        utype = false;
end


if(length(map)>1)
    
    % build a composite map
    tiles = length(map);
    if(params.tileX<=0 & params.tileY <=0)
        tilesX = ceil(sqrt(tiles));
        tilesY = ceil(tiles/tilesX);
    elseif(params.tileX<=0 & params.tileY>0)
        tilesY = params.tileY;
        tilesX = ceil(tiles/tilesY);    
    elseif(params.tileX>0)
        tilesX = params.tileX;
        tilesY = ceil(tiles/tilesX);
    end        
    
    tileSz = size(map(1).background);
    
    background      = zeros([tilesY*tileSz(1) tilesX*tileSz(2)]);
    overlay         = zeros([tilesY*tileSz(1) tilesX*tileSz(2)]);
    underlay        = zeros([tilesY*tileSz(1) tilesX*tileSz(2)]);
    alpha           = zeros([tilesY*tileSz(1) tilesX*tileSz(2)]);
    addAlpha        = zeros([tilesY*tileSz(1) tilesX*tileSz(2)]);
    trueActivation  = zeros([tilesY*tileSz(1) tilesX*tileSz(2)]);
    
    x=1;
    y=1;   
    for(z=1:length(map))
        xIndex = (x-1)*tileSz(2)+1:(x)*tileSz(2);
        yIndex = (y-1)*tileSz(1)+1:(y)*tileSz(1);
       
        if(isequal(x,tilesX))
            x=1;
            y=y+1;
        else
            x = x+1;
        end
        
        background(yIndex, xIndex) = map(z).background; 
    
        if(~isequal(params.trueActivation, 'not specified'))            
            trueActivation(yIndex, xIndex) = params.trueActivation(:,:,z);        
        end

        eval(sprintf('overlay(yIndex, xIndex) = map(z).%s;', otype));

        if(utype)
            eval(sprintf('underlay(yIndex, xIndex) = map(z).%s;', utype));
        end
        
        eval(sprintf('alphaValue   = map(z).alpha.%s;', params.useAlpha));        

        while(ischar(alphaValue))
            eval(sprintf('alphaValue = map(z).alpha.%s;', alphaValue));
        end

        alpha(yIndex, xIndex) = alphaValue;
                
        addAlpha(yIndex, xIndex) = params.additionalAlpha(:,:,z);
    end
    
    additionalAlpha = addAlpha;
    
else
    background = map.background;
    trueActivation = params.trueActivation;
    eval(sprintf('overlay = map.%s;', otype));
    if(utype)
        eval(sprintf('underlay = map.%s;', utype));
    end
    
    eval(sprintf('alpha   = map.alpha.%s;', params.useAlpha));
    
    while(ischar(alpha))
        eval(sprintf('alpha = map.alpha.%s;', alpha));
    end
    
    additionalAlpha = params.additionalAlpha;
end

% fix overlay to be scaled by underlay (if it was specified)
if(utype)
   idx = find(underlay~=0);
   overlay(idx) = overlay(idx)./underlay(idx);
end

% force contrast in the range [0 1]
background = double(background);
background = background - minall(background);
background = background./maxall(background);

% add in additional alpha
alpha = double(alpha).*double(additionalAlpha);

visible = find(alpha~=0);

mx = maxall(overlay(visible));
mn = minall(overlay(visible));

if(params.logOverlay)
    % non-zero indexes
    idx = find(overlay~=0);
    % zero-indexes that are to be shown
    idx0 = intersect(find(overlay==0), find(alpha>0));
    
    overlay(idx) = -log10(overlay(idx));
    
    if(~isempty(idx0))
        warning('Some overlay values have zero value, ajusting -log10 value to maximum');
        overlay(idx0) = maxall(overlay);
    end
end

% convert to full color
background = ind2rgb(ceil(100*background), gray(100));

% clear overlay values that will not be shown
overlay(find(alpha==0)) = min(overlay(:));

% if the display type is 'active' color the know false positives and false
% negatives
if(isequal(type, 'active') && ~isequal(params.trueActivation, 'not specified'))
   idxFP = intersect(find(overlay==1), find(trueActivation==0));
   idxFN = intersect(find(overlay==0), find(trueActivation==1));
   idxTP = intersect(find(overlay==1), find(trueActivation==1));
   idxTN = intersect(find(overlay==0), find(trueActivation==0));      
   
   % recolor the overlay
   overlay(idxTN) = 0;
   overlay(idxFP) = 1;
   overlay(idxFN) = 2;
   overlay(idxTP) = 3;
   
   % correct alpha to show FN (FP an TP are already visible)
   alpha(idxFN) = 1;
   
   % build a colormap for overlay -- use 4 colors
   cm(1,:) = [0 0  0];   % TN - value of 0
   cm(2,:) = [1 0  0];    % FP - value of 1
   cm(3,:) = [0 1  0];    % FN - value of 2
   cm(4,:) = [1 1  0];   % TP - value of 3     
   
   % use a forced overlay range
   params.overlayRange = [0 3];
   
   % display summary statistics
   disp(['  True positives: ' num2str(length(idxTP)) '  (' num2str(100*length(idxTP)/(length([idxTP; idxFN]))) '%)']);   
   disp([' False Negatives: ' num2str(length(idxFN)) '  (' num2str(100*length(idxFN)/(length([idxTP; idxFN]))) '%)']); 
   disp([' False positives: ' num2str(length(idxFP)) '  (' num2str(100*length(idxFP)/(length([idxTP; idxFP]))) '% of all positives)']);

else 

    % build colormap for overlay -- use a nice gradient
    start_color = [1 0 0];
    end_color   = [1 1 0];
    len = 100;
    cm = [start_color; repmat(start_color, len-1, 1) + cumsum(repmat(end_color-start_color, len-1, 1), 1)/(len-1)]; 
end

if(isequal(params.overlayRange, 'auto'))
    params.overlayRange = [min(overlay(:)), max([overlay(:); 1])];
end
% show background with maximum contrast but use a very small range since
% this will affect any color bar attached
imagesc(background, params.overlayRange);

hold on;

imagesc(overlay, 'AlphaData', alpha, params.overlayRange);
colormap(cm);

hold off; 

axis off;


if(length(map)>1)
   pbaspect([tilesX tilesY 1]); 
end

function params = validate_params(type, map, params)

    params.useAlpha                     = field_default(params, 'useAlpha', type);
    params.scaleOverlay                 = field_default(params, 'scaleOverlay', false);
    params.logOverlay                   = field_default(params, 'logOverlay', false);
    params.additionalAlpha              = field_default(params, 'additionalAlpha', ones([size(map(1).background), length(map)]));
    params.overlayRange                 = field_default(params, 'overlayRange', 'auto');
    params.tileX                        = field_default(params, 'tileX', -1);
    params.tileY                        = field_default(params, 'tileY', -1);    
    
    params.trueActivation               = field_default(params, 'trueActivation', 'not specified');
    

        