function  map=activationmap(voxel, type, sz, background, varargin)
%ACTIVATIONMAP - generates an activation map
%  ACTIVATIONMAP(voxel, type, sz, backgound, ...) will process columns of 
%  voxels as independent time courses and produce and activation map with 
%  dimensions of sz (size of the original dataset [y x z]).  The parameter 
%  type identifies the type of test used to determine activation and can be any
%  of the following:
%
%  'ttest'  - pooled variance t-test of active and rest voxels compared to 
%             a threshold
%  'signal' - estimated signal level compared to threshold
%  'count'  - set the largest X voxels active
%
%  The various tests take optional parmeters, which occur in 'name',value
%  pairs
%  'idxActive'        - index of active voxels (required for 'ttest' and 'signal')
%  'idxRest'          - index of rest voxels (required for 'ttest' and 'signal')
%  'threshold.mean'   - mean level (minimum) discriminator for test voxels
%  'threshold.var'    - variance (maximum) discriminator for test voxels
%  'threshold.t'      - t-statistic threshold (for each voxel) (required for 'ttest')
%  'threshold.signal' - signal-level threshold (for each voxel) (required for 'signal')
%  'testIndex'        - index of voxels to include in tests
%  'count'            - number of voxels to set active for count detection
%
%  The parameter background specifies the anatomy that is used for the
%  background.
%
%Ian Atkinson - 2005


%==========================================================================
%== Revision History ======================================================
%==========================================================================
% Date     | Author | Description 
%--------------------------------------------------------------------------
% 01.12.05 | IA     | Original write 
%--------------------------------------------------------------------------
% 01.16.05 | IA     | Modified to allow for various types to be easily 
%          |        | used/added
%--------------------------------------------------------------------------
% 03.23.05 | IA     | Added detrend option
%--------------------------------------------------------------------------
% 04.03.05 | IA     | Added mrf method
%--------------------------------------------------------------------------
% 01.05.06 | IA     | Added contrast, noise, and baseline measurements
%--------------------------------------------------------------------------
% 04.02.06 | IA     | Added fixed voxel count detection method
%--------------------------------------------------------------------------
% 06.02.09 | IA     | Added check for threshold length for t-test
%--------------------------------------------------------------------------

    if(length(sz)<3)
        sz = [sz 1];
    end

    params = parse_inputs([],varargin{:});
    params = validate_params(voxel, sz, params);
    params.background = background;

    switch(lower(type))
        case 'ttest'
            map = ttest_active(voxel, sz, params);
        case 'ttestfdr'
            map = ttest_fdr(voxel, sz, params);
        case 'signal'
            map = signal_active(voxel, sz, params);    
        case 'count'
            map = count_active(voxel, sz, params);
        case 'mrf'
            map = mrf_active(voxel, sz, params);
            
        case 'freq'
            map = freq_active(voxel, sz, params);
        
        otherwise
            error(sprintf('Unknown type ''%s''', type));
    end
    
    for(k=1:length(map))                
        % convert to simple types to save space        
        map(k).active = uint8(map(k).active);
        map(k).background = uint16(map(k).background);
        
        if(~ischar(map(k).alpha.active))    map(k).alpha.active     = uint8(map(k).alpha.active); end        
        if(~ischar(map(k).alpha.pvalue))    map(k).alpha.pvalue     = uint8(map(k).alpha.pvalue); end
        if(~ischar(map(k).alpha.tvalue))    map(k).alpha.tvalue     = uint8(map(k).alpha.tvalue); end
        if(~ischar(map(k).alpha.signal))    map(k).alpha.signal     = uint8(map(k).alpha.signal); end       
        if(~ischar(map(k).alpha.contrast))  map(k).alpha.contrast   = uint8(map(k).alpha.contrast); end        
        if(~ischar(map(k).alpha.noise))     map(k).alpha.noise      = uint8(map(k).alpha.noise); end  
        if(~ischar(map(k).alpha.baseline))  map(k).alpha.baseline   = uint8(map(k).alpha.baseline); end  
               
        if(isstruct(map(k).alpha.threshold))            
            if(~ischar(map(k).alpha.threshold.t))       map(k).alpha.threshold.t = uint8(map(k).alpha.threshold.t); end
            if(~ischar(map(k).alpha.threshold.signal))  map(k).alpha.threshold.signal = uint8(map(k).alpha.threshold.signal); end
        else
            if(~ischar(map(k).alpha.threshold)) map(k).alpha.threshold = uint8(map(k).alpha.threshold); end
        end        
    end
    
    % set extra field in map identifying the type
    [map.type] = deal(type);

function map = ttest_active(voxel, sz, params)

    
    % calculate t-statistics
    tvalue = zeros(1,size(voxel,2));
    
    [tvalue(params.testIndex) df] = tstatistic(voxel(params.idxActive,params.testIndex), voxel(params.idxRest,params.testIndex));
    
    % determine if active
    active = zeros(1,size(voxel,2));
    
    if(params.option.ttest.twosided)
        % two-sided t-test
        active(params.testIndex) = abs(tvalue(params.testIndex)) > params.threshold.t(params.testIndex);
    else
        % a one-sided t-test
        active(params.testIndex) = tvalue(params.testIndex) > params.threshold.t(params.testIndex);
    end
       
    % calculate the approximate amount of signal, contast, and noise
    signal = calcSignal(voxel, params);
    contrast = calcSignal(voxel, params, false);
    noise = calcNoise(voxel, params);
    baseline = params.mrest;
    
    sz = sz(:).';
 
    active      = reshape(active,               sz);
    tvalue      = reshape(tvalue,               sz);
    threshold   = reshape(params.threshold.t,   sz);
    signal      = reshape(signal,               sz);
    contrast    = reshape(contrast,             sz);
    noise       = reshape(noise,                sz);      
    baseline    = reshape(baseline,             sz);
    
    processed = zeros(1,size(voxel,2));
    processed(params.testIndex) = 1;  
    processed = reshape(processed, sz);

    % p-value overlay
    % note that tcdf(tval) gives the (approximate) probability of correctly
    % saying "no signal" for a value of tval.  Thus, 1-tcdf(tval) is the
    % false-positive probability.
    pvalue      = 1-tcdf(tvalue,df);


    for(z=1:size(params.background,3))

        map(z).background = params.background(:,:,z);

        map(z).active     = active(:,:,z);
        map(z).pvalue     = pvalue(:,:,z);
        map(z).signal     = signal(:,:,z);
        map(z).tvalue     = tvalue(:,:,z);
        map(z).threshold  = threshold(:,:,z);
        map(z).contrast   = contrast(:,:,z);
        map(z).noise      = noise(:,:,z);
        map(z).baseline   = baseline(:,:,z);
        
        map(z).alpha.processed = processed(:,:,z);
        map(z).alpha.active    = active(:,:,z);
        
        map(z).alpha.threshold  = 'processed';
        map(z).alpha.pvalue     = 'active';
        map(z).alpha.tvalue     = 'processed';
        map(z).alpha.signal     = 'active';
        map(z).alpha.contrast   = 'processed';
        map(z).alpha.noise      = 'processed';
        map(z).alpha.baseline   = 'processed';
    end

function map = ttest_fdr(voxel, sz, params)

    % calculate test statistics
    tvalue = zeros(1,size(voxel,2));
    
    [tvalue(params.testIndex) df] = tstatistic(voxel(params.idxActive,params.testIndex), voxel(params.idxRest,params.testIndex));
       
    % calculate the resulting p-values
    pvalue = 1-tcdf(tvalue, df);
    
    % sort pvalues (of test indexes)
    spvalue = sort(pvalue(params.testIndex));
    
    % find the cutoff p-value
    N = length(params.testIndex);
    index = find(spvalue < linspace(1/N,1,N)*params.fdr);
    pthresh = max([spvalue(max(index)),0]);
    
    % determine activity based on p-values    
    active = zeros(1,size(voxel,2));
    active(params.testIndex) = pvalue(params.testIndex) <= pthresh;   
    
    % calculate the approximate amount of signal
    signal = calcSignal(voxel, params);
    contrast = calcSignal(voxel, params, false);
    noise = calcNoise(voxel, params);
    baseline = params.mrest;
    
    sz = sz(:).';

    active      = reshape(active, sz);
    tvalue      = reshape(tvalue, sz);
    signal      = reshape(signal, sz);
    contrast    = reshape(contrast, sz);
    noise       = reshape(noise,    sz);       
    baseline    = reshape(baseline,    sz);       

    pvalue      = reshape(pvalue, sz);    
    threshold   = tinv(1-pthresh,df)*ones(sz);

    processed = zeros(1,size(voxel,2));
    processed(params.testIndex) = 1;  
    processed = reshape(processed, sz);
    
    for(z=1:size(params.background,3))

        map(z).background = params.background(:,:,z);

        map(z).active     = active(:,:,z);
        map(z).pvalue     = pvalue(:,:,z);
        map(z).signal     = signal(:,:,z);
        map(z).contrast   = contrast(:,:,z);
        map(z).noise      = noise(:,:,z);
        map(z).tvalue     = tvalue(:,:,z);
        map(z).threshold  = threshold(:,:,z);
        map(z).baseline   = baseline(:,:,z);
        
        map(z).alpha.processed = processed(:,:,z);
        map(z).alpha.active    = active(:,:,z);
        
        map(z).alpha.threshold  = 'processed';
        map(z).alpha.pvalue     = 'active';
        map(z).alpha.tvalue     = 'processed';
        map(z).alpha.signal     = 'active';
        map(z).alpha.contrast   = 'processed';
        map(z).alpha.noise      = 'processed';    
        map(z).alpha.baseline   = 'processed';
    end    
    
function map = ttest_fdr2(voxel, sz, params)

    % calculate test statistics
    tvalue = zeros(1,size(voxel,2));
    
    [tvalue(params.testIndex) df] = tstatistic(voxel(params.idxActive,params.testIndex), voxel(params.idxRest,params.testIndex));
       
    % calculate the approximate amount of signal
    signal = calcSignal(voxel, params);
    contrast = calcSignal(voxel, params, false);
    noise = calcNoise(voxel, params);
    baseline = params.mrest;
    
    processed = zeros(1,size(voxel,2));
    processed(params.testIndex) = 1;  
    processed = reshape(processed, sz);     
        
    for(z=1:sz(3))
        sidx = [((z-1)*prod(sz(1:2))+1):z*prod(sz(1:2))];
        idx = intersect(sidx,params.testIndex);
        lidx = idx - (z-1)*prod(sz(1:2)); 
        
        % calculate the resulting p-values
        pvalue = zeros(1,prod(sz(1:2)));
        pvalue(lidx) = 1-tcdf(tvalue(idx), df);
    
        % sort pvalues (of test indexes)
        spvalue = sort(pvalue(lidx));
    
        % find the cutoff p-value
        N = length(idx);
        index = find(spvalue < linspace(1/N,1,N)*params.fdr);
        pthresh = max([spvalue(max(index)),0]);
    
        % determine activity based on p-values    
        active = zeros(1,prod(sz(1:2)));
        active(lidx) = (pvalue(lidx) <= pthresh);   
           
        sz = sz(:).';

        active      = reshape(active, sz(1:2));
        tvaluez     = reshape(tvalue(sidx), sz(1:2));
        signalz     = reshape(signal(sidx), sz(1:2));
        contrastz   = reshape(contrast(sidx), sz(1:2));
        noisez      = reshape(noise(sidx), sz(1:2));
        baselinez   = reshape(baseline(sidx), sz(1:2));
        
        pvalue      = reshape(pvalue, sz(1:2));    
        threshold   = tinv(1-pthresh,df)*ones(sz(1:2));     

        map(z).background = params.background(:,:,z);

        map(z).active     = active;
        map(z).pvalue     = pvalue;
        map(z).signal     = signalz;
        map(z).contrast   = contrastz;
        map(z).noise      = noisez;        
        map(z).tvalue     = tvaluez;
        map(z).threshold  = threshold;
        map(z).baseline   = baselinez;
        
        map(z).alpha.processed = processed(:,:,z);
        map(z).alpha.active    = active;
        
        map(z).alpha.threshold  = 'processed';
        map(z).alpha.pvalue     = 'active';
        map(z).alpha.tvalue     = 'processed';
        map(z).alpha.signal     = 'active';
        map(z).alpha.contrast   = 'processed';
        map(z).alpha.noise      = 'processed';  
        map(z).alpha.baseline   = 'processed';
    end        
    
function map = signal_active(voxel, sz, params)   

    % calculate t-statistics
    tvalue = zeros(1,size(voxel,2));
    [tvalue(params.testIndex) df] = tstatistic(voxel(params.idxActive,params.testIndex), voxel(params.idxRest,params.testIndex));
       
    % calculate the resulting p-values (in percent for nicer overlay)
    pvalue = 1-tcdf(tvalue, df);
    
    % calculate the approximate amount of signal
    signal = calcSignal(voxel, params);
    contrast = calcSignal(voxel, params, false);
    noise = calcNoise(voxel, params);    
    
    % determine if active
    active = zeros(1,size(voxel,2));
    active(params.testIndex) = signal(params.testIndex) >= params.threshold.signal(params.testIndex);
    active(params.testIndex) = active(params.testIndex).*(tvalue(params.testIndex) >= params.threshold.t(params.testIndex));
    
    sz = sz(:).';

    active           = reshape(active,                  sz);
    pvalue           = reshape(pvalue, sz);   
    tvalue           = reshape(tvalue,                  sz);
    threshold.t      = reshape(params.threshold.t,      sz);
    threshold.signal = reshape(params.threshold.signal, sz);
    signal           = reshape(signal,                  sz);

    processed = zeros(1,size(voxel,2));
    processed(params.testIndex) = 1;  
    processed = reshape(processed, sz);

    for(z=1:size(params.background,3))

        map(z).background = params.background(:,:,z);

        map(z).active           = active(:,:,z);
        map(z).pvalue           = pvalue(:,:,z);
        map(z).signal           = signal(:,:,z);
        map(z).tvalue           = tvalue(:,:,z);
        map(z).threshold.t      = threshold.t(:,:,z);
        map(z).threshold.signal = threshold.signal(:,:,z);

        map(z).alpha.active             = active(:,:,z);
        map(z).alpha.pvalue             = active(:,:,z);
        map(z).alpha.threshold.t        = processed(:,:,z);
        map(z).alpha.threshold.signal   = processed(:,:,z);
        map(z).alpha.tvalue             = processed(:,:,z);
        map(z).alpha.signal             = active(:,:,z);
    end


function map = count_active(voxel, sz, params)

    
    % calculate t-statistics
    tvalue = zeros(1,size(voxel,2));
    
    [tvalue(params.testIndex) df] = tstatistic(voxel(params.idxActive,params.testIndex), voxel(params.idxRest,params.testIndex));
    
    % determine if active
    active = zeros(1,size(voxel,2));

    % for a scalar count we can set active voxels now
    if(isscalar(params.count))
        [dummy, idx] = sort(tvalue);
    
        active(idx(end-params.count:end)) = 1;
    end
          
    % calculate the approximate amount of signal, contast, and noise
    signal = calcSignal(voxel, params);
    contrast = calcSignal(voxel, params, false);
    noise = calcNoise(voxel, params);
    baseline = params.mrest;
    
    sz = sz(:).';
 
    active      = reshape(active,               sz);
    tvalue      = reshape(tvalue,               sz);
    threshold   = reshape(params.threshold.t,   sz);
    signal      = reshape(signal,               sz);
    contrast    = reshape(contrast,             sz);
    noise       = reshape(noise,                sz);      
    baseline    = reshape(baseline,             sz);
    
    processed = zeros(1,size(voxel,2));
    processed(params.testIndex) = 1;  
    processed = reshape(processed, sz);

    % p-value overlay
    % note that tcdf(tval) gives the (approximate) probability of correctly
    % saying "no signal" for a value of tval.  Thus, 1-tcdf(tval) is the
    % false-positive probability.
    pvalue      = 1-tcdf(tvalue,df);            

    for(z=1:size(params.background,3))

        % for a non-scalar count we will satisfy the requested number of
        % active voxels for each slice
        if(~isscalar(params.count))
            t = tvalue(:,:,z);                                    
            [dummy, idx] = sort(t(:));
            
            a = active(:,:,z);
            a(idx(end-params.count(z):end)) = 1;    
            active(:,:,z) = a;
        end               
        
        map(z).background = params.background(:,:,z);

        map(z).active     = active(:,:,z);
        map(z).pvalue     = pvalue(:,:,z);
        map(z).signal     = signal(:,:,z);
        map(z).tvalue     = tvalue(:,:,z);
        map(z).threshold  = threshold(:,:,z);
        map(z).contrast   = contrast(:,:,z);
        map(z).noise      = noise(:,:,z);
        map(z).baseline   = baseline(:,:,z);
        
        map(z).alpha.processed = processed(:,:,z);
        map(z).alpha.active    = active(:,:,z);
        
        map(z).alpha.threshold  = 'processed';
        map(z).alpha.pvalue     = 'active';
        map(z).alpha.tvalue     = 'processed';
        map(z).alpha.signal     = 'active';
        map(z).alpha.contrast   = 'processed';
        map(z).alpha.noise      = 'processed';
        map(z).alpha.baseline   = 'processed';
    end    
    
    
function map = mrf_active(voxel, sz, params)

    % Use fdr as a starting point...
    map = ttest_fdr(voxel, sz, params);
    
    % process each channel...
    for(z=1:sz(3))
    
        idx = [prod(sz(1:2))*(z-1)+1:prod(sz(1:2))*(z)];
        
        % estimate the variance from the rest state
        vrest = var(double(voxel(params.idxRest,idx)));
        
        % get current active state (estimated from fdr)
        afdr = map(z).active;
        afdr = afdr(:).';
        
        
        if(isequal(params.mrf.beta, 'estimate'))            
            as = sort(map(z).signal(find(map(z).active==1)));            
            if(length(as)>0)                
              %  beta = as(max(floor(length(as)*.15),1))/2;                        
              beta = maxall(as)/3;
            else
                beta = 0;
            end
            
            beta = max(beta, .001);
            
            disp(sprintf('z=%d, beta=%g', z, beta));
        else
            beta = params.mrf.beta(z);
        end
            
        % update the active voxels...                
        a = mrf_detect(double(voxel(:,idx)), afdr, vrest, params.hdr, params.mrest(idx), 'gamma', params.mrf.gamma, 'testIndex', intersect(params.testIndex, idx)-(z-1)*prod(sz(1:2)), 'background', params.background(:,:,z), 'size', sz(1:2), 'beta', beta);
    
        % store back the active value
        map(z).active = reshape(a,sz(1:2));
        
        % update alpha channels that uses active
        map(z).alpha.active             = map(z).active;
        map(z).alpha.signal             = map(z).active;
        map(z).alpha.pvalue             = map(z).active;
    end

    
function map = freq_active(voxel, sz, params)

    % calculate the mean of each voxel
    mvox = mean(voxel, 1);

    params.testIndex = find(mvox>params.threshold.mean);

    % compute FFT of 
    Vox = fft(double(voxel(:,params.testIndex))-repmat(mean(double(voxel(:,params.testIndex)),1), size(voxel,1), 1), params.fftsize, 1);
 
    Vvar = var(Vox(end/4:end/2-1,:), 0, 1);
    
    % determine if active
    active = zeros(1,size(voxel,2));
   
    R = abs(Vox(10,:)).^2./sqrt(Vvar);
    Rs = sort(R);
    
    N = length(params.testIndex);
    index = find(Rs < linspace(1/N,1,N)*.10);
    Rthresh = Rs(max(index));
    
    active(params.testIndex) = abs(Vox(10,:)).^2 > 3*sqrt(Vvar);
    sz = sz(:).';

    active      = reshape(active,       sz);
    processed   = reshape(mvox>params.threshold.mean, sz);

    for(z=1:size(params.background,3))

        map(z).background = params.background(:,:,z);

        map(z).active     = active(:,:,z);

        map(z).alpha.active     = active(:,:,z);
    end
     
function signal = calcSignal(voxel, params, normalize)

    if(nargin<3) normalize = true; end        

    % calculate the approximate amount of signal
    mActive = mean(voxel(params.idxActive, params.testIndex),1);
    mRest   = mean(voxel(params.idxRest,   params.testIndex),1);

    signal = zeros(1,size(voxel,2));
    signal(params.testIndex) = (mActive-mRest);
    
    if(normalize)
        signal(params.testIndex) = signal(params.testIndex)./params.mrest(params.testIndex);
    end
        
function noise = calcNoise(voxel, params)
      
    % calculate the noise level as the std of the rest periods
    noise = zeros(1,size(voxel,2));   
    noise(params.testIndex)   = std(double(voxel(params.idxRest,   params.testIndex)),[],1);

function params = validate_params(voxel, sz, params)

    params.threshold.mean               = field_default(params, 'threshold.mean',   -inf);
    params.threshold.t                  = field_default(params, 'threshold.t',      1.65*ones(1,size(voxel,2)));
    params.threshold.signal             = field_default(params, 'threshold.signal', .02*ones(1,size(voxel,2)));
    
    % maks sure threshold is long enough
    if length(params.threshold.t < prod(sz))
       params.threshold.t(1:prod(sz)) =params.threshold.t(1); 
    end
    
    params.count                        = field_default(params, 'count', ceil(size(voxel,2)*.01));
    
    params.testIndex                    = field_default(params, 'testIndex', 1:size(voxel,2));
    
    params.fftsize                      = field_default(params, 'fftsize', size(voxel,1));
    params.freq.activation              = field_default(params, 'freq.activation', 1/60);
    
    params.fdr                          = field_default(params, 'fdr', .15);  
    
    params.mrest                        = field_default(params, 'mrest', mean(voxel(params.idxRest,:),1));
        
    params.option.ttest.twosided        = field_default(params, 'option.ttest.twosided', false);
        
    % create a default hdr
    hdr = zeros(size(voxel,1),1);
    hdr(params.idxActive) = 1;
    hdr(params.idxRest) = -1;
    params.hdr                          = field_default(params, 'hdr', hdr);
    
    % mrf parameters
    params.mrf.gamma                    = field_default(params, 'mrf.gamma', 2*ones(1, sz(3)));
    params.mrf.beta                     = field_default(params, 'mrf.beta',  .01*ones(1,sz(3)));
    