/**************************************************************
 * Utility functions used for handling packet transmission and reception
 *
 * @author: Hongwei zhang
 *                    hzhang@motorola.com, zhangho@cse.ohio-state.edu
 *
 * @modified: August 1, 2005
 *
 **************************************************************/

/**
 * Handle packet reception at the base station
 *
 */
#include "APS.h"

void receive(message_t *pmsg, void* payload, uint16_t payloadLen) {   //void * payload
  TOS_APSPayload * thisPayload;
  TOS_APSPE * thisPE;
  uint8_t PEOverallLen, i;
  uint8_t * PEbase;
  
  atomic {
    thisPayload = (TOS_APSPayload *)payload;//(&(pmsg->data[ALR_OVERHEAD]));
    PEOverallLen = thisPayload->length;
    PEbase = &thisPayload->data[0];
  }

  //sanity check
  if (PEOverallLen > MAX_APS_PE_LEN) {
    dbg("DBG_ERROR", "error: corrupted thisPayload->length, = %d\n", PEOverallLen);
    return;
  }

  //iterate over all payload elements: signal the reception
  i = 0;
  while (i < PEOverallLen) {
    thisPE = (TOS_APSPE *)(&PEbase[i]);
    if (thisPE->length > MAX_APS_PE_LEN - PE_OVERHEAD || thisPE->length < MIN_PAYLOAD_ELEMENT_LEN - PE_OVERHEAD) {
      dbg("DBG_ERROR", "error: corrupted thisPE->length, = %d\n", thisPE->length);
      break;
    }
    else 
      signal Receive.receive[thisPE->type](pmsg, &(thisPE->data[0]), thisPE->length);

    i += (thisPE->length + PE_OVERHEAD);
    //dbg(DBG_TEMP, "receive packet: PEOverallLen = %d,   lenSignaled = %d \n", PEOverallLen, i);
  }

} //end of receive()



/*
 * Decrease the allowable latency for all the PEs in a packet
 *
 * Used in: forward()
 *
 * @param PEbase Pointer to the starting address of PE
 * @param PEOverallLen Total length of all PEs
 * @param duration Amount of decrease
 *
 * @return Whether "length" field has been corrupted
 */
bool decreasePELatency(uint8_t * PEbase, uint8_t PEOverallLen, uint16_t duration) {
  uint8_t i;
  TOS_APSPE * thisPE;

  i = 0;
  while (i < PEOverallLen) {
    thisPE = (TOS_APSPE *)(&PEbase[i]);
    if (thisPE->length > MAX_APS_PE_LEN - PE_OVERHEAD || thisPE->length < MIN_PAYLOAD_ELEMENT_LEN - PE_OVERHEAD) {
      dbg("DBG_ERROR", "error: corrupted packet. thisPE->length = %d\n", thisPE->length);
      return FAIL;
    }
    /*
    if (thisPE->maxLatency > MAX_PKT_INTERVAL) //simulate non-intelligent packing
      return SUCCESS;
      */
    if (thisPE->maxLatency > duration)
      thisPE->maxLatency -= duration;
    else 
      thisPE->maxLatency = 0;

    i += (thisPE->length + PE_OVERHEAD);
  }

  return SUCCESS;
} //end of decreasePELatency()



/*
 * Increase the allowable latency for all the PEs in a packet
 *
 * Used in: forward()
 *
 * @param PEbase Pointer to the starting address of PE
 * @param PEOverallLen Total length of all PEs
 * @param duration Amount of increase
 *
 * @return Whether "length" field has been corrupted
 */
bool increasePELatency(uint8_t * PEbase, uint8_t PEOverallLen, uint16_t duration) {
  uint8_t i;
  TOS_APSPE * thisPE;

  i = 0;
  while (i < PEOverallLen) {
    thisPE = (TOS_APSPE *)(&PEbase[i]);
    if (thisPE->length > MAX_APS_PE_LEN - PE_OVERHEAD || thisPE->length < MIN_PAYLOAD_ELEMENT_LEN - PE_OVERHEAD) {
      dbg("DBG_ERROR", "error: corrupted packet. thisPE->length = %d\n", thisPE->length);
      return FAIL;
    }
    /*
    if (thisPE->maxLatency > MAX_PKT_INTERVAL) //simulate non-intelligent packing
      return SUCCESS;
    */
    thisPE->maxLatency += duration;

    i += (thisPE->length + PE_OVERHEAD);
  }

  return SUCCESS;
} //end of increasePELatency()



/**
 * Forward a received packet at a non-base node
 *
 * Used in: Intercept.intercept() of APSM.nc
 *
 * @param pmsg The complete buffer received.
 *
 * @param payload The payload portion of the packet for APS layere
 *
 * @param payloadLen The length of the payload buffer. 
 *
 */
void forward(message_t *pmsg, void* payload, uint16_t payloadLen) {
  TOS_APSPayload * thisPayload;
  TOS_APSPE * thisPE;
  uint8_t PEOverallLen, i;
  uint8_t * PEbase;
  uint32_t curTime, thisInterval;
	am_addr_t parent; 
#ifdef SCHEDULE_PACKING
  uint32_t ett0;
#endif

  atomic {
    thisPayload = (TOS_APSPayload *)(&(pmsg->data[ALR_OVERHEAD]));
    PEOverallLen = thisPayload->length;
    PEbase = &thisPayload->data[0];
  }

  //sanity check
  if (PEOverallLen > MAX_APS_PE_LEN) {
    dbg("DBG_ERROR", "error: corrupted thisPayload->length, = %d\n", PEOverallLen);
    return;
  }
	
  /* Forward immediately, if the radio is ready to transmit, has a parent, and 
   * if the queue is full or the packet is already fully packed */
  if (!pending &&
#ifndef NO_PACKING_AT_ALL
      (numBusyBuffers >= QUEUE_SIZE || 
       PEOverallLen + MIN_PAYLOAD_ELEMENT_LEN > MAX_APS_PAYLOAD_LEN - PAYLOAD_OVERHEAD
       ) &&
#endif
	  (call CtpInfo.getParent(&parent)) != FAIL
     ) {
    dbg("DBG_USR1", "warning: forward directly. numBusyBuffers = %d,   PEOverallLen =%d\n", numBusyBuffers, PEOverallLen);
#ifdef SCHEDULE_PACKING
    //decrease max. allowable latency for each PE
    ett0 = call CtpInfo.getETT0();
	 if(ett0==0)
		  call Leds.led1Toggle();
    if (ett0 >= TIME_INVALID)
      goto toBuffer;
    if (!decreasePELatency(PEbase, PEOverallLen, ett0))
      return;
#endif
    //set control variables
    atomic {
      pending = TRUE;
      directlyForwarded = TRUE;
    }
    //try to send it
    if ((call Send.send(pmsg, payloadLen))!=SUCCESS) {  //failure
#ifdef SCHEDULE_PACKING
      increasePktMaxLatency(busyQHead, ett0);
#endif
      //miscl.
      atomic {
	pending = FALSE;
	directlyForwarded = FALSE;
      }
      dbg("DBG_ERROR", "failed to send packet to ALR layer\n");

    }
    else{ //success
      //for parameter estimation: transmission 
#ifdef TOSSIM
      curTime = (call SysTime.get()); // TICKS_PER_MILLISECOND;
#else
      curTime = (call LocalTime.get()); // TICKS_PER_MILLISECOND;
#endif
      //EWMA style estimation
      atomic {
	if (lastSendTime ==0) {//the first time to receive an packet to process
	  lastSendTime = curTime;
	  expSendSize = Q[busyQHead].length;
	}
	else {                                //otherwise
	  //interval
	  thisInterval = curTime - lastSendTime;
	  if (thisInterval < maxPktInterval) {//"reasonable"
	    if (!passedSecondSend) {
	      passedSecondSend = TRUE;
	      expSendInterval = thisInterval;
	    }
	    else
	      expSendInterval = (expSendInterval - (expSendInterval >> compPastW)) + (thisInterval >> compPastW);	
	  }
	  lastSendTime = curTime;
	  //size
	  expSendSize = (expSendSize - (expSendSize >> compPastW)) + (Q[busyQHead].length >> compPastW);
	}
      }//atomic
      //reflect to ALR layer
      call CtpInfo.setAPSParams(expSendInterval, expSendSize);

      return;
    } //success
  } //forward immediatly
   
  /* Otherwise, iterate over all payload elements: try to put each of them into packet queue */
 toBuffer:
  i = 0;
  while (i < PEOverallLen) {
    thisPE = (TOS_APSPE *)(&PEbase[i]);
    if (thisPE->length > MAX_APS_PE_LEN - PE_OVERHEAD || 
        thisPE->length < MIN_PAYLOAD_ELEMENT_LEN - PE_OVERHEAD
       ) {
      dbg("DBG_ERROR", "error: corrupted packet. thisPE->length = %d\n", thisPE->length);
      break;
    }
    else if (enqueue(thisPE->type, &(thisPE->data[0]), thisPE->length, thisPE->maxLatency, thisPE->SLatency) != SUCCESS) {
		//call Leds.led1Toggle();
      dbg("DBG_ERROR", "cannot forward PE: thisPE->length = %d,  numBusyBuffers =%d, freeQHead =%d \n", thisPE->length, numBusyBuffers, freeQHead);
    }

    i += (thisPE->length + PE_OVERHEAD);
  }

}  //end of forward()


/**
 * Decide whether it is beneficial to hold a packet
 *
 * Used in: Timer.fired()
 *
 * @param t_f Flextime of the head packet
 * @param s_f Extra spare space in the packet
 */

bool isBeneficialToHold(uint16_t t_f, uint8_t s_f) {
	uint16_t t_p_prime, t_p;
	uint8_t s_p;
	uint16_t t_l = expRcvInterval;
	uint8_t s_l = expRcvSize;
	uint16_t t_l_prime = expSendInterval;
	uint8_t s_l_prime = expSendSize;
	
	uint32_t etx_j, etx0, etx_parent;
	uint16_t etxptr;
	uint8_t etx_scale_factor = 8;
	
	uint32_t B_l, B_p_prime, B_p_doublePrime, C_p, B_p;
	uint32_t AC_l_prime, AC_l; //hongwei
	uint32_t AC_p_prime, AC_p; //hongwei
	
	uint8_t tmp, tmp1;
	dbg("DBG_ERROR", "calculating utility111111111111111!!!!!!!!!!!!!!!!!!!!!!!!\n");
	if(isFirstTime == TRUE)
	{
		isFirstTime = FALSE;
		return TRUE;
	}
	
	if (t_l >= maxPktInterval)
		return FALSE;
	else {
		call CtpInfo.getParentAPSParams(&t_p_prime, &s_p);
		if (t_p_prime >= maxPktInterval || s_p <= MIN_PKT_SIZE) 
			return TRUE;
	}
	
	/*Calculate the benefit B_l of holding the packet */
	dbg("DBG_ERROR", "calculating utility222222222222222222!!!!!!!!!!!!!!!!!!!!!!!!\n");
	if ((call CtpInfo.getEtx(&etxptr)) == FAIL)
		etx_j = ROUTE_INVALID;
	else 
		etx_j = (uint32_t)(etxptr << etx_scale_factor); 
	etx_parent = (uint32_t)(call CtpInfo.getParentETX());
	if (etx_parent != ROUTE_INVALID)
		etx_parent <<= etx_scale_factor; 
	etx0 = (uint32_t)(call CtpInfo.getETX0());
	if (etx0 != ROUTE_INVALID)
		etx0 <<= etx_scale_factor; 
	if (etx_j == ROUTE_INVALID || etx_parent == ROUTE_INVALID || etx0 == ROUTE_INVALID)
		return TRUE; //has to hold it
	else { //do the calculation
		//hongwei: if (t_f *s_l > (MAX_APS_PAYLOAD_LEN - s_f) * t_l)
		if (t_f *s_l > s_f * t_l)
			tmp = s_f; 
		else {
			if (t_l == 0)
				dbg("DBG_ERROR", "error: t_l == 0\n");
			tmp = t_f * s_l / (t_l == 0? 1 : t_l);
		}
		AC_l_prime = etx_j / (MAX_APS_PAYLOAD_LEN - s_f);
		AC_l = etx_j /(MAX_APS_PAYLOAD_LEN - s_f + tmp);
		B_l = AC_l_prime - AC_l;
	}
	
	/* Calculate the benefit B_p of transmitting the packet now */
	dbg("DBG_ERROR", "calculating utility33333333333333333333!!!!!!!!!!!!!!!!!!!!!!!!\n");
	//t_p
	if (t_l_prime * s_p <= t_p_prime * s_l_prime) //t_p will be < 0 or = infinity, invalid
		return TRUE;
	else
		t_p = (t_p_prime * t_l_prime * s_p) / (t_l_prime * s_p - t_p_prime * s_l_prime);
	if (t_p == 0 || s_p == 0) {
		dbg("DBG_ERROR", "error: t_p ==0 or s_p ==0 \n");
		return TRUE;
	}
	//B_p
	if (t_f * (MAX_APS_PAYLOAD_LEN - s_p) <= (MAX_APS_PAYLOAD_LEN - s_f) * t_p) {
		B_p_prime = ((MAX_APS_PAYLOAD_LEN - s_p) * etx_parent) / (s_p * MAX_APS_PAYLOAD_LEN);
		B_p = B_p_prime;  
	}
	else {
		AC_p_prime = etx_parent / s_p;
		tmp1 = (MAX_APS_PAYLOAD_LEN-s_f)/(MAX_APS_PAYLOAD_LEN - s_p);
		if (tmp1*(MAX_APS_PAYLOAD_LEN - s_p)<(MAX_APS_PAYLOAD_LEN-s_f))
			tmp1++;
		AC_p = (tmp1 * etx_parent) / (tmp1 * s_p + MAX_APS_PAYLOAD_LEN - s_f);
		B_p_doublePrime = AC_p_prime - AC_p; 
		B_p = B_p_doublePrime;
	}
	
	/* decide */
	 if (B_p > B_l){
                dbg("DBG_ERROR", "gad to hold!!!!!!!!!!!!!!!!!!!!!!!!\n");  
                return FALSE;
        }
        else{
                dbg("DBG_ERROR", "GOOD to hold!!!!!!!!!!!!!!!!!!!!!!!!!\n");
                return TRUE;
        }

} //end of isBeneficialToHold()





#ifdef TWOHOP


uint32_t calculateETX_l(uint8_t len, uint8_t Switch){//switch: 
                                       //1--getETX 2--getParentETX 3--getETX0 4--getGrandparentETX
    uint32_t unitETX;
    
    if(Switch == 1)                         //Swich to different function
    call CtpInfo.getEtx(&unitETX);
    else if (Switch ==2)
    unitETX = (call CtpInfo.getParentETX());
    else if (Switch ==3)
    unitETX = (call CtpInfo.getETX0());
    else if (Switch ==4)
    unitETX = (call CtpInfo.getGrandParentETX());
    
    if(unitETX == ROUTE_INVALID)
        return ROUTE_INVALID;
	else
		return unitETX;
}//calculateETX_l



/**
 * Decide whether it is beneficial to hold a packet based on two hop
 *
 * Used in: Timer.fired()
 *
 * @param t_f Flextime of the head packet
 * @param s_f Extra spare space in the packet
 */

bool isBeneficialToHold2Hop(uint16_t t_f,uint8_t s_f){
    uint16_t t_p_prime, t_p;
/***********************************************************/
    uint16_t t_g_prime, t_g;    //for 2-hop calculating
    uint8_t  s_g;
    uint32_t u_p_prime, u_p_dounble_prime, u_p_triple_prime,u_p;
    uint32_t u_g_prime, u_g_double_prime, u_g_triple_prime, u_g;
    uint32_t u_im;
    uint32_t etx_g_sg, etx_g_L;
    uint32_t g_mod, G, p_mod, P;
/***********************************************************/
	uint8_t s_p;
	uint16_t t_l = expRcvInterval;
	uint8_t s_l = expRcvSize;
	uint16_t t_l_prime = expSendInterval;
	uint8_t s_l_prime = expSendSize;
	
	uint32_t etx_j, etx_j_prime, etx0, etx_parent ;
	uint32_t etx_p_sp, etx_p_L, etx_p_sp_lmod;
	uint16_t l_mod;
	uint16_t etxptr;
	uint8_t etx_scale_factor = 8;
	
	uint32_t B_l, B_p_prime, B_p_doublePrime, C_p, B_p;
	uint32_t AC_l_prime, AC_l; //hongwei
	uint32_t AC_p_prime, AC_p; //hongwei
	
	uint32_t tmp, tmp1, tmp2, tmp3, tmp4,Switch;
	dbg("DBG_ERROR", "calculating utility for two hop111111111111111!!!!!!!!!!!!!!!!!!!!!!!!\n");
	if(isFirstTime == TRUE)
	{
		isFirstTime = FALSE;
		return TRUE;
	}
	
	if (t_l >= maxPktInterval)
		return FALSE;
	else {
		call CtpInfo.getParentAPSParams(&t_p_prime, &s_p);
		call CtpInfo.getGrandParentAPSParams(&t_g_prime, &s_g);
		//dbg(DBG_TEMP, "t_p_prime = %d,      s_p = %d\n", t_p_prime, s_p);
		if (t_p_prime >= maxPktInterval || s_p <= MIN_PKT_SIZE) 
			return TRUE;
	}
	
	/*Calculate the benefit B_l of holding the packet */
		if (t_f *s_l > s_f * t_l)
			tmp = s_f; 
		else {
			if (t_l == 0)
				dbg("DBG_ERROR", "error: t_l == 0\n");
			tmp = t_f * s_l / (t_l == 0? 1 : t_l);
		}
//#ifdef ETX_CAL
		tmp2 = MAX_APS_PAYLOAD_LEN - s_f;
		etx_j = calculateETX_l(tmp2,1);	
		tmp2 = MAX_APS_PAYLOAD_LEN - s_f + tmp;
		etx_j_prime =calculateETX_l(tmp2,1);
		if (etx_j == ROUTE_INVALID || etx_j_prime == ROUTE_INVALID)
		return TRUE; //has to hold it
//#endif		
		AC_l_prime = etx_j / (MAX_APS_PAYLOAD_LEN - s_f);
		AC_l = etx_j_prime /(MAX_APS_PAYLOAD_LEN - s_f + tmp);
		B_l = AC_l_prime - AC_l;
   //}
	
	
	
	/* Calculate the benefit u_p of transmitting for parent now */
	//dbg("DBG_ERROR", "calculating utility33333333333333333333!!!!!!!!!!!!!!!!!!!!!!!!\n");
	
//#ifdef ETX_CAL
		etx_p_sp =calculateETX_l(s_p , 2);
		etx_p_L = calculateETX_l(MAX_APS_PAYLOAD_LEN, 2);
		etx_g_sg = calculateETX_l(s_g,4);
		etx_g_L = calculateETX_l(MAX_APS_PAYLOAD_LEN, 4);
		if (etx_p_sp == ROUTE_INVALID || etx_p_L == ROUTE_INVALID||etx_g_sg == ROUTE_INVALID ||
		     etx_g_L == ROUTE_INVALID)
		return TRUE; //has to hold it
//#endif	
   if(s_p ==0||s_g ==0)
   {
      dbg("DBG_ERROR", "error: sp ==0 or sg ==0 \n");
		return TRUE;  
   }
   
   
   if (t_l_prime * s_p <= t_p_prime * s_l_prime) //t_p will be < 0 or = infinity, invalid
		return TRUE;
	else
		t_p = (t_p_prime * t_l_prime * s_p) / (t_l_prime * s_p - t_p_prime * s_l_prime);
	if (t_p == 0 || s_p == 0) {
		dbg("DBG_ERROR", "error: t_p ==0 or s_p ==0 \n");
		return TRUE;
	}
	
	
	if (t_l_prime * s_g <= t_g_prime * s_l_prime)
		return TRUE;
	else
		t_g = (t_g_prime * t_l_prime * s_g) / (t_l_prime * s_g - t_g_prime * s_l_prime);
	if (t_g == 0 || s_g == 0) {
		dbg("DBG_ERROR", "error: t_p ==0 or s_p ==0 \n");
		return TRUE;
	}
   /*************calculating u_p_prime***************/
   tmp = etx_p_sp/s_p;
   tmp1 = etx_g_sg/s_g;
   tmp2 = (etx_g_L+etx_p_L)/MAX_APS_PAYLOAD_LEN;
   if(tmp+tmp1<tmp2)
        u_p_prime = 0;
   else
        u_p_prime = tmp+tmp1-tmp2;
   /********calculating u_p_double_prime*********/
    tmp2 = etx_p_L/MAX_APS_PAYLOAD_LEN;
    tmp3 = (MAX_APS_PAYLOAD_LEN-s_f)*t_p-(MAX_APS_PAYLOAD_LEN - s_g)*(MAX_APS_PAYLOAD_LEN-s_p)*t_f;
    g_mod = (MAX_APS_PAYLOAD_LEN - s_f-(t_f*(MAX_APS_PAYLOAD_LEN-s_p))/t_p)%(MAX_APS_PAYLOAD_LEN-s_g);
    if(g_mod > 0)
        G = calculateETX_l((s_g+g_mod),4);
    else
        G=0;
    if(G==ROUTE_INVALID)
        return TRUE; //has to hold it
    tmp4 = (tmp3*etx_g_L+G)/(tmp3*s_g+MAX_APS_PAYLOAD_LEN - s_f);
    if((tmp1+tmp2)>=(tmp2+tmp4))
        u_p_dounble_prime = tmp+tmp1-tmp2-tmp4;
    else
        u_p_dounble_prime = 0;
   /**********calculating u_p_triple_prime*******************/
        AC_p_prime = etx_p_sp / s_p;
		tmp1 = (MAX_APS_PAYLOAD_LEN-s_f)/(MAX_APS_PAYLOAD_LEN - s_p);
        if (tmp1*(MAX_APS_PAYLOAD_LEN - s_p)<(MAX_APS_PAYLOAD_LEN-s_f))
			tmp1++;			

		l_mod = (MAX_APS_PAYLOAD_LEN-s_f)%(MAX_APS_PAYLOAD_LEN - s_p);
		if (l_mod > 0){
           	etx_p_sp_lmod = calculateETX_l(s_p+l_mod, 2);
           	if (etx_p_sp_lmod == ROUTE_INVALID)
		        return TRUE; //has to hold it           	
           	 AC_p = ((tmp1 * etx_p_L)+etx_p_sp_lmod) / (tmp1 * s_p + MAX_APS_PAYLOAD_LEN - s_f);	
		}else{
		    AC_p = (tmp1 * etx_p_L) / (tmp1 * s_p + MAX_APS_PAYLOAD_LEN - s_f);
		}		
		u_p_triple_prime= AC_p_prime - AC_p; 
     /*************Deciding u_p*********************************************/
     if((t_f * (MAX_APS_PAYLOAD_LEN - s_p) <= (MAX_APS_PAYLOAD_LEN - s_f) * t_p) &&
     (t_f*(MAX_APS_PAYLOAD_LEN - s_p) <= ((MAX_APS_PAYLOAD_LEN - s_f)*t_g-t_f*t_g*(MAX_APS_PAYLOAD_LEN-s_g)/t_p)))
        u_p = u_p_prime;
     else if((t_f * (MAX_APS_PAYLOAD_LEN - s_p) <= (MAX_APS_PAYLOAD_LEN - s_f) * t_p) &&
     (t_f * (MAX_APS_PAYLOAD_LEN - s_p) > (((MAX_APS_PAYLOAD_LEN - s_f) * t_g)-t_f*t_g*(MAX_APS_PAYLOAD_LEN-s_g)/t_p)))
        u_p = u_p_dounble_prime;
     else
        u_p = u_p_triple_prime;
      
      
      
      
     /**************Calculating u_g_prime*************************************/
        tmp = etx_g_sg/s_g;
        tmp1 = etx_p_sp/s_p;
        tmp2 = (etx_g_L+etx_p_L)/MAX_APS_PAYLOAD_LEN;
        if(tmp+tmp1<tmp2)
        u_g_prime = 0;
            else
        u_g_prime = tmp+tmp1-tmp2;
      /*************Calculating u_g_double_prime*********************************/
        p_mod = (MAX_APS_PAYLOAD_LEN - s_f-(t_f*(MAX_APS_PAYLOAD_LEN-s_g))/t_g)%(MAX_APS_PAYLOAD_LEN-s_p);
        if(g_mod > 0)
                P = calculateETX_l((s_p+p_mod),4);
        else
                P = 0;
        tmp2 = etx_g_L/MAX_APS_PAYLOAD_LEN;
        tmp3 = (MAX_APS_PAYLOAD_LEN-s_f)*t_g-(MAX_APS_PAYLOAD_LEN - s_p)*(MAX_APS_PAYLOAD_LEN-s_g)*t_f;
        tmp4 = (tmp3*etx_p_L+P)/(tmp3*s_p+MAX_APS_PAYLOAD_LEN - s_f);
        if((tmp1+tmp2)>=(tmp2+tmp4))
              u_g_double_prime = tmp+tmp1-tmp2-tmp4;
        else
              u_g_double_prime = 0;
       /*************Calculating u_g_triple_prime**********************************/
        AC_p_prime = etx_g_sg / s_g;
		tmp1 = (MAX_APS_PAYLOAD_LEN-s_f)/(MAX_APS_PAYLOAD_LEN - s_g);
        if (tmp1*(MAX_APS_PAYLOAD_LEN - s_p)<(MAX_APS_PAYLOAD_LEN-s_f))
			tmp1++;			
		l_mod = (MAX_APS_PAYLOAD_LEN-s_f)%(MAX_APS_PAYLOAD_LEN - s_g);
		if (l_mod > 0){
           	etx_p_sp_lmod = calculateETX_l(s_g+l_mod, 4);
           	if (etx_p_sp_lmod == ROUTE_INVALID)
		        return TRUE; //has to hold it           	
           	AC_p = ((tmp1 * etx_g_L)+etx_p_sp_lmod) / (tmp1 * s_g + MAX_APS_PAYLOAD_LEN - s_f);	
		}else{
		    AC_p = (tmp1 * etx_g_L) / (tmp1 * s_g + MAX_APS_PAYLOAD_LEN - s_f);
		}		
		u_g_triple_prime= AC_p_prime - AC_p; 
       /*************Deciding u_p*********************************************/
     if((t_f * (MAX_APS_PAYLOAD_LEN - s_g) <= (MAX_APS_PAYLOAD_LEN - s_f) * t_p) &&
     t_f * (MAX_APS_PAYLOAD_LEN - s_p) <= (((MAX_APS_PAYLOAD_LEN - s_f) * t_p)-t_f*t_p*(MAX_APS_PAYLOAD_LEN-s_g)/t_g))
        u_g = u_g_prime;
     else if((t_f * (MAX_APS_PAYLOAD_LEN - s_g) <= (MAX_APS_PAYLOAD_LEN - s_f) * t_p) &&
     t_f * (MAX_APS_PAYLOAD_LEN - s_p) > (((MAX_APS_PAYLOAD_LEN - s_f) * t_p)-t_f*t_p*(MAX_APS_PAYLOAD_LEN-s_g)/t_g))
        u_g = u_g_double_prime;
     else
        u_g = u_g_triple_prime;
      
        if(u_p >=u_g)
            u_im = u_p;
        else
            u_im = u_g;
      
      
      /***************DECIDE*************************/
       if(B_l > u_im){
            dbg("DBG_ERROR", "GOOD TO HOLD!!!!!!!!!!!!!!!!!!!!!!!!\n");
            return TRUE;
            }
       else{
            dbg("DBG_ERROR", "BAD TO HOLD!!!!!!!!!!!!!!!!!!!!!!!!\n");
            return FALSE;
            }
} //end of isBeneficialToHold2Hop

#endif