/* *
 * @ author: Xiaohui Liu (whulxh@gmail.com) 
 * @ updated: 06/27/2012 
 * @ description: data plane of iMAC, including data & ctrl packets
 *
 * priority calculation: f(link, t) = link_idx ^ (t & 0xFF)
 * 1) efficient: only involves native bit operation and link_idx is pre-computed only once
 * 2) fair: Semi round-robin as t goes repeatedly from 0 to 255
 
 * in CONTROL slot
 * besides CONTROL packets, ftsp beacon is also sent
 
// * slot integrity: tx/rx at most 1 pkt to ensure everything finishes within a slot
//		|- DATA channel
//			|-- enforce: DATA and ACK can be lost bcoz ACKs are not filter by h/w address recognition
//			|-- Not enforce (current solution): if contention resolution works, a sender not rx DATA & a receiver at most 1 DATA bcoz of h/w addr recognition. Acks may cause issue but they are short so hopefully processing them does not go beyond a slot
//		|- CTRL channel
//			|-- sender: not allowed to rx
//			|-- receiver: at most 1 rx
 */
#include "IMACForwarder.h"
#include "IMAC.h"
#include "Util.h"
#include "IMACController.h"
#include "SignalMap.h"
#include <Tasklet.h>

module IMACForwarderP {
	provides {
		interface AsyncAMSend as AMSend;
		interface AsyncReceive as Receive;
		interface AsyncPacket as Packet;
		
		interface AsyncSplitControl as SplitControl;
		interface Init;
		interface ForwarderInfo;
	};
	
	uses {
		// DATA pkt
		interface AsyncAMSend as SubSend;
		interface AsyncReceive as SubReceive;
		interface AsyncReceive as SubSnoop;
		interface AsyncPacket as SubPacket;
		interface AsyncAMPacket as SubAMPacket;
		// control pkt
		interface AsyncAMSend as CtrlSend;
	#ifdef SCREAM
		interface GeneralIO as CCA;
	#endif
		interface PacketAcknowledgements as Acks;
//		interface CC2420Config;
		interface CC2420Packet;
		interface RadioState;
		interface PacketField<uint8_t> as PacketTransmitPower;
		//interface GeneralIO as CCA;
		
		interface LinkEstimator;
		interface SignalMap;
		interface IMACController as Controller;
		
		interface LocalTime<TMicro>;
		interface GlobalTime<TMicro>;
	//#warning T32khz alarm
		//interface Alarm<TMicro, uint16_t> as SlotTimerMicro;
		// select 32khz bcoz micro alarm can only support 2 ^ 16 us natively, not enough for 256 ms
		interface Alarm<T32khz, uint32_t> as SlotTimer32khz;
		// select micro bcoz 32khz alarm is limited and runs out
		// used to further divide a slot into multiple sub-slots to tx/rx more ctrl packets
		interface Alarm<TMicro, uint32_t> as CommSubSlotTimerMicro;
		interface Alarm<TMicro, uint32_t> as ComputationSubSlotTimerMicro;
		
		interface Random;
		interface Util;
		interface BusyWait<TMicro, uint16_t>;
		interface UartLog;
		interface DriverInfo;
		
		interface Boot;
	};
}

implementation {

uint16_t seqno;
am_addr_t my_ll_addr;

uint8_t my_local_link_idx;

// receiver address of the unicast data pkt being transmitted
am_addr_t m_data_addr;
message_t *m_data_p;
uint8_t m_data_len;
message_t *m_control_p;
message_t m_control;

bool is_data_pending;
bool is_1st_tx_slot;

#if defined(VARY_PDR_REQ) || defined(VARY_PERIOD)
//#warning VARY_PDR_REQ enabled
uint8_t wraparound_cnt;
bool is_wraparound;
bool is_inc;
uint8_t pdr_req_idx;
uint8_t pdr_reqs[] = {70, 80, 90, 95};
#endif

link_t *activeLinks;
uint8_t active_link_size;
#ifdef SCREAM
// i-th element indicates if the link starting with this node is activated when i-th link is controlling
// this size should be equal to active_link_size
bool isAllocatedSlot[ACTIVE_LINK_SIZE];
// schedule computed
bool isScheduled = FALSE;
// is any tx in the concurrent set failed
bool is_any_tx_fail;
// # of tx success in a window
uint16_t success_cnt;
// link index in activeLinks
uint8_t control_link_idx, active_link_idx, my_outgoing_link_idx, round_offset;
uint32_t slot_since_tdma_start;
uint8_t wraparound_cnt;
bool is_wraparound;
#endif

local_link_er_table_entry_t *localLinkERTable;
link_er_table_entry_t *linkERTable;

// prefix "g_" represents global time
uint32_t g_next_firing_time;
uint32_t next_tx_slot;

// communication subslot
//bool is_ftsp_slot;
uint8_t beacon_cnt;
uint8_t elapsed_slot_since_last_win;

// to estimate tx probability
uint32_t data_tx_slot_cnt;
uint32_t slot_cnt;
// scale 100x
uint8_t data_tx_slot_ratio;
// TODO: wraparound
async command uint8_t ForwarderInfo.getDataTxSlotRatio() {
	return data_tx_slot_ratio;
//	call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, 0, 0, 0, 0, data_tx_slot_cnt, slot_cnt);
//	return (slot_cnt ? (100 * data_tx_slot_cnt / slot_cnt) : 100);
}

// tx ftsp beacon instead of ctrl pkt 1 out of const_ctrl_slot_ftsp_chance when ctrl channel available
// initially be small for quick convergence of ftsp
//uint16_t const_ctrl_slot_ftsp_chance_mask;

// # of consecutive CCA contension failure
//uint8_t cca_fail_cnt;

// is forwarder enabled
bool enabled;
task void splitControlStartDoneTask() {
	signal SplitControl.startDone(SUCCESS);	
}

// start data tx/rx after ftsp converges
//async command error_t SplitControl.start() {
//	atomic enabled = TRUE;
//	atomic const_ctrl_slot_ftsp_chance_mask = CTRL_SLOT_FTSP_CHANCE_MASK;
//#ifdef VARY_PDR_REQ	
//	is_wraparound = FALSE;
//	is_inc = TRUE;
//	pdr_req_idx = 0;
//	call UartLog.logEntry(DBG_FLAG, DBG_CONTROLLER_FLAG, __LINE__, pdr_req_idx);
//	call Controller.setLinkPdrReq(pdr_reqs[pdr_req_idx]);
//#endif
//	// we can look up once here bcoz my_link's position in localLinkERTable does not change
//	atomic my_local_link_idx = call Controller.findMyLinkLocalIdx();
//#ifndef SCREAM
//	post splitControlStartDoneTask();
//#endif
//	// freeze SM
//	call SignalMap.freeze();
//	return SUCCESS;
//}
async command error_t SplitControl.start() {
	return SUCCESS;
}

event void Boot.booted() {
	call SlotTimer32khz.start((uint32_t)SM_BEACON_PERIOD_MILLI << 5);
}

// ---------------- snapshot conflict graph ------------------------------
uint8_t link_idx;
// use task to avoid log overload
task void getConflictSetTask() {
	uint8_t i, my_local_link_idx_;
	link_er_table_entry_t *le;
	
	atomic {
		i = link_idx++;
		my_local_link_idx_ = my_local_link_idx;
	}
//	for (i = 0; i < LINK_ER_TABLE_SIZE; i++) {
		le = &linkERTable[i];
		if (!(le->flags & VALID_FLAG))
			return;
		// contend w/ this link
		if (le->contend_flags & (0x1 << my_local_link_idx_))
			call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, i, my_ll_addr, call Util.getReceiver(), le->sender, le->receiver, 0);
//	}
	if (i < LINK_ER_TABLE_SIZE)
		post getConflictSetTask();
}

void getConflictSet() {
	// not a sender; or the corresponding receiver fails to be programmed
	uint8_t my_link_idx = call Controller.findLinkERTableIdx(my_ll_addr, call Util.getReceiver());
	if (my_link_idx >= LINK_ER_TABLE_SIZE) {
		return;
	}
	
	link_idx = 0;
	post getConflictSetTask();
}

void startForwarder() {
	atomic enabled = TRUE;
	//atomic const_ctrl_slot_ftsp_chance_mask = CTRL_SLOT_FTSP_CHANCE_MASK;
#if defined(VARY_PDR_REQ) || defined(VARY_PERIOD)
	wraparound_cnt = 0;
	is_wraparound = FALSE;
	is_inc = TRUE;
	pdr_req_idx = 0;
	call UartLog.logEntry(DBG_FLAG, DBG_CONTROLLER_FLAG, __LINE__, pdr_req_idx);
	#if defined(VARY_PDR_REQ)
	call Controller.setLinkPdrReq(pdr_reqs[pdr_req_idx]);
	#endif
#endif
	// we can look up once here bcoz my_link's position in localLinkERTable does not change
	atomic my_local_link_idx = call Controller.findMyLinkLocalIdx();
#ifndef SCREAM
	post splitControlStartDoneTask();
#endif
	// freeze SM
	call SignalMap.freeze();
	
//#warning snapshot conflict graph
//	getConflictSet();	
}

async command error_t SplitControl.stop() {
	atomic enabled = FALSE;
	call SlotTimer32khz.stop();
	return SUCCESS;
}

async command bool ForwarderInfo.isForwarderEnabled() {
	bool enabled_;
	atomic enabled_ = enabled;
	return enabled_;
}

//async command bool ForwarderInfo.isDataPending() {
//	bool is_data_pending_;
//	atomic is_data_pending_ = is_data_pending;
//	return is_data_pending_;
//}
#if defined(VARY_PERIOD)
uint32_t period = SLOT_LEN;
// represent saturated, heavy, medium, and light traffic load accordingly
uint32_t period_scales[] = {1, 8, 64, 512}; //too heavy and maybe undifferentiable {1, 2, 5, 10};
//uint32_t periods[] = {20, 50, 100, 200};
// @return ms from us
async command uint32_t ForwarderInfo.getPeriod() {
	uint32_t period_;
	atomic period_ = period;
	return (period_ >> 10);
}
#endif

// get the link estimation header in the packet
imac_header_t* getHeader(message_t* m) {
	return (imac_header_t*)call SubPacket.getPayload(m, sizeof(imac_header_t));
}
// get the pdr footer in the packet
// @param len: payload length of this layer
void* getFooter(message_t* m, uint8_t len) {
	return (void*)(len + (uint8_t *)call Packet.getPayload(m, len + sizeof(imac_header_t)));
}
uint8_t addLinkEstHeaderAndFooter(message_t *msg, uint8_t len, uint32_t next_slot_by_tx);

command error_t Init.init() {
#ifdef SCREAM
	uint8_t i;
#endif
	//call SlotTimer32khz.start(SM_BEACON_PERIOD_MILLI << 10);
	g_next_firing_time = INVALID_TIME;
	//const_ctrl_slot_ftsp_chance_mask = INIT_CTRL_SLOT_FTSP_CHANCE_MASK;
	atomic {
		enabled = FALSE;
		seqno = 0;
		is_data_pending = FALSE;
	}
	is_1st_tx_slot = TRUE;
	next_tx_slot = INVALID_SLOT;

	my_ll_addr = call SubAMPacket.address();
	activeLinks = call Util.getActiveLinks(&active_link_size);
	localLinkERTable = call Controller.getLocalLinkERTable();
	linkERTable = call Controller.getLinkERTable();
	m_control_p = &m_control;
	
	// has never won initially
	elapsed_slot_since_last_win = CONTENTION_INTERVAL;
#ifdef SCREAM
	my_outgoing_link_idx = active_link_size;
	for (i = 0; i < active_link_size; i++) {
		if (activeLinks[i].sender == my_ll_addr)
			my_outgoing_link_idx = i;
	}
	wraparound_cnt = 0;
	is_wraparound = FALSE;
#endif
	data_tx_slot_cnt = 0;
	slot_cnt = 0;
	data_tx_slot_ratio = 100;

	return SUCCESS;
}

//uint32_t tdma_cnt = 0;
//uint32_t tx_win_cnt = 0;
//uint32_t rx_win_cnt = 0;
uint32_t start_time;
// prevent start_time being override
bool pending;

#include "IMACForwarderPUtil.nc"

// --------------------------------------------------------------------------------------
// 							TDMA
// --------------------------------------------------------------------------------------
// each slot starts
async event void SlotTimer32khz.fired() {
	bool is_fired_early = FALSE;
	uint16_t backoff;
	uint32_t g_now, local_interval, elapsed_interval;
	uint32_t slack = 0;

	if (enabled) {
		// jump start finishes and sync
		if (SUCCESS == call GlobalTime.getGlobalTime(&g_now)) {
			// align to the next next slot boundary
			// elapsed_interval = g_now & SLOT_HEX_MODULAR;
			elapsed_interval = g_now % SLOT_LEN;
			
			if (g_next_firing_time != INVALID_TIME) {
				// too early
				//if (g_now < g_next_firing_time) {
				if ((int32_t)(g_next_firing_time - g_now) > 0) {
					is_fired_early = TRUE;
					local_interval = (uint32_t)2 * SLOT_LEN - elapsed_interval;
					call SlotTimer32khz.start(local_interval >> 5);
					// expected next firing instant
					g_next_firing_time = g_now + local_interval;
					
					slack = SLOT_LEN - elapsed_interval;
					// wait till slot starts to be aligned; g_next_firing_time remains
					call BusyWait.wait(slack);
					// do not have to be precise; fine as long as the corresponding slot is right
					g_now += slack;
				} else {
					// too late
					local_interval = SLOT_LEN - elapsed_interval;
					g_next_firing_time = g_now + local_interval;
				}
			} else {
				// last slot was invalid, fire after (SLOT_LEN - elapsed_interval)
				local_interval = SLOT_LEN - elapsed_interval;
				g_next_firing_time = g_now + local_interval;
			}
		#if defined(VARY_PDR_REQ) || defined(VARY_PERIOD)
			//#warning online pdr req change
			// change period: 2 ^ 32 us; scale 100 times to avoid skipped slots
			if (g_now < ((uint32_t)SLOT_LEN * 100)) {
				if (!is_wraparound) {
					// wraparound just occurred
					is_wraparound = TRUE;
					// switch every WRAPAROUND_CNT rounds, one round is 2^32 us, including the 1st switch
					if (++wraparound_cnt >= WRAPAROUND_CNT) {
						wraparound_cnt = 0;
						// 0 -> 1 -> 2 -> 3 -> 2 -> 1 -> 0
						pdr_req_idx  = is_inc ? (pdr_req_idx + 1) : (pdr_req_idx - 1);
						if (pdr_req_idx == sizeof(pdr_reqs) / sizeof(pdr_reqs[0])) {
							is_inc = FALSE;
							pdr_req_idx = pdr_req_idx - 2;
						}
						if (0 == pdr_req_idx)
							is_inc = TRUE;
						call UartLog.logEntry(DBG_FLAG, DBG_CONTROLLER_FLAG, __LINE__, pdr_req_idx);	
					#if defined(VARY_PDR_REQ)
						// 70 -> 80 -> 90 -> 95 -> 90 -> 80 ->70
						call Controller.setLinkPdrReq(pdr_reqs[pdr_req_idx]);
					#elif defined(VARY_PERIOD)
						// 1 -> 2 -> 5 -> 10 -> 5 -> 2 -> 1x
						period = SLOT_LEN * period_scales[pdr_req_idx];
					#else
						#warning "sth smells fishy here"
					#endif
					}
				}
			} else {
				is_wraparound = FALSE;
			}
		#endif	
		} else {
			// indicate invalid global time; overridden in getGlobalTime()
			g_now = INVALID_TIME;
			g_next_firing_time = INVALID_TIME;
			// in case not sync
			local_interval = SLOT_LEN;
		}
		if (!is_fired_early)
			call SlotTimer32khz.start(local_interval >> 5);
		//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, is_fired_early, g_next_firing_time >> 16, g_next_firing_time, g_now >> 16, g_now, local_interval);

		scheduleSlot(g_now);
    } else {
    	// start TDMA together; otherwise receiver who has not started may miss DATA packets
    	if (SUCCESS == call GlobalTime.getGlobalTime(&g_now)) {
    		if (g_now >= GLOBAL_TDMA_START_TIME) {
    			call UartLog.logEntry(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, g_now - GLOBAL_TDMA_START_TIME);
    			//call SplitControl.start();
			//#warning disabled fowarder	
				startForwarder();
    		}
    	}
		// add uint32_t cast, otherwise overflow for 50 ms even decleared as 50UL
    	local_interval = ((uint32_t)SM_BEACON_PERIOD_MILLI << 10);
		call SlotTimer32khz.start(local_interval >> 5);
    	// always in CTRL channel b4 forwarder is started
    	//call RadioState.setChannel(CC2420_CONTROL_CHANNEL);
//		signal ForwarderInfo.slotStarted(FALSE);
		//// directly tx/rx control; no contention resolution or channel switch
		backoff = call Random.rand16();
		backoff &= CW_HEX_MODULAR;
		// backoff is to random backoff for channel contention
		call BusyWait.wait(backoff);
		//txrxCtrl(SLOT_MASK);
		// alternate between ctrl and ftsp beacon
	#ifndef SCREAM
		if (backoff & 0x1) {
			call PacketTransmitPower.set(m_control_p, CONTROL_POWER_LEVEL);
			call CtrlSend.send(AM_BROADCAST_ADDR, m_control_p, 0);
		} else {
			call GlobalTime.sendFtspBeacon();
		}
	#else
		// SCREAM does not need to exchagne control
		call GlobalTime.sendFtspBeacon();
	#endif
		//call UartLog.logTxRx(DBG_FLAG, DBG_HEARTBEAT_FLAG, __LINE__, 0, 0, 0, SM_BEACON_PERIOD_MILLI, local_interval >> 10, local_interval);
    }
    // do not start timer here bcoz long processing time of scheduleSlot() & txrxCtrl(), to name a few
	//call SlotTimer32khz.start(local_interval);
}

#ifdef SCREAM
// listen to channel to see if it is busy
inline bool listen() {
	uint8_t i;
	
	for (i = 0; i < LISTEN_CNT; i++) {
		if (!call CCA.get()) {
			return TRUE;
		} else {
			// pause and listen again
			call BusyWait.wait(LISTEN_INTERVAL);
		}
	}
	return FALSE;
}
#endif

// do scheduling in a slot specified by g_slot_time
void scheduleSlot(uint32_t g_slot_time) {
//	uint32_t next_slot_by_tx;
	uint32_t current_slot;
//	uint8_t status;
	uint8_t newlen;
#ifdef SCREAM
	uint32_t t, frame_offset;
#elif defined(OLAMA_DISABLED)
	uint16_t backoff;
#endif
	error_t ret = ECANCEL;
	if (slot_cnt >= TX_PROB_SAMPLE_WINDOW) {
		// EWMA
		data_tx_slot_ratio = data_tx_slot_ratio - (data_tx_slot_ratio >> 3) + (uint8_t)((100 * data_tx_slot_cnt / slot_cnt) >> 3);
		// reset
		slot_cnt = 0;
		data_tx_slot_cnt = 0;
	}
	
	slot_cnt++;
	if (elapsed_slot_since_last_win < 255)
		elapsed_slot_since_last_win++;
	/*
	 * DATA: unicast, regular power, channel CC2420_DEF_CHANNEL
	 * CONTROL: broadcast, highest power, channel CC2420_CONTROL_CHANNEL
     * 
     *	switch to DATA channel
     *  compute next slot to tx at receiver
     *	if reach my slot to tx
     *		compute next slot to tx
     *		tx data as sender if any
     *	elif reach my slot to rx
     *		stay in DATA channel
     *	else
     *		switch to CONTROL channel		
     *		random backoff: implicit through processing jitter and global time jitter
     *		if CCA clear (CCA not done here, at CC2420TransceiverP$CC2420XDriverConfig$requiresRssiCca())
     *			set max power
     *			tx ctrl pkt
     *		fi
     *		// switch back to DATA channel: not switch here bcoz it's hard to guess when ctrl tx finishes and alarm firing jitter
     *	fi
     */
//	// slot is divided into 2 subslots: communication subslot 0 to exchange ctrl info; computation subslot 1 to update OLAMA states
//	// initial some state needed for OLAMA
//	beacon_cnt = 0;
//	call CommSubSlotTimerMicro.start(DATA_SUBSLOT_LEN);
//	call Controller.initOLAMA();
//	call ComputationSubSlotTimerMicro.start(COMM_SUBSLOT_LEN);
	//call UartLog.logEntry(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, call ComputationSubSlotTimerMicro.isRunning());
	
#ifdef OLAMA_DISABLED
	// stay in default DATA channel to avoid DATA miss due to switch to channel late
	call RadioState.setChannel(CC2420X_DEF_CHANNEL);
#endif
	// contention resolution

 	// convert global time to time slot index bcoz
	// 1) even 1 jiffy's difference in global time across diff. nodes can lead to priority inconsistency and thus "collision"; while time slot index is consensus if sync error is less than one slot
	// 2) implicit assumption: alarm fires right after perfect slot boundary, almost never before it, which can be seem from "alarm_firing_modulo_1024.fig"
	current_slot = g_slot_time / SLOT_LEN;
//#warning take a snapshot of controller every 16 slots
//	if (0 == (current_slot & 0x0F))
//		call Controller.getControllerStatus();
	
#ifndef SCREAM
	// slot is divided into 2 subslots: communication subslot 0 to exchange ctrl info; computation subslot 1 to update OLAMA states
	// initial some state needed for OLAMA
	beacon_cnt = 0;
#ifndef OLAMA_DISABLED
	call CommSubSlotTimerMicro.start(DATA_SUBSLOT_LEN);
	call Controller.initOLAMA();
	call ComputationSubSlotTimerMicro.start(COMM_SUBSLOT_LEN);
#endif
	// OLAMA disables this bcoz next slot is only computed at sender
	// receiver-based tx slot; computed every slot
	//nextRxSlot(current_slot);

	// jump start
//	if (is_1st_tx_slot && g_slot_time != INVALID_TIME) {
//		next_slot_by_tx = call Controller.nextTxSlot(current_slot, TRUE);
//		//call UartLog.logEntry(DBG_FLAG, DBG_HEARTBEAT_FLAG, __LINE__, next_slot_by_tx - current_slot);
//		if (next_slot_by_tx != INVALID_SLOT)
//			is_1st_tx_slot = FALSE;
//	}
	if (INVALID_SLOT == next_tx_slot) {
		next_tx_slot = call Controller.nextTxSlot(current_slot, TRUE);
	}
	// signal start of a slot in CONTROL channel, DATA channel mostly unnecessary bcoz of h/w address recoginition; used in CC2420TransmitP to ensure slot integrity?? no h/w addr recog in cc2420x
	//signal ForwarderInfo.slotStarted(tx_win || rx_win);
	// in OLAMA both isTxSlot() and isRxSlot() can be TRUE bcoz of conservatively staying in DATA channel, tx instead of rx
	// this cannot occur in LAMA
	if (call Controller.isTxSlot(current_slot)) {
		data_tx_slot_cnt++;
		//status = 0;
		
		// ensure receiver has switched to DATA channel
		call BusyWait.wait(MIN_CW);
//		// randomize to sample NI during rx
//		call BusyWait.wait(MIN_CW + backoff);
		next_tx_slot = call Controller.nextTxSlot(current_slot, FALSE);
		if (is_data_pending) {
			// piggyback right before tx
			newlen = addLinkEstHeaderAndFooter(m_data_p, m_data_len, next_tx_slot);
		#ifdef HETER_TX_POWER
			call PacketTransmitPower.set(m_data_p, call Util.getNodeTxPowerLevel(my_ll_addr));
		#endif
			start_time = call LocalTime.get();
			ret = call SubSend.send(m_data_addr, m_data_p, newlen);
		} else {
			// this can happen if AMSend.send() is called within task; not serious
			//assert(tdma_cnt);
		}
		// log here to account for tx slots w/o data		
		call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, is_data_pending, call Controller.isRxSlot(current_slot), ret, getConflictSetSize(), next_tx_slot - current_slot, current_slot);
	} else if (call Controller.isRxSlot(current_slot)) {
		; //status = 1;
	} else {
	// control slot
		//status = 2;
	#ifdef OLAMA_DISABLED
		// switch to control channel
		//call CC2420Config.switchChannel(CC2420_CONTROL_CHANNEL);
		call RadioState.setChannel(CC2420_CONTROL_CHANNEL);
		backoff = call Random.rand16();
		//backoff &= CW_HEX_MODULAR;
		//MIN_CW is wait to tx to prevent rx before receiver's SlotTimer32khz fires, causing jitter and slot misalignment
		// backoff is to random backoff for channel contention
		//call BusyWait.wait(MIN_CW + backoff);
		call BusyWait.wait(MIN_CW);
		// txrxCtrl(current_slot);
		if (backoff & 0x1) {
			call PacketTransmitPower.set(m_control_p, CONTROL_POWER_LEVEL);
			call CtrlSend.send(AM_BROADCAST_ADDR, m_control_p, 0);
		} else {
			call GlobalTime.sendFtspBeacon();
		}
	#endif
	}
//#warning log
//	call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, current_slot & SLOT_MASK, call RadioState.getChannel(), status, m_data_addr, next_tx_slot - current_slot, current_slot);

#else		//SCREAM start here
	// time wraparound every 2 ^ 32 us; scale 100 times to avoid skipped slots
	if (g_slot_time < ((uint32_t)SLOT_LEN * 100)) {
		if (!is_wraparound) {
			// wraparound just occurred
			is_wraparound = TRUE;
			wraparound_cnt++;
		}
	} else {
		is_wraparound = FALSE;
	}	
	// overflow when wraparound
	//slot_since_tdma_start = (g_slot_time - GLOBAL_TDMA_START_TIME) / SLOT_LEN;
	slot_since_tdma_start = g_slot_time / SLOT_LEN;
	// 1 wraparound is (2^32 / SLOT_LEN) slots; not use 2^32 bcoz of overflow
	slot_since_tdma_start +=  (((uint32_t)1 << 31) / SLOT_LEN) * 2 * wraparound_cnt;
	slot_since_tdma_start -= GLOBAL_TDMA_START_TIME / SLOT_LEN;
#warning log
	if (0 == (slot_since_tdma_start & 0xF))
		call UartLog.logTxRx(DBG_FLAG, DBG_HEARTBEAT_FLAG, __LINE__, active_link_size, is_wraparound, wraparound_cnt, g_slot_time, (g_slot_time >> 16), slot_since_tdma_start);	
	frame_offset = slot_since_tdma_start % FRAME_LEN;
	// compute schedule
	if (!isScheduled) {
		if (slot_since_tdma_start < (uint32_t)active_link_size * FRAME_LEN) {
			// controlling link; frame_idx
			control_link_idx = slot_since_tdma_start / FRAME_LEN;
			// this is right bcoz ftsp slot is the last one in a frame
			round_offset = frame_offset % ROUND_LEN;
			
			// DATA slot
			if (round_offset < WINDOW_SIZE) {
				// initialize
				if (0 == round_offset) {
					// by default, all nodes relay SCREAM
					is_any_tx_fail = FALSE;
					success_cnt = 0;	
				}
				
				// active link; round_idx
				active_link_idx = frame_offset / ROUND_LEN;
				// starting from control_link_idx for fairness; if from 0, links ordered behind are less likely to be included in concurrent set
				active_link_idx += control_link_idx;
				if (active_link_idx >= active_link_size)
					active_link_idx -= active_link_size;
				
				// TODO: cope w/ incident links specially
				// a link tx only if it is allocated, in control, or active
				if (isAllocatedSlot[control_link_idx] || my_outgoing_link_idx == control_link_idx || my_outgoing_link_idx == active_link_idx) {
					// tx m_control_p since it is not used anyway and no application data submitted yet; its payload is not initialized
					// reset to default power
					call PacketTransmitPower.set(m_control_p, CC2420X_DEF_RFPOWER);
					call Acks.requestAck(m_control_p);
					//newlen = addLinkEstHeaderAndFooter(m_data_p, m_data_len, next_slot_by_tx);
					call SubSend.send(call Util.getReceiver(), m_control_p, call SubSend.maxPayloadLength());
				}
			} else {
			// SCREAM slot
				//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, isAllocatedSlot[control_link_idx], control_link_idx, active_link_idx, FRAME_LEN, round_offset, slot_since_tdma_start);
				// SCREAM
				if (round_offset < (ROUND_LEN - 1)) {
					//#warning scream at control channel
					//call RadioState.setChannel(CC2420_CONTROL_CHANNEL);
					if (is_any_tx_fail) {
						// shout using largest power
						call PacketTransmitPower.set(m_control_p, CONTROL_POWER_LEVEL);
						call SubSend.send(AM_BROADCAST_ADDR, m_control_p, call SubSend.maxPayloadLength());
					} else {
						is_any_tx_fail = listen();
//						if (is_any_tx_fail)
//							call UartLog.logEntry(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, slot_since_tdma_start);
					}
				} else {
				// make decision & ftsp
				// TODO: move to the 1st slot of next round to save 1 slot
					// for active link
					if (my_outgoing_link_idx == active_link_idx) {
						isAllocatedSlot[control_link_idx] = !is_any_tx_fail;
						if (is_any_tx_fail)
							call UartLog.logEntry(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, slot_since_tdma_start); // round_offset);
					}
					// a controlling link is always concurrent w/ itself
					if (my_outgoing_link_idx == control_link_idx) {
						isAllocatedSlot[control_link_idx] = TRUE;
					}
					txrxCtrl();
				}
			}
		} else {
			// put here instead of outside bcoz of time wrap around
			isScheduled = TRUE;
			post splitControlStartDoneTask();
		}
	} else {
		// ftsp frequency: FTSP_SLOT_RATIO_BASE / (FTSP_SLOT_RATIO_BASE + active_link_size)
		t = current_slot % (active_link_size + FTSP_SLOT_RATIO_BASE);
		// data
		if (t < active_link_size) {
			// activate links according to the computed schedule
			if (isAllocatedSlot[t]) {
				call BusyWait.wait(MIN_CW);
				if (is_data_pending) {
					//#warning ack
					//call Acks.requestAck(m_data_p);
					newlen = addLinkEstHeaderAndFooter(m_data_p, m_data_len, next_tx_slot);
					ret = call SubSend.send(m_data_addr, m_data_p, newlen);
				}
				call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, is_data_pending, g_slot_time >> 16, ret, isScheduled, t, current_slot);
			}
		} else {
			txrxCtrl();
		}
	}
#endif
}

// communication subslot	
void startBeaconTimer() {
	//error_t ret;
	uint16_t backoff;
	
	// abort if reach computation subslot to avoid interfering w/ computation
	if (!call ComputationSubSlotTimerMicro.isRunning()) {
		call UartLog.logEntry(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, beacon_cnt);
		return;
	}
	
	if (++beacon_cnt > COMM_SUBSLOT_BEACON_CNT)
		return;
	
//	backoff = call Random.rand16();
//	backoff &= COMM_SUBSLOT_BEACON_CW_MASK;
//	backoff += COMM_SUBSLOT_BEACON_CW;
	backoff = (beacon_cnt <= COMM_SUBSLOT_FTSP_BEACON_CNT) ? COMM_SUBSLOT_FTSP_BEACON_PERIOD : COMM_SUBSLOT_CTRL_BEACON_PERIOD;
//#warning TDMA not CSMA	
//	//if (!is_ftsp_slot) {
//	if (beacon_cnt >= COMM_SUBSLOT_FTSP_BEACON_CNT) {
//		if (elapsed_slot_since_last_win < CONTENTION_INTERVAL) {
//			// do not participate in csma contention if won in the last CONTENTION_INTERVAL slots
//			// to yield channel to others to mimic round robin for fairness, thus to enforce SUCCESSFUL_SIGNALLING_INTERVAL
//			// abort timer here to save processing; no need to contend again, including start timer, in the slot bcoz already won once
//			return;
//		}
//	}
	call CommSubSlotTimerMicro.start(backoff);
	
	//if (!is_ftsp_slot) {
	if (beacon_cnt > COMM_SUBSLOT_FTSP_BEACON_CNT) {
		uint8_t my_turn_idx;
		uint32_t current_slot = g_next_firing_time / SLOT_LEN - 1;
		//call CC2420Packet.setPower(m_control_p, CONTROL_POWER_LEVEL);
		call PacketTransmitPower.set(m_control_p, CONTROL_POWER_LEVEL);
	//#warning 25 nodes
		my_turn_idx = beacon_cnt - COMM_SUBSLOT_FTSP_BEACON_CNT + (current_slot % CONTENTION_INTERVAL) * COMM_SUBSLOT_CTRL_BEACON_CNT;
		//my_turn_idx = beacon_cnt - COMM_SUBSLOT_FTSP_BEACON_CNT + 10 + (current_slot % 5) * 15;
		if (TOS_NODE_ID != my_turn_idx)
			return;
		call CtrlSend.send(AM_BROADCAST_ADDR, m_control_p, 0);
//		if (SUCCESS == ret) {
//			elapsed_slot_since_last_win = 0;
//		}
		//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, 0, 0, my_turn_idx, beacon_cnt, ret, current_slot);
	} else {
		// ftsp beacon is sent at regular power: NO
		call GlobalTime.sendFtspBeacon();
	}
	//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, is_ftsp_slot, COMM_SUBSLOT_BEACON_CW, COMM_SUBSLOT_BEACON_CNT, beacon_cnt, backoff, COMM_SUBSLOT_LEN);
}

async event void CommSubSlotTimerMicro.fired() {	
	startBeaconTimer();
}

// tx/rx control packets & ftsp beacons
// wait for reception if not sent 
void txrxCtrl() {
#ifdef SCREAM
	// randomize; otherwise out of sync even for FTSP_SLOT_RATIO_BASE of 28
	// PRKS does not have to randomize probably already implicit in processing delay while SCREAM has to
	uint16_t backoff;
	backoff = call Random.rand16();
	backoff &= CW_HEX_MODULAR;
	call BusyWait.wait(MIN_CW + backoff);
	call GlobalTime.sendFtspBeacon();
#else
//#warning ftsp disabled control
	//uint8_t t = current_slot & SLOT_MASK;
	//is_ftsp_slot = t < (MAX_SLOT_FORWARD - active_link_size);
	beacon_cnt = 0;
	startBeaconTimer();
#endif
}

// computation subslot
async event void ComputationSubSlotTimerMicro.fired() {	
	//call UartLog.logEntry(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, call ComputationSubSlotTimerMicro.isRunning());
	// run OLAMA here to avoid interference w/ packet tx/rx due to excessive computation
	//start_time = call LocalTime.get();
//#ifndef OLAMA_DISABLED
	//#warning disable OLAMA	
	call Controller.runOLAMA();
//#endif
	//call UartLog.logEntry(DBG_FLAG, DBG_DELAY_FLAG, __LINE__, call LocalTime.get() - start_time);
}

	
async event void SubSend.sendDone(message_t* msg, error_t error) {
	am_addr_t m_data_addr_;
#ifdef SCREAM
	bool isScheduled_;
#endif

	atomic {
		is_data_pending = FALSE;
		m_data_addr_ = m_data_addr;
	}
	// pin the receiver; only chance for m_data_addr to be in neighbor table if it never sends
	call LinkEstimator.pinNeighbor(m_data_addr_);
#ifndef SCREAM
	signal AMSend.sendDone(msg, error);
	//atomic call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, 0, 0, 0, enabled, error, call LocalTime.get() - start_time);
#else
	atomic isScheduled_ = isScheduled;
	if (isScheduled_)
		signal AMSend.sendDone(msg, error);
	//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, 1, SOFTWAREACK_TIMEOUT, error, isScheduled_, call SubAMPacket.destination(msg), call Acks.wasAcked(msg));
	if (SUCCESS == error) {
		if (!isScheduled_) {
			atomic {
				// only for data slot in a round
				if (round_offset < WINDOW_SIZE) {
					if (call Acks.wasAcked(msg)) {
						success_cnt++;
					}
				}
				// only for last data slot in a round
				if ((WINDOW_SIZE - 1) == round_offset) {
					is_any_tx_fail = (success_cnt * 100 / WINDOW_SIZE < REFERENCE_DATA_PDR);
//					is_any_tx_fail = !call Acks.wasAcked(msg);
					if (is_any_tx_fail)
						call UartLog.logEntry(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, slot_since_tdma_start);
				}
			}
		}
	}
#endif
}


// TODO: add busy protection to ctrl pkt 
async event void CtrlSend.sendDone(message_t* msg, error_t error) {
	//atomic call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, 0, 0, 0, enabled, error, call LocalTime.get() - start_time);
}

// channel switched
//async event void CC2420Config.syncDone(error_t error) {}
tasklet_async event void RadioState.done() {}


//-------------------------------------------------------------------------------
// Interface AMSend
//-------------------------------------------------------------------------------
// called right before SubSend$send(), not in AMSend$send(), to piggyback latest info
uint8_t addLinkEstHeaderAndFooter(message_t *msg, uint8_t len, uint32_t next_slot_by_tx) {
	int16_t k;
	uint8_t newlen;
	
	imac_header_t *hdr;
#ifdef SCREAM
	// make data packet long without piggyback of ctrl info
	k = 0;
#else
	link_er_footer_t *er_footer;

  	er_footer = (link_er_footer_t *)getFooter(msg, len);
	k = call Controller.loadLinkER(er_footer);
#endif	
	hdr = getHeader(msg);
  	hdr->seqno = seqno++;
  	hdr->link_er_cnt = k;
  	hdr->next_slot_by_tx = next_slot_by_tx;
  	newlen = sizeof(imac_header_t) + len + k * sizeof(link_er_footer_t);
	//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, 0, hdr->link_er_cnt, er_footer[0].sender, er_footer[0].receiver, er_footer[0].rx_er_version, hdr->seqno);
	return newlen;
}

async command error_t AMSend.send(am_addr_t addr, message_t* msg, uint8_t len) {
	atomic {
		if (is_data_pending)						return EBUSY;
	}
	if (len > call AMSend.maxPayloadLength())		return ESIZE;
	// not accept broadcast now for simplicity
	if (AM_BROADCAST_ADDR == addr) 					return FAIL;

	atomic {
		is_data_pending = TRUE;
		// "store" data pkt for later tx
		m_data_addr = addr;
		m_data_p = msg;
		m_data_len = len;
	}
	//call Packet.setPayloadLength(msg, len);
	return SUCCESS;
}

// cascade the calls down
async command uint8_t AMSend.cancel(message_t* msg) {
	return call SubSend.cancel(msg);
}

async command uint8_t AMSend.maxPayloadLength() {
	return call Packet.maxPayloadLength();
}

async command void* AMSend.getPayload(message_t* msg, uint8_t len) {
	return call Packet.getPayload(msg, len);
}


// new messages are received here
async event message_t* SubReceive.receive(message_t* msg, void* payload, uint8_t len) {
	am_addr_t from = call SubAMPacket.source(msg);
	imac_header_t *hdr = getHeader(msg);
	link_er_footer_t *er_footer = (link_er_footer_t *)getFooter(msg, call Packet.payloadLength(msg));
	
#ifdef SCREAM
	// do not forward control packets up
	atomic {
		if (!isScheduled)
			return msg;
	}
#endif
	// update next rx slot from piggyback info
	call Controller.updateNextSlot(from, TRUE, hdr->next_slot_by_tx);
	// received; do not have to conservatively wait in DATA channel anymore
	call Controller.clearDataPending(from);
	
	call Controller.updateLinkERTable(er_footer, hdr->link_er_cnt, from);
	//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, from, hdr->link_er_cnt, er_footer[0].sender, er_footer[0].receiver, er_footer[0].rx_er_version, hdr->seqno);
	
	return signal Receive.receive(msg, call Packet.getPayload(msg, call Packet.payloadLength(msg)), call Packet.payloadLength(msg));
}

// overhear
async event message_t* SubSnoop.receive(message_t* msg, void* payload, uint8_t len) {
	am_addr_t from = call SubAMPacket.source(msg);
	imac_header_t *hdr = getHeader(msg);
	link_er_footer_t *er_footer = (link_er_footer_t *)getFooter(msg, call Packet.payloadLength(msg));
	//call UartLog.logTxRx(DBG_FLAG, DBG_TDMA_FLAG, __LINE__, from, hdr->link_er_cnt, er_footer[0].sender, er_footer[0].receiver, er_footer[0].rx_er_version, hdr->seqno);
	call Controller.updateLinkERTable(er_footer, hdr->link_er_cnt, from);
	return msg;
}
//------------------------------------------------------------------------
// Interface Packet
//------------------------------------------------------------------------
async command void Packet.clear(message_t* msg) {
	call SubPacket.clear(msg);
}

// subtract the space occupied by the signal map header and footer from the incoming payload size
async command uint8_t Packet.payloadLength(message_t* msg) {
	imac_header_t *hdr = getHeader(msg);
	return (call SubPacket.payloadLength(msg) - sizeof(imac_header_t) - hdr->link_er_cnt * sizeof(link_er_footer_t));
}

// account for the space used by header and footer while setting the payload length
async command void Packet.setPayloadLength(message_t* msg, uint8_t len) {
	imac_header_t *hdr = getHeader(msg);
	call SubPacket.setPayloadLength(msg, len + sizeof(imac_header_t) + hdr->link_er_cnt * sizeof(link_er_footer_t));
}

async command uint8_t Packet.maxPayloadLength() {
	return (call SubPacket.maxPayloadLength() - sizeof(imac_header_t));
}

// application payload pointer is just past the link estimation header
async command void* Packet.getPayload(message_t* msg, uint8_t len) {
	void* payload = call SubPacket.getPayload(msg, len + sizeof(imac_header_t));
	if (payload != NULL) {
		payload += sizeof(imac_header_t);
	}
	return payload;
}

default async event void AMSend.sendDone(message_t* msg, error_t error ) {}
default async event message_t* Receive.receive(message_t* msg, void* payload, uint8_t len) {
	return msg;
}

// new inbound data/ack pdr arrives
async event error_t LinkEstimator.inLinkPdrUpdated(am_addr_t nb, bool is_ack) {	return SUCCESS;	}

}
