
#ifndef TEST_ROUTING_M_DEF
#define TEST_ROUTING_M_DEF 1 


#ifdef MINT_ROUTING
includes MultiHop;
#endif
#ifdef GRID_ROUTING
includes GridTreeMsg;	
#define GRID_HEAD_LEN (sizeof(RouteMsg))				
#endif
#if defined(GRID_ROUTING) || defined(LOF_ROUTING)
#define LOF_OR_GRID_ROUTING 1
#endif

#endif

module TestRoutingM 
{
  provides{
    interface StdControl;
  }
  uses{
    interface StdControl as RoutingControl;	
#ifdef LOF_ROUTING 
    interface Routing;
#endif    
#ifdef MINT_ROUTING
    interface Send;
    interface Intercept;
    interface RouteControl;
#endif
#ifdef GRID_ROUTING
    interface Routing;
#endif
   
#if !defined(TOSSIM) && defined(TSYCH)
    interface OTime as Time;
    interface StdControl as TimeSyncControl;
#endif

    interface StdControl as QueuedSendControl;
    interface QueueControl;
    interface SendMsg as UARTQueuedSend;

    interface ReceiveMsg as UARTReceive;
    interface SendMsg as UARTSend;

    interface StdControl as TimerControl; 
    interface Timer;

    interface Random32;

    interface Leds;
  }
}

implementation
{

  bool bPeriodicTraffic; //whether to generate periodic traffic
  uint32_t pktInterval;
  uint32_t startLatency_periodic; //for periodic traffic pattern

  bool bGoodput; //whether to test goodput

  //for trace-driven experiments
  uint16_t startTime, endTime;
  uint32_t nextStart, nextEnd;
  uint32_t roundControl;

  uint16_t seqNo, exptId; //seqno and exptID

  uint8_t powerLevels[NUM_POWER_LEVELS] = {3, 3, 3, 3};
  uint8_t maxTransmitCounts[NUM_MAX_TRANSMIT_COUNTS] = {8, 8, 8};
  uint8_t powerLevelPtr, maxTransmitCountPtr;
  uint8_t actualStartingPowerPtr, actualFinishingPowerPtr, actualStartingMTPtr, actualFinishingMTPtr;

  uint8_t numExptsDone; //for each parameter combination

  bool toExpt;  //whether to start underlying protocol execution
  bool toTraffic;  //whether to generate traffic 
  bool toSnoop;
  uint32_t baseSnoopingLatency; //if set to NULL32, then do not test base snooping at all

  bool startOnly, stopOnly, oddIdOnly;

  TOS_Msg outMsgBuf;
  TOS_Msg debugMsg;
  
  QueuedSend_Tuning_Msg tuningMsg;

  bool bInitialized; //Whether parameters have been initialized 

  uint16_t actualNumExptsPerParameterSet;

  bool UARTpending;
  uint32_t pendingDeadPeriod;

#ifdef DEBUG_ONLY
  uint16_t ticks;
  uint16_t counter;
#endif

  //debug
  uint16_t numUARTDrops, numRoutingDrops, 
                   numUARTUsedQS; //number of times UART_Send() used QueuedSend 


  /* =========================================================*/
  /* Initial most of the state variables
   *
   * Used in: StdControl.init(), UARTReceive.receive() 
   */
  void TestRoutingInit() {
    uint8_t i;
#if 0
    /* trace-driven expts: node id assignment based on 7*13 grid */
    uint16_t trace[TOTAL_NUM_OF_SENDERS][3] ={
      {1, 930, 7229}, //node id, start time, end time (milliseconds)
      {2, 1218, 8285},
      {3, 781, 7178},
      {4, 1014, 6490},
      {5, 1448, 6872},
      {6, 1306, 6782},

      {13, 0, 5533},
      {14, 1195, 6726},
      {15, 1837, 6549},
      {16, 1335, 7937},
      {17, 841, 8058},
      {18, 1844, 6757},
      {19, 2224, 6114},

      {26, 1670, 6433},
      {27, 1079, 7429},
      {28, 1473, 8847},
      {29, 2059, 8098}, 
      {30, 1793, 7323},
      {31, 2120, 7291},
      {32, 2376, 7036},

      {39, 2057, 6869},
      {40, 1957, 7745}, 
      {41, 2202, 7989},
      {42, 2362, 9220},
      {43, 1794, 8551},
      {44, 2119, 7597},
      {45, 2954, 7048},

      {52, 2755, 7055},
      {53, 1921, 7962},
      {54, 2116, 7798},
      {55, 2263, 9429},
      {56, 2518, 7792},
      {57, 3099, 6426},
      {58, 3128, 7121},

      {65, 2754, 7259}, 
      {66, 1871, 8984},
      {67, 2381, 8420},
      {68, 3035, 7794},
      {69, 3211, 7766},
      {70, 3040, 8107},
      {71, 9353, 14421},

      {78, 2967, 7676},
      {79, 3101, 7810}, 
      {80, 3270, 7928},
      {81, 2911, 9515},
      {82, 3093, 8674},
      {83, 2730, 8516},
      {84, 3343, 8258}
    };
    
#endif

    /* trace-driven expts: node id assignment based on 7*7 grid */
    uint16_t trace[TOTAL_NUM_OF_SENDERS][3] ={
      {1, 930, 7229}, //node id, start time, end time (milliseconds)
      {2, 1218, 8285},
      {3, 781, 7178},
      {4, 1014, 6490},
      {5, 1448, 6872},
      {6, 1306, 6782},

      {7, 0, 5533},
      {8, 1195, 6726},
      {9, 1837, 6549},
      {10, 1335, 7937},
      {11, 841, 8058},
      {12, 1844, 6757},
      {13, 2224, 6114},

      {14, 1670, 6433},
      {15, 1079, 7429},
      {16, 1473, 8847},
      {17, 2059, 8098}, 
      {18, 1793, 7323},
      {19, 2120, 7291},
      {20, 2376, 7036},

      {21, 2057, 6869},
      {22, 1957, 7745}, 
      {23, 2202, 7989},
      {24, 2362, 9220},
      {25, 1794, 8551},
      {26, 2119, 7597},
      {27, 2954, 7048},

      {28, 2755, 7055},
      {29, 1921, 7962},
      {30, 2116, 7798},
      {31, 2263, 9429},
      {32, 2518, 7792},
      {33, 3099, 6426},
      {34, 3128, 7121},

      {35, 2754, 7259}, 
      {36, 1871, 8984},
      {37, 2381, 8420},
      {38, 3035, 7794},
      {39, 3211, 7766},
      {40, 3040, 8107},
      {41, 9353, 14421},

      {42, 2967, 7676},
      {43, 3101, 7810}, 
      {44, 3270, 7928},
      {45, 2911, 9515},
      {46, 3093, 8674},
      {47, 2730, 8516},
      {48, 3343, 8258}
    };
    
    atomic{
      for (i = 0; i < TOTAL_NUM_OF_SENDERS && trace[i][0] != TOS_LOCAL_ADDRESS; i++)
	;
      if (i < TOTAL_NUM_OF_SENDERS) { //find the trace record
	  startTime = trace[i][1];
	  endTime = trace[i][2];
      }
      else 
	startTime = endTime = NULL16;

      toExpt =  toSnoop = toTraffic = FALSE; 
      roundControl = NULL32;
      nextStart = nextEnd = NULL32;

      seqNo = exptId = 0;
      numExptsDone = 0;

      baseSnoopingLatency = NULL32; //not test base snooping 

      startOnly = stopOnly =  oddIdOnly = FALSE;

      powerLevelPtr = actualStartingPowerPtr = STARTING_POWER_PTR;
      actualFinishingPowerPtr = FINISHING_POWER_PTR;
      maxTransmitCountPtr = actualStartingMTPtr = STARTING_MT_PTR;
      actualFinishingMTPtr = FINISHING_MT_PTR;

    }//atomic

    atomic UARTpending = FALSE;

  } //end of TestRoutingInit()


  command result_t StdControl.init(){

    TestRoutingInit(); //also used in UART receive

    actualNumExptsPerParameterSet = DEFAULT_NUM_EXPTS_PER_PARAMETER_SET;
    bInitialized = FALSE;

    bPeriodicTraffic = FALSE;
    startLatency_periodic = NULL32;

    bGoodput = FALSE;

    //Kansei
#if !defined(TOSSIM) && !defined(LOCAL_DEBUG)
    atomic TOS_LOCAL_ADDRESS = 100;  //to prevent GridRouting from firing; any non-zero value
#endif

#if defined(TOSSIM) || defined(LOCAL_DEBUG)
    atomic toExpt = TRUE;
    atomic toTraffic = TRUE;
    atomic roundControl = START_LATENCY;

    //for periodic traffic
    bPeriodicTraffic = TRUE; //for periodic traffic
    pktInterval = 60 * 1000; //packet interval in milliseconds
    if (startLatency_periodic == NULL32)
      startLatency_periodic = START_LATENCY + (call Random32.rand32());
    nextStart = startLatency_periodic % (2 * pktInterval);
    nextEnd = nextStart + pktInterval;

    //dbg(DATA_DEBUG, "INIT: nextStart=%u, nextEnd=%u\n", nextStart, nextEnd);

    call RoutingControl.init();
#endif

#ifdef LOCAL_DEBUG
    //Hongwei: atomic TOS_LOCAL_ADDRESS =1;		
    powerLevels[0]=powerLevels[1] = 2;    	
    actualNumExptsPerParameterSet = DEFAULT_NUM_EXPTS_PER_PARAMETER_SET;    	
    maxTransmitCounts[0]=maxTransmitCounts[1] = 1;
    
    powerLevelPtr = actualStartingPowerPtr = STARTING_POWER_PTR;
    tuningMsg.transmissionPower = powerLevels[powerLevelPtr];
    maxTransmitCountPtr = actualStartingMTPtr = STARTING_MT_PTR;
    tuningMsg.maxTransmitCount = maxTransmitCounts[maxTransmitCountPtr];
    actualFinishingMTPtr = FINISHING_MT_PTR;
    parameterTuning(&tuningMsg);
#endif

#ifdef DEBUG_ONLY 
    ticks = 0;
    counter = 0;
#endif

    numUARTDrops = 0;
    numRoutingDrops = 0;
    numUARTUsedQS = 0;

    call Random32.init();

    call TimerControl.init();
    call Leds.init();

    return call QueuedSendControl.init();    
  } //end of StdContro.init()


  command result_t StdControl.start(){

    call TimerControl.start();
    call Timer.start(TIMER_REPEAT, TIMER_INTERVAL);
    
#if defined(TOSSIM) || defined(LOCAL_DEBUG)
    call RoutingControl.start();
#endif

    return call QueuedSendControl.start();		    
  }


  command result_t StdControl.stop(){
    result_t ok1, ok2;
		
    ok1 = call RoutingControl.stop();
    ok2 = call Timer.stop();

    call TimerControl.stop();

#if !defined(TOSSIM) && defined(TSYCH)
    call TimeSyncControl.stop();
#endif

    call QueuedSendControl.stop();

    return rcombine(ok1,ok2);
  }


  /* =========================================================*/
  /* Lifeng: tune parameters*/
  void parameterTuning(QueuedSend_Tuning_Msg* tuningMsgPtr)
  { 
    //integrity checking
    if (tuningMsgPtr->maxTransmitCount > MAX_TRANSMIT_COUNT)
      return;

    call QueueControl.parameterTuning(tuningMsgPtr);
  } //end of parameterTuning(...)


  uint8_t UART_Send(uint8_t size,TOS_Msg* msg) 
  {
    if (!UARTpending) { //to directly send to UART
      atomic {
	UARTpending = TRUE;
	pendingDeadPeriod = 0;
      }

      if (!(call UARTSend.send(TOS_UART_ADDR, size, msg))) {
	atomic UARTpending = FALSE;

	numUARTUsedQS++;

	//still needs to turn to QueuedSend 
	if (!(call UARTQueuedSend.send(TOS_UART_ADDR, size, msg))) {
	  numUARTDrops++;
	  return FAIL;
	}
      }
    }
    else { //to use QueuedSend
      numUARTUsedQS++;

      if (!(call UARTQueuedSend.send(TOS_UART_ADDR, size, msg))) {
	numUARTDrops++;
	return FAIL;
      }
    }

    return SUCCESS;
  } //end of UART_send()


  /* used for testing goodput */
  void nextSend()
  {
    AppMsg * msg;
    link_pkt_head* link_hd;
#ifdef  MINT_ROUTING
    uint16_t Len;
#endif

    if (!bGoodput || numExptsDone >= actualNumExptsPerParameterSet)
      return;

    link_hd = (link_pkt_head*)outMsgBuf.data;				
    link_hd->source = TOS_LOCAL_ADDRESS;

#ifdef GRID_ROUTING
    msg = (AppMsg *)(outMsgBuf.data+GRID_HEAD_LEN);
#elif defined(LOF_ROUTING)
    msg = (AppMsg *)(outMsgBuf.data+LOF_HEAD_LEN);
#elif defined(MINT_ROUTING)
    msg=(AppMsg *)call Send.getBuffer(&outMsgBuf, &Len);				
#endif

#ifndef TOSSIM
#ifdef TSYCH
      msg->sendingTime = (call Time.getGlobalTime32()); //currentTime.clock;
#else
      msg->sendingTime = 0;
#endif
#else
      msg->sendingTime = 0;
#endif
 
    atomic {
      msg->src = TOS_LOCAL_ADDRESS;
      msg->seq = seqNo;
      if (seqNo%2 == 0)
	msg->type = 10;
      else
	msg->type = 11;

      msg->exptId = exptId;
      msg->powerLevel = tuningMsg.transmissionPower;//powerLevels[powerLevelPtr];
      msg->maxTransmitCount = tuningMsg.maxTransmitCount;//maxTransmitCounts[maxTransmitCountPtr];

      seqNo++;
      if (msg->type == 11) {
	exptId++;
	numExptsDone++;
      }
    } //atomic

#if defined(GRID_ROUTING) || defined(LOF_ROUTING)
    if (!(call Routing.send(&outMsgBuf,LOF_HEAD_LEN+sizeof(AppMsg))))
      numRoutingDrops++; 
#elif defined(MINT_ROUTING)
    if (!(call Send.send(&outMsgBuf,sizeof(AppMsg))))
      numRoutingDrops++;
#endif

  } //end of nextSend()


  /* Timer fired: event run */
  event result_t Timer.fired(){
    AppMsg * msg;
    link_pkt_head* link_hd;
#ifdef  MINT_ROUTING
    uint16_t Len;
#endif
#ifdef DEBUG_ONLY 
    uint16_t *p16;
#endif

#if 0  //debug periodic-traffic pattern only
    startTime = 1;
    toExpt = TRUE;
    toTraffic = TRUE;
    //numExptsDone = 0;
    bPeriodicTraffic = TRUE; //for periodic traffic
    pktInterval = (uint32_t)((uint32_t)60 * (uint32_t)1000); //packet interval in milliseconds
#endif

#ifdef DEBUG_ONLY 
    //debug whether UART interface suite is working well
    ticks++;
    if (!toExpt && ticks > 60000) //for debugging UART only 
      return SUCCESS;
    //send to uart 
    if (ticks%FREQ_UART_SEND == 0) {
      p16 = (uint16_t *)(&(debugMsg.data[0]));
      *p16 = counter; //exptId;//ticks;
      //
      p16 = (uint16_t *)(&(debugMsg.data[2])); 
      *p16 = numUARTDrops;
      p16 = (uint16_t *)(&(debugMsg.data[4])); 
      *p16 = numRoutingDrops;
      //
      p16 = (uint16_t *)(&(debugMsg.data[6])); 
      *p16 = call Routing.getNumSendButNoParent(); //should be no more than RoutingDrops
      p16 = (uint16_t *)(&(debugMsg.data[8])); 
      *p16 = call Routing.getNumFwdButNoParent();
      //
      p16 = (uint16_t *)(&(debugMsg.data[10])); 
      *p16 = call QueueControl.get_num_pkt_dropped();
      p16 = (uint16_t *)(&(debugMsg.data[12]));
      *p16 = (uint16_t)(call QueueControl.get_num_self_pkt_dropped());
      //
      p16 = (uint16_t *)(&(debugMsg.data[14]));
      *p16 = (uint16_t)(call QueueControl.getOccupancy());
      p16 = (uint16_t *)(&(debugMsg.data[16]));
      *p16 = call QueueControl.getNumUnicastTxes();
      p16 = (uint16_t *)(&(debugMsg.data[18])); 
      *p16 = call QueueControl.getNumPendingBeenStabilized();
      //
      p16 = (uint16_t *)(&(debugMsg.data[20])); 
      *p16 = call Routing.getNumProbeReplySent();
      p16 = (uint16_t *)(&(debugMsg.data[22])); 
      *p16 = call Routing.getNumRouteWithdrawalSent();
      p16 = (uint16_t *)(&(debugMsg.data[24])); 
      *p16 = call Routing.getNumReplyBasedNgbrRevitalization();
      //
      debugMsg.data[26] = call Routing.getNumChildren();
      //
      debugMsg.data[27] = call Routing.getNumSH();
      //
      debugMsg.data[28] = call Routing.getNumSH2();

      UART_Send(29, &debugMsg);
    }
#endif

    //stabilization
    if (UARTpending) {
      pendingDeadPeriod += TIMER_INTERVAL;
      if (pendingDeadPeriod > PENDING_DEAD_THRESHOLD)
	atomic UARTpending = FALSE;
  }

#ifdef INJECT_TRAFFIC_PATTERN
    return SUCCESS;	
#endif 

    //is base station, not generating traffic, protocol not started, or not to generate traffic
    if (!toExpt || !toTraffic || TOS_LOCAL_ADDRESS == BASE_ID || startTime == NULL16)
      return SUCCESS;

    /* non-base node checks whether to send packets and change system parameters */
    if (bPeriodicTraffic == FALSE) { //is for event traffic
      if (roundControl != NULL32 && roundControl > TIMER_INTERVAL) {
	roundControl -= TIMER_INTERVAL;
      }
      else if (roundControl != NULL32 && startTime != NULL16) {
	atomic {
	  roundControl = EXPT_INTERVAL; 
	  nextStart = startTime;
	  nextEnd = endTime;
	}

#ifdef DEBUG_ONLY 
	counter++;
#endif
      }
    }
    else { //for periodic traffic 
      if (startLatency_periodic == NULL32)
	startLatency_periodic = (call Random32.rand32());

      if (nextEnd == NULL32) { //another "period/round"
#ifdef DEBUG_ONLY 
	counter++;
#endif

	if (numExptsDone == 0) {	
#if 0
	  nextStart = startLatency_periodic % (2 * pktInterval);
	  nextEnd = nextStart + pktInterval;
#else
	  nextStart = (call Random32.rand32()) % (2 * pktInterval);  
	  nextEnd = nextStart + ((call Random32.rand32()) % (2 * pktInterval)); 
#endif
	}
	else {
#if 0
	  nextStart = pktInterval;
	  nextEnd = nextStart + pktInterval; 
#else
	  nextStart = (call Random32.rand32()) % (2 * pktInterval); 
	  nextEnd = nextStart + ((call Random32.rand32()) % (2 * pktInterval));
#endif
	}
 
	if (nextEnd == NULL32)
	  nextEnd--;

	//dbg(DATA_DEBUG, " RESET timing: nextStart=%u, nextEnd=%u \n", nextStart, nextEnd);
      }
    }

    if (
#ifdef TOSSIM
          TRUE //keeps generating traffic for ever 
#else 
           numExptsDone < actualNumExptsPerParameterSet
           //&& seqNo < (NUM_PKTS_PER_EVENT_PER_NODE * actualNumExptsPerParameterSet)
#endif
       ){ //still need to do more expts with the current parameter setting

      if (bGoodput == TRUE) {
	nextSend();
	atomic toTraffic = FALSE;  //traffic will be generated in .sendDone() event
	return SUCCESS;
      }

      //start-packet
      if (nextStart != NULL32 && nextStart > TIMER_INTERVAL)
	atomic nextStart -= TIMER_INTERVAL;
      else if (nextStart != NULL32 && !stopOnly &&
	(!oddIdOnly || (TOS_LOCAL_ADDRESS%2 == 1)) ) { //need to send; also schedule the next send
	atomic nextStart = NULL32;

	//compose the message to send
	link_hd = (link_pkt_head*)outMsgBuf.data;				
	link_hd->source =TOS_LOCAL_ADDRESS;

#ifdef GRID_ROUTING
	msg = (AppMsg *)(outMsgBuf.data+GRID_HEAD_LEN);
#elif defined(LOF_ROUTING)
	msg = (AppMsg *)(outMsgBuf.data+LOF_HEAD_LEN);
#elif defined(MINT_ROUTING)
	msg=(AppMsg *)call Send.getBuffer(&outMsgBuf, &Len);				
#endif

#ifndef TOSSIM
#ifdef TSYCH
	  msg->sendingTime = (call Time.getGlobalTime32()); //currentTime.clock;
#else
	  msg->sendingTime = 0;
#endif
#else					  
	  msg->sendingTime = 0;
#endif					

	atomic {
	  msg->src = TOS_LOCAL_ADDRESS;
	  msg->seq = seqNo;
	  msg->type = 10;

	  msg->exptId = exptId;
	  msg->powerLevel = tuningMsg.transmissionPower;//powerLevels[powerLevelPtr];
	  msg->maxTransmitCount = tuningMsg.maxTransmitCount;//maxTransmitCounts[maxTransmitCountPtr];

	  seqNo++;
	  if (startOnly) {
	    numExptsDone++;
	    exptId++;
	  }

	  dbg(DATA_DEBUG, " ----> sends packet %d;   nextStart=%u, nextEnd=%u \n", seqNo-1, nextStart, nextEnd);
	  //dbg(DATA_DEBUG, " ----> sends packet %d, time = %d;   nextStart=%u, nextEnd=%u \n", seqNo-1, tos_state.tos_time, nextStart, nextEnd);
#if defined(GRID_ROUTING) || defined(LOF_ROUTING)
	  if (!(call Routing.send(&outMsgBuf,LOF_HEAD_LEN+sizeof(AppMsg))))
	    numRoutingDrops++;
#elif defined(MINT_ROUTING)
	  if (!(call Send.send(&outMsgBuf,sizeof(AppMsg))))
	    numRoutingDrops++; 
#endif
	}
      }

      //end-packet; if send, also increase numExptsDone & exptId by one
      if (nextEnd != NULL32 && nextEnd > TIMER_INTERVAL) {
	atomic nextEnd -= TIMER_INTERVAL;
      }
      else if (nextEnd != NULL32 && !startOnly &&
	(!oddIdOnly || (TOS_LOCAL_ADDRESS%2 == 1))) { //need to send; also schedule the next send
	atomic nextEnd = NULL32;

	//compose the packet to send
	link_hd = (link_pkt_head*)outMsgBuf.data;				
	link_hd->source = TOS_LOCAL_ADDRESS;

#ifdef GRID_ROUTING
	msg = (AppMsg *)(outMsgBuf.data+GRID_HEAD_LEN);
#elif defined(LOF_ROUTING)
	msg = (AppMsg *)(outMsgBuf.data+LOF_HEAD_LEN);
#elif defined(MINT_ROUTING)
	msg = (AppMsg *)call Send.getBuffer(&outMsgBuf, &Len);
#endif

#ifndef TOSSIM					
#ifdef TSYCH
	  msg->sendingTime = (call Time.getGlobalTime32()); //currentTime.clock;
#else
	  msg->sendingTime = 0;
#endif 
#else				
	  msg->sendingTime = 0;
#endif

	atomic {
	  msg->src = TOS_LOCAL_ADDRESS; 
	  msg->seq = seqNo;
	  msg->type = 11;

	  msg->exptId = exptId;
	  msg->powerLevel = tuningMsg.transmissionPower;//powerLevels[powerLevelPtr];
	  msg->maxTransmitCount = tuningMsg.maxTransmitCount;//maxTransmitCounts[maxTransmitCountPtr];

	  seqNo++;
	  numExptsDone++;
	  exptId++;
	}

	  dbg(DATA_DEBUG, " ----> sends packet %d;   nextStart=%u, nextEnd=%u \n", seqNo-1, nextStart, nextEnd);
	  //dbg(DATA_DEBUG, " ----> sends packet %d, time = %d;   nextStart=%u, nextEnd=%u \n", seqNo-1, tos_state.tos_time, nextStart, nextEnd);
#if defined(GRID_ROUTING) || defined(LOF_ROUTING)
	if (!(call Routing.send(&outMsgBuf,LOF_HEAD_LEN+sizeof(AppMsg))))
	  numRoutingDrops++;
#elif defined(MINT_ROUTING)
	if (!(call Send.send(&outMsgBuf,sizeof(AppMsg))))
	  numRoutingDrops++;
#endif
      } //end-packet
    }//if (numExptsDone < actualNumExptsPerParameterSet)
    else { //need to change to another parameter setting, or to exit
      //Lifeng: stop generating traffic 
      atomic {
	toTraffic = FALSE;
	bPeriodicTraffic = FALSE;
	bGoodput = FALSE;
      }
      if (TOS_LOCAL_ADDRESS != BASE_ID)
	roundControl = START_LATENCY;
      else 
	roundControl = NULL32;

      return SUCCESS;
      /*Original
      atomic{
	numExptsDone = 0;
	if (powerLevelPtr < (actualFinishingPowerPtr+1) && maxTransmitCountPtr == ((actualFinishingMTPtr+1)-1))
	  powerLevelPtr++;
	maxTransmitCountPtr = ((maxTransmitCountPtr + 1) % (actualFinishingMTPtr+1));
	if (maxTransmitCountPtr == 0)
	  maxTransmitCountPtr = actualStartingMTPtr;
      }
      if (powerLevelPtr < (actualFinishingPowerPtr+1)) {
	atomic {
	  tuningMsg.transmissionPower = powerLevels[powerLevelPtr];
	  tuningMsg.maxTransmitCount = maxTransmitCounts[maxTransmitCountPtr];
	}
	parameterTuning(&tuningMsg);
      }
      else if (!toSnoop && baseSnoopingLatency != NULL32) {
	atomic {
	  toSnoop = TRUE;
	  powerLevelPtr = actualStartingPowerPtr;
	  maxTransmitCountPtr = actualStartingMTPtr; //SNOOP_BEGINING_TRANXIT_COUNT; //only snoop for ... 
	  tuningMsg.transmissionPower = powerLevels[powerLevelPtr];
	  tuningMsg.maxTransmitCount = maxTransmitCounts[maxTransmitCountPtr];
					
	  parameterTuning(&tuningMsg);
	  roundControl = BASE_SET_SNOOPING_LATENCY;
	  //roundControl += (START_LATENCY - EXPT_INTERVAL);
	}
      }
      else //has done Snoop test
	{ 
	  atomic toTraffic = FALSE;
	}
      */
    }//change to another parameter setting 
     
    return SUCCESS;
  } //end of Timer.fired()
 

#ifdef INJECT_TRAFFIC_PATTERN
  void SendPacket(uint8_t seq)
  {				  	
    AppMsg * msg;
    				
#ifdef  MINT_ROUTING
    uint16_t Len;
#endif   

    if (seq < seqNo)
      return;
				
#ifdef GRID_ROUTING
    msg = (AppMsg *)(outMsgBuf.data+GRID_HEAD_LEN);
#endif				
#ifdef LOF_ROUTING					
    msg = (AppMsg *)(outMsgBuf.data+LOF_HEAD_LEN);
#endif
#ifdef MINT_ROUTING
    msg=(AppMsg *)call Send.getBuffer(&outMsgBuf, &Len);				
#endif

    atomic {
      msg->src = TOS_LOCAL_ADDRESS;
      msg->seq = seqNo;
      msg->type = 9;
    }
    atomic {
#ifndef TOSSIM
#ifdef TSYCH
      msg->sendingTime = (call Time.getGlobalTime32()); //currentTime.clock;
#else
      msg->sendingTime = 0;
#endif
#else					  
      msg->sendingTime = 0;
#endif			
		
      msg->exptId = exptId;
      msg->powerLevel = tuningMsg.transmissionPower;
      msg->maxTransmitCount = tuningMsg.maxTransmitCount;
					  
      seqNo++;
      exptId++;
    }

#if defined(GRID_ROUTING) || defined(LOF_ROUTING)
    if (!(call Routing.send(&outMsgBuf,LOF_HEAD_LEN+sizeof(AppMsg))))
      numRoutingDrops++;
#endif
#ifdef MINT_ROUTING
    if (!(call Send.send(&outMsgBuf,sizeof(AppMsg))))
      numRoutingDrops++;
#endif
 
  }//end of SendPacket
#endif  //INJECT_TRAFFIC_PATTERN


  /* receive expt. commands */
  event TOS_MsgPtr UARTReceive.receive(TOS_MsgPtr m)
  {
    uint16_t *p16;

#ifdef INJECT_TRAFFIC_PATTERN 
    SendPacket(m->data[0]);
    return m;
#endif

    if (toExpt) {
      if (m->data[0] == 100) { //start event traffic
	atomic {
	  toTraffic = TRUE;
	  numExptsDone = 0; //useful in generating temporally dynamic traffic
	}
	debugMsg.data[0] = TOS_LOCAL_ADDRESS;
	debugMsg.data[1] = powerLevels[0];
	debugMsg.data[2] = actualNumExptsPerParameterSet;
	debugMsg.data[3] = maxTransmitCounts[0];
	UART_Send(4, &debugMsg);
      }
      else if (m->data[0] == 110) { //start periodic traffic
	atomic {
	  toTraffic = TRUE;
	  numExptsDone = 0;
	  bPeriodicTraffic = TRUE; //for periodic traffic
	  pktInterval = (uint32_t)(((uint32_t)(m->data[1])) * (uint32_t)1000); //packet interval in milliseconds
	}
	call Random32.init(); //so as to produce more-or-less repeatable random sequences for individual expts
	debugMsg.data[0] = TOS_LOCAL_ADDRESS + 1;
	debugMsg.data[1] = powerLevels[0];
	debugMsg.data[2] = actualNumExptsPerParameterSet;
	debugMsg.data[3] = m->data[1];//maxTransmitCounts[0];
	UART_Send(4, &debugMsg);
      }
      else if (m->data[0] == 120) { //start goodput-test traffic
	atomic {
	  toTraffic = TRUE;
	  numExptsDone = 0; 
	  bGoodput = TRUE; //for goodput test
	}
	debugMsg.data[0] = TOS_LOCAL_ADDRESS + 2;
	debugMsg.data[1] = powerLevels[0];
	debugMsg.data[2] = actualNumExptsPerParameterSet;
	debugMsg.data[3] = maxTransmitCounts[0];
	UART_Send(4, &debugMsg);
      }
      else if (m->data[0] == 101) {//end of experiment: test whether motes are still alive (e.g., without segmentation fault)
	//to mark the completion of this experiment, and note down the number of failed UART transmission 
	//(due to queue overflow)
	debugMsg.data[0] = TOS_LOCAL_ADDRESS; 
	debugMsg.data[1] = powerLevels[0]; 
	debugMsg.data[2] = actualNumExptsPerParameterSet;
	debugMsg.data[3] = 0; //padding
	//
	p16 = (uint16_t *)(&(debugMsg.data[4]));
	*p16 = numUARTDrops;
	p16 = (uint16_t *)(&(debugMsg.data[6]));
	*p16 = numRoutingDrops;
	//
	p16 = (uint16_t *)(&(debugMsg.data[8]));
	*p16 = (uint16_t)(call QueueControl.get_num_self_pkt_dropped());
	p16 = (uint16_t *)(&(debugMsg.data[10])); 
	*p16 = call QueueControl.get_num_pkt_dropped();
	//
	p16 = (uint16_t *)(&(debugMsg.data[12])); 
	*p16 = numUARTUsedQS;
	//
	p16 = (uint16_t *)(&(debugMsg.data[14])); 
	*p16 = call QueueControl.getNumPendingBeenStabilized();

	UART_Send(16, &debugMsg); 
      }

      return m;
    }//toExpt

    if (bInitialized || m->data[4] != 1) //if it has already been initialized, or it is not a part of the experiment
      return m;

    atomic bInitialized=TRUE;
    atomic {//format of injected packet: <node-id, power-level, num-of-expts, max-num-tx-count, whether-to-be-part-of-an-expt>
      TOS_LOCAL_ADDRESS = (uint8_t)(m->data[0]);  //node id 
      powerLevels[0] = powerLevels[1] = (uint8_t)(m->data[1]);   //power level 
      actualNumExptsPerParameterSet = (uint8_t)(m->data[2]); //# of expts
      if(((uint8_t)m->data[3]) > MAX_TRANSMIT_COUNT)    //# of tx counts
	maxTransmitCounts[0] = maxTransmitCounts[1] = MAX_TRANSMIT_COUNT; 
      else
	maxTransmitCounts[0] = maxTransmitCounts[1] = (uint8_t)(m->data[3]); 
    } //atomic
 
    call Random32.init(); //depends on TOS_LOCAL_ADDRESS
    TestRoutingInit();
		
    /* initialize and start lower layer components */ 
    call RoutingControl.init();
#if !defined(TOSSIM) && defined(TSYCH)
    call TimeSyncControl.init();
#endif

    call RoutingControl.start();
#ifdef MINT_ROUTING		
    call RouteControl.setUpdateInterval(20/2); 
#endif
#if !defined(TOSSIM) && defined(TSYCH)	
    call TimeSyncControl.start(); 
#endif

    /*11-26-06: No need to restart timer anymore, since only TestRouting will initialize TimerC!
    //MUST restart timer since TimerC has been initialized by other components 
    call Timer.stop(); 
    call Timer.start(TIMER_REPEAT, TIMER_INTERVAL); 
    //restart timer for QueuedSend
    call QueueControl.restartTimer();
    */

    /* Expt control */
    atomic {
      toExpt = TRUE;
      startOnly = stopOnly = oddIdOnly = FALSE;
      if (TOS_LOCAL_ADDRESS != BASE_ID)
	roundControl = START_LATENCY;
      else 
	roundControl = NULL32; //base does not send data
      seqNo = exptId = numExptsDone = 0;
      nextStart = nextEnd = NULL32;
      baseSnoopingLatency = NULL32; //not to test base snooping at all

      //set whether to use baseSnooping
      toSnoop = FALSE;

      //set power level ptrs
      powerLevelPtr = actualStartingPowerPtr = STARTING_POWER_PTR;
      tuningMsg.transmissionPower = powerLevels[powerLevelPtr];
      actualFinishingPowerPtr = FINISHING_POWER_PTR;

      //set MT ptrs
      maxTransmitCountPtr = actualStartingMTPtr = STARTING_MT_PTR;
      tuningMsg.maxTransmitCount = maxTransmitCounts[maxTransmitCountPtr];
      actualFinishingMTPtr = FINISHING_MT_PTR;

      //parameters related to RC
      //tuningMsg.compConsrvContWaiting = 0; //to use the default
      //tuningMsg.compChannelUtilControl = 0; 
      //tuningMsg.compResendConsrv = 0; 
    } //atomic
    //call ReliableCommControl.setSnooping(toSnoop);
    parameterTuning(&tuningMsg);
    
    //for debugging purpose: write back to UART
    atomic {
    	debugMsg.data[0] = TOS_LOCAL_ADDRESS;
    	debugMsg.data[1] = powerLevels[0];    	
    	debugMsg.data[2] = actualNumExptsPerParameterSet;
	debugMsg.data[3] = maxTransmitCounts[0];
	debugMsg.data[4] = m->data[4];
      }
    UART_Send(5, &debugMsg);

    return m;
  } //UARTReceive.receive(...)


#ifdef LOF_ROUTING
  /*
   * Send log information via UART
   */
  event result_t Routing.reportValue(uint8_t* data, uint8_t flag, uint8_t size)
  {
    uint8_t i;

    debugMsg.data[0] = flag;
			
    if (data != NULL) {	
	for (i=0; i<size; i++)
	  debugMsg.data[i+1] = data[i];
    }
    for (i=size+1; i<sizeof(ReportedMsg); i++)
      debugMsg.data[i]=0xff;

    UART_Send(sizeof(ReportedMsg), &debugMsg);
    return SUCCESS;
  }
#endif


#ifdef MINT_ROUTING
  event result_t Send.sendDone(TOS_MsgPtr pmsg, result_t success)
#elif defined(LOF_OR_GRID_ROUTING)
    event result_t Routing.sendDone(TOS_MsgPtr pmsg, result_t success)
#endif
  {

    link_pkt_head* link_hd = (link_pkt_head*)pmsg->data;

    ReportedMsg * reportMsg=(ReportedMsg *)(debugMsg.data);
#ifdef LOF_ROUTING  
    AppMsg * msg=(AppMsg*)(pmsg->data+LOF_HEAD_LEN);    
#endif
#ifdef  MINT_ROUTING
    AppMsg * msg=(AppMsg*)(pmsg->data+offsetof(TOS_MHopMsg,data));
#endif
#ifdef GRID_ROUTING
    AppMsg * msg=(AppMsg*)(pmsg->data+GRID_HEAD_LEN);
#endif

#ifdef REDIRECT_FAIL_PKT  //defined in QueuedSend.h
    if (pmsg->ack == 0 && link_hd->tx_count < MAX_TX_IF_REDIRECT) //skip this senddone: failed and QueudSend is 
                                                                                                                                            //yet to redirect again 
      return SUCCESS;
#endif 

    if (TOS_LOCAL_ADDRESS == BASE_ID || pmsg->addr==TOS_UART_ADDR || pmsg->addr==TOS_BCAST_ADDR)
      return SUCCESS; 

    atomic {
      reportMsg->flag=1;    //send
      reportMsg->src = msg->src;
      reportMsg->seq = msg->seq;
      reportMsg->type = msg->type;
      reportMsg->sendingTime = msg->sendingTime;
		
      reportMsg->exptId = msg->exptId;
      reportMsg->powerLevel = msg->powerLevel;
      reportMsg->maxTransmitCount = msg->maxTransmitCount;
			
      reportMsg->dst = pmsg->addr; 
		
#ifndef TOSSIM
#ifdef TSYCH						    
      reportMsg->receptionTime = (call Time.getGlobalTime32()); //currentTime.clock;
#else
      reportMsg->receptionTime = 0;
#endif
#else	
      reportMsg->receptionTime = 0;
#endif
      reportMsg->tx_count = link_hd->tx_count;
      reportMsg->mac_latency = link_hd->delay;

      //if (success == SUCCESS)
      if (pmsg->ack != 0)
	reportMsg->success = 1; 
      else
	reportMsg->success = 0; 

      reportMsg->last_hop_id = link_hd->source; 

      dbg(DBG_USR3,"mac=%d\n",reportMsg->mac_latency);
			
      //
      reportMsg->queue_length=call QueueControl.getOccupancy();
      reportMsg->num_pkt_dropped=call QueueControl.get_num_pkt_dropped();
      reportMsg->num_self_pkt_dropped=call QueueControl.get_num_self_pkt_dropped();

#ifdef LOF_OR_GRID_ROUTING  	
      reportMsg->num_table_regeneration=call Routing.getNumofTableRegeneration();
#else
      reportMsg->num_table_regeneration=call RouteControl.getQuality();
#endif
      reportMsg->num_dead_ngbr=0;
      reportMsg->num_ngbr_switch=0;

#ifdef LOF_ROUTING
      reportMsg->num_dead_ngbr=call Routing.getNumofDeadNeighbors();
      reportMsg->num_ngbr_switch=call Routing.getSwitchCount();
#endif
    }
	
    dbg(DBG_USR3, "senddone, dst=%d with seq =%d\n", reportMsg->dst,reportMsg->seq);			
			
    UART_Send(sizeof(ReportedMsg), &debugMsg);

    if (bGoodput == TRUE)
      nextSend();

    return success;
  }//end of senddone 

  
#ifdef LOF_OR_GRID_ROUTING  
  /* receive packet from routing */
  event TOS_MsgPtr Routing.receive(TOS_MsgPtr pmsg) 
  {
    ReportedMsg * reportMsg = (ReportedMsg *)(debugMsg.data);

    link_pkt_head* link_hd = (link_pkt_head*)pmsg->data;
#ifdef LOF_ROUTING
    AppMsg * msg=(AppMsg*)(pmsg->data+LOF_HEAD_LEN);
#else
    AppMsg * msg=(AppMsg*)(pmsg->data+GRID_HEAD_LEN);
#endif

    atomic {	
      reportMsg->flag=0;    //receive
      reportMsg->src = msg->src;
      reportMsg->seq = msg->seq;
      reportMsg->type = msg->type;
      reportMsg->sendingTime = msg->sendingTime;
		
      reportMsg->exptId = msg->exptId;
      reportMsg->powerLevel = msg->powerLevel;
      reportMsg->maxTransmitCount = msg->maxTransmitCount;
			
      reportMsg->dst = TOS_LOCAL_ADDRESS;
			
#ifndef TOSSIM
#ifdef TSYCH						    
      reportMsg->receptionTime = (call Time.getGlobalTime32()); //currentTime.clock; 
#else
      reportMsg->receptionTime = 0;
#endif
#else	
      reportMsg->receptionTime = 0;
#endif
		
      //dummy value
      reportMsg->tx_count = 0xff;
      reportMsg->mac_latency = 0;
      reportMsg->success = 0xff;
		
      reportMsg->last_hop_id = link_hd->source;
		
      /*************************/
      reportMsg->queue_length = call QueueControl.getOccupancy();
      reportMsg->num_pkt_dropped = call QueueControl.get_num_pkt_dropped();
      reportMsg->num_self_pkt_dropped = call QueueControl.get_num_self_pkt_dropped();
  	
      reportMsg->num_table_regeneration=call Routing.getNumofTableRegeneration();
  	
      reportMsg->num_dead_ngbr=0;
      reportMsg->num_ngbr_switch=0;

#ifdef LOF_ROUTING
      reportMsg->num_dead_ngbr=call Routing.getNumofDeadNeighbors();
      reportMsg->num_ngbr_switch=call Routing.getSwitchCount();
#endif  	
    }

    if (TOS_LOCAL_ADDRESS == BASE_ID)
      dbg(DATA_DEBUG, "BASE receives packet: src = %d,   seq =%d\n", reportMsg->src,reportMsg->seq);
			
    UART_Send(sizeof(ReportedMsg), &debugMsg);
		  
    return pmsg;
  }//end of Routing.receive(...)
#endif

	
#ifdef MINT_ROUTING	
  event result_t Intercept.intercept(TOS_MsgPtr pmsg, void* payload, uint16_t payloadLen)
  {
    ReportedMsg * reportMsg=(ReportedMsg *)(debugMsg.data);
    AppMsg * msg=(AppMsg*)payload;
    link_pkt_head* link_hd=(link_pkt_head*)pmsg->data;
	    
    atomic {
      reportMsg->flag=0;    //receive			
      reportMsg->src = msg->src;
      reportMsg->seq = msg->seq;
      reportMsg->type = msg->type;
      reportMsg->sendingTime = msg->sendingTime;
			
      reportMsg->exptId = msg->exptId;
      reportMsg->powerLevel = msg->powerLevel;
      reportMsg->maxTransmitCount = msg->maxTransmitCount;
      reportMsg->dst=TOS_LOCAL_ADDRESS;
			
#ifndef TOSSIM
#ifdef TSYCH						    
      reportMsg->receptionTime = (call Time.getGlobalTime32()); //currentTime.clock;
#else
      reportMsg->receptionTime = 0;
#endif
#else	
      reportMsg->receptionTime = 0;
#endif
    }
			
    //dum value
    reportMsg->tx_count=0xff;
    reportMsg->mac_latency=0;
    reportMsg->success=0xff;
    reportMsg->last_hop_id=link_hd->source;

    reportMsg->queue_length = call QueueControl.getOccupancy();
    reportMsg->num_pkt_dropped = call QueueControl.get_num_pkt_dropped();
    reportMsg->num_self_pkt_dropped = call QueueControl.get_num_self_pkt_dropped();

    reportMsg->num_table_regeneration=call RouteControl.getQuality();
	  	
    reportMsg->num_dead_ngbr=0;
    reportMsg->num_ngbr_switch=0;
		
    dbg(DBG_USR3, "receive from %d with seq =%d\n", reportMsg->src,reportMsg->seq);

    UART_Send(sizeof(ReportedMsg), &debugMsg);
			  
    return SUCCESS;
  }	
#endif


  event result_t UARTSend.sendDone(TOS_MsgPtr pmsg, result_t success)
  {
    atomic UARTpending = FALSE;

    return SUCCESS;
  }//end of UARTSend.sendDone(...)


  event result_t UARTQueuedSend.sendDone(TOS_MsgPtr pmsg, result_t success)
  {
    return SUCCESS;
  }

}
