includes Alarm;

module AlarmM 
{
  provides { 
    interface Alarm[uint8_t id];
    interface StdControl;
    }
  uses {
    interface TimerMilli;
    interface StdControl as TimerControl;
    }
}

implementation

{
  uint32_t nextCheck;
  uint32_t theClock;
  uint16_t subClock;
  sched_list schedList[sched_list_size];

  void reNext() {
     uint32_t smallTime = 0xffffffff;
     uint8_t i;
     for (i=0; i<sched_list_size; i++) 
        if ((schedList+i)->wake_time < smallTime) 
        	smallTime = (schedList+i)->wake_time;  
     nextCheck = smallTime;
     }

  /**
   * Initializes the Alarm and Timer components.
   * @author herman@cs.uiowa.edu
   * @return Always returns SUCCESS.
   */
  command result_t StdControl.init() {
     uint8_t i;
     for(i=0;i<sched_list_size;i++) 
       (schedList+i)->wake_time = 0xffffffff; 
     nextCheck = 0xffffffff;
     theClock = 0;
     subClock = 0;
     call TimerControl.init();
     return SUCCESS;
     }

  /**
   * (Re)-Starts the Alarm and Timer components;
   * also kicks off the Timer with one second wait.
   * @author herman@cs.uiowa.edu
   * @return Always returns SUCCESS.
   */
  command result_t StdControl.start() {
     call TimerControl.start();
     call TimerMilli.setPeriodic( INTER_RATE ); 
     return SUCCESS;
     }

  /**
   * Stops and reinitializes the Alarm; stops the Timer.
   * @author herman@cs.uiowa.edu
   * @return Always returns SUCCESS.
   */
  command result_t StdControl.stop() {
     uint8_t i;
     for (i=0; i<sched_list_size; i++) 
   	(schedList+i)->wake_time = 0xffffffff;
     call TimerMilli.stop(); 
     call TimerControl.stop();
     return SUCCESS;
     }

  /**
   * Clears out the list of Alarm events (intended mainly
   * for failure/reset).
   * @author herman@cs.uiowa.edu
   * @return Always returns SUCCESS.
   */
  command result_t Alarm.clear[uint8_t id]() {
     uint8_t i;
     for (i=0; i<sched_list_size; i++)  
        if ((schedList+i)->wake_time != 0xffffffff &&
	    (schedList+i)->id == id)
   	    (schedList+i)->wake_time = 0xffffffff;
     return SUCCESS;
     }

  /**
   * Reads the current Alarm "clock" (seconds since initialized). 
   * @author herman@cs.uiowa.edu
   * @return Always returns current Alarm time (in seconds) 
   */
  command uint32_t Alarm.clock[uint8_t id]() {
     return theClock;
     }

  /**
   * Subroutine: sets the Alarm to fire at a specified Alarm time. 
   * @author herman@cs.uiowa.edu
   * @return Returns SUCCESS if alarm scheduled, otherwise FAIL 
   */
  result_t setAlarm ( uint8_t id, uint8_t indx, uint32_t wake_time) {
     uint8_t i;
     for (i=0; i<sched_list_size; i++) 
       if ((schedList+i)->wake_time == 0xffffffff) {
          (schedList+i)->wake_time = wake_time;
	  (schedList+i)->indx = indx;
	  (schedList+i)->id = id;
	  reNext();
	  return SUCCESS;
          }
     dbg(DBG_USR1, "*** Alarm setting failed for id %d, index %d, time %d\n",
          id, indx, wake_time);
     return FAIL;  // did not find empty slot
     }

  /**
   * Sets the Alarm to fire at a specified Alarm time. 
   * @author herman@cs.uiowa.edu
   * @return Returns SUCCESS if alarm scheduled, otherwise FAIL 
   */
  command result_t Alarm.schedule[uint8_t id] ( uint8_t indx, 
     uint32_t wake_time) {
     return setAlarm(id,indx,wake_time);
     }

  /**
   * Sets the Alarm to fire at a specified delay with respect 
   * to the current Alarm time (ie seconds since initialization). 
   * @author herman@cs.uiowa.edu
   * @return Returns SUCCESS if alarm scheduled, otherwise FAIL 
   */
  command result_t Alarm.set[uint8_t id] ( uint8_t indx, 
     uint16_t delay_time) {
     return setAlarm(id,indx,(theClock+delay_time));
     }

  default event result_t Alarm.wakeup[uint8_t id](uint8_t indx,
     uint32_t wake_time) {
     return SUCCESS;
     }

  /**
   * Periodic firing of Timer so Alarm can check for scheduled
   * events (mostly done once per second, unless Alarm gets behind
   * in its scheduling -- then it fires more often).  This event
   * signals possibly one scheduled Alarm.wakeup event, and also
   * calls Timer.start for the next firing. 
   * @author herman@cs.uiowa.edu
   * @return Always returns SUCCESS.
   */
  event result_t TimerMilli.fired() {
     uint8_t i, id, indx;

     // return ASAP if firing rate is more frequent than
     // once per second;  this is tunable in Alarm.h
     if (INTER_RATE < 1024) {
        subClock += INTER_RATE;  // add # milliseconds
        if (subClock < 1024) return SUCCESS;
        subClock = 0;
	}

     dbg(DBG_USR1, "*** Alarm wakeup by Timer @ %d ***\n",theClock);

     id = 255;
     indx = 255;

     if (++theClock < nextCheck) return SUCCESS; 

     // find all candidates to signal
     for (i=0; i<sched_list_size; i++) 
       if ((schedList+i)->wake_time != 0xffffffff &&
           (schedList+i)->wake_time <= theClock) {
	       (schedList+i)->wake_time = 0xffffffff;
	       id = (schedList+i)->id;
	       indx = (schedList+i)->indx;
	       signal Alarm.wakeup[id](indx,theClock);
               }
     reNext();
     return SUCCESS;
     }

}
