%%      A function to gerenate lower bound on coded caching problem
%           Written by Hooshang Ghasemi, Feb. 16.
%
%           The goal is to generate lower bund:
%           alpha * R_star + beta * M >= L
%
%           Note:   if alpha and beta are not given then the code will find
%                   the optimal ones.
%           Note:   if M is not given then the code computes L as output

function R_star = comp_bound(N, K, M, alpha, beta)

switch nargin
    
    case 5          %   Inputs are "N,K, M, alpha, beta". Here output is R_star
        
        if(M > N)
            fprintf(2, '\nError: "M", cache size, must be less than "N", number of files\n');
            return;
        end
        
        if(alpha * min(beta, K) <= N)
            R_star  =   min(beta,K) - beta * M / alpha;
            fprintf('\nThe lower bound is:\n');
            fprintf('R^*(%1.1f) >= %1.1f\n', M, R_star);
        else
            [Nsat, Nsat_l]       =   ComputeNsat(alpha, beta, K);       %   Nsat: saturation number N_sat(alpha,beta,K)
            if(N >= Nsat)                                               %   Nsat_l: saturation number of left branch: max(N_sat(alpha_r,beta_r,K), N_sat(alpha_l,beta_l,K))
                R_star  =   min(beta,K) - beta * M / alpha;
                fprintf('\nThe lower bound is:\n');
                fprintf('R^*(%1.1f) >= %1.1f\n', M, R_star);
            elseif(N < Nsat_l)
                %   In this case problem instance is non atomic.
                fprintf(2, '\nProblem instance associated with given alpha and beta is non-atomic\n');
                fprintf(2, 'For definition of atomic problem instance please refer to the manuscript\n');
                fprintf(2, 'Please try smaller alpha and beta.\n');
                R_star  =   0;
            else
                L       =   (alpha * min(beta, K) - mod(alpha,2)*mod(min(beta,K),2))/2 + N - Nsat_l;
                R_star  =   (min(L, alpha*min(beta,K)) - beta * M) / alpha;
                fprintf('\nThe lower bound is:\n');
                fprintf('R^*(%1.1f) >= %1.1f\n', M, R_star);
            end
        end
        
    case 4          %   Inputs are "N, K, alpha, beta". Here output is L
        beta    =   alpha;
        alpha   =   M;
        if(alpha * min(beta, K) <= N)
            R_star  =   alpha * min(beta, K);
            fprintf('\nThe lower bound is:\n');
            fprintf('%dR^* + %dM >= %1.1f\n', alpha, beta, R_star);
        else
            [Nsat, Nsat_l]       =   ComputeNsat(alpha, beta, K);       %   Nsat: saturation number N_sat(alpha,beta,K)
            if(N >= Nsat)                                               %   Nsat_l: saturation number of left branch: max(N_sat(alpha_r,beta_r,K), N_sat(alpha_l,beta_l,K))
                R_star  =   alpha * min(beta, K);
                fprintf('\nThe lower bound is:\n');
                fprintf('%dR^* + %dM >= %1.1f\n', alpha, beta, R_star);
            elseif(N < Nsat_l)
                %   In this case problem instance is non atomic.
                fprintf(2, '\nProblem instance associated with given alpha and beta is non-atomic\n');
                fprintf(2, 'For definition of atomic problem instance please refer to the manuscript\n');
                fprintf(2, 'Please try smaller alpha and beta.\n');
                R_star  =   0;
            else
                ab       =  (alpha * min(beta, K));
                R_star   =   (ab - mod(alpha,2)*mod(min(beta,K),2))/2 + N - Nsat_l;
                fprintf('\nThe lower bound is:\n');
                fprintf('%dR^* + %dM >= %1.3f\n', alpha, beta, R_star);
            end
        end
        
    case 3          %   Inputs are "N, K, M". Function finds optimal alpha and beta
        
        if(M > N)
            fprintf(2, '\nError: "M", cache size, must be less than "N", number of files\n');
            return;
        end
        %   Here we find optimal alpha and beta and output is R_star
        max_Rate    =   0;
        alpha_opt   =   1;
        beta_opt    =   1;
        for n_alpha = 1 : 2 * N
            min_beta    =   1;
            max_beta    =   2*K;
%             min_beta    =   max(min(K, floor(N/n_alpha/2)), 1);
%             max_beta    =   min(2*K, ceil(6*N/n_alpha));
            for n_beta = min_beta : max_beta
                
                if(n_alpha * min(n_beta, K) <= N)
                    Tmp_R  =   min(n_beta,K) - n_beta * M / n_alpha;
                else
                    [Nsat, Nsat_l]       =   ComputeNsat(n_alpha, n_beta, K);       %   Nsat: saturation number N_sat(alpha,beta,K)
                    if(N >= Nsat)                                               %   Nsat_l: saturation number of left branch: max(N_sat(alpha_r,beta_r,K), N_sat(alpha_l,beta_l,K))
                        Tmp_R  =   min(n_beta,K) - n_beta * M / n_alpha;
                    elseif(N < Nsat_l)
                        %   In this case problem instance is non atomic.
                        Tmp_R  =   0;
                    else
                        L_tmp       =   (n_alpha * min(n_beta, K) - mod(n_alpha,2)*mod(min(n_beta,K),2))/2 + N - Nsat_l;
                        Tmp_R  =   (min(L_tmp, n_alpha*min(n_beta,K)) - n_beta * M) / n_alpha;
                    end
                end
                if(Tmp_R > max_Rate)
                    max_Rate     = Tmp_R;
                    alpha_opt    = n_alpha;
                    beta_opt     = n_beta;
                end
            end
        end
        R_star  =   max_Rate;
        fprintf('\nThe lower bound is:\n');
        fprintf('%dR^* + %dM >= %d\n', alpha_opt, beta_opt, (alpha_opt*R_star+beta_opt*M));
        fprintf('Therefore:\t R^*(%1.3f) >= %1.3f\n', M, R_star);
    otherwise       %   Insufficient input.
        fprintf(2, '\nNot enough input argument. The function need either "N, K, M, alpha, beta" or "N, K, alpha, beta" or "N, K, M" as input.\n');
        
end

