function [w_recovered] = HierarchicalKalmanFilter(yv,Phi_t,sigma2,x0)
T = length(yv);

%% Hierarchical Kalman filter
% -------------------------------------------------------------------------
% Bayesian Subspace Pursuit [SampTA paper] runs as the inference algorithm
% for SBL; extending the state-space model. Temporal information influences
% sparse recovery. HBK tracks the scaled covariance matrix Sig_I.
                   %   Initial variance estimate
                                        %   too low causes instability
eta        = 1e-5;                     %   Convergence threshold 
iterations = 10;                        %   Iteration threshold

% Initialise with atom that best alligns with measurements                
Phi     = Phi_t{1};
[N,M]   = size(Phi);
alpha   = zeros(M,1);                   %   BSP hyper-parameters
s       = zeros(M,1);                   %   s, q values [Tipping&Faul SBL]
q       = zeros(M,1);                   %   The actual expressions used 
                                        %   are in the SampTA paper.
p = M;                        
                        
[v, I]  = max(abs(Phi'*yv{1}));         %   I is the overall support set
alpha_I = abs(1./v);                    %   The overall hyper-parameters

% Tracked Covariance matrix and mean value
Sig_I  = 1./(sigma2*diag(alpha_I) + Phi(:,I)'*Phi(:,I));    
mu_I   = Sig_I*Phi(:,I)'*yv{1};
  
% e_hb_sp     = zeros(1,T);               %   reconstruction error
% e_hb_sp(1)  = mean((y{1}- Phi(:,I)*mu_I).^2);
w_recovered = zeros(p,T);               %   recovered dynamic sparse signal
w_recovered(I,1) = mu_I;

for j=2:T
    %disp(['-----------------------> Iteration: ', num2str(j)]);     
    Phi     = Phi_t{j};
    Phi_I   = Phi(:,I);    
    
    yt      = yv{j} - Phi_I*mu_I;               %   Prediction error
    
    alpha         = zeros(M,1);                 %   BSP hyper-parameters
    [v, That]     = max(abs(Phi'*yt));          %   Support set of residual
    alpha(That)   = abs(1./v);          

    [N,M] = size(Phi);
    it    = 1;
    ns    = [inf];
    eyeN  = eye(N);    

    B     = eyeN + Phi_I*Sig_I*Phi_I'./sigma2;  % Temporal influence 
    Bm1   = inv(B);

%---------------------------------> BSP <----------------------------------
    while 1
        it = it + 1;        
        max_alpha_bb = max(alpha(That));
        
        Phi_That    = Phi(:,That);
        A           = diag(alpha(That));
        Sigma       = inv(sigma2*A + Phi_That'*Bm1*Phi_That);
        
        Cm1         = Bm1 - Bm1*Phi_That*Sigma*Phi_That'*Bm1;
        C           = B   + Phi_That*diag(1./alpha(That))*Phi_That';
        
        ml = sigma2*log(det(C)) + yt'*Cm1*yt;  
        ns = [ns, ml];
        
        if (abs(ns(it)-ns(it-1)) <= eta  || it>iterations)                                                    
            x_p = Sigma*Phi(:,That)'*yt;         
            break;  
        else
            notinThat    = setdiff(1:M,That);
            Phi_ni_That  = Phi(:,notinThat);
            s(notinThat) = diag( Phi_ni_That'*Cm1*Phi_ni_That );
            q(notinThat) = Phi_ni_That'*Cm1*yt;
            for i=1:length(That)     
                Sig_Imi     = Sigma - Sigma(:,i)*Sigma(:,i)'./Sigma(i,i);
                keep        = setdiff(1:length(That),i);
                Sig_Imi     = Sig_Imi(keep,keep);
                Phi_That_mi = Phi(:,That(keep));      
                
                Cm1_Imi     = Bm1-Bm1*Phi_That_mi*Sig_Imi*Phi_That_mi'*Bm1;
                s(That(i))  = Phi(:,That(i))'*Cm1_Imi*Phi(:,That(i));
                q(That(i))  = Phi(:,That(i))'*Cm1_Imi*yt;        
            end
            theta = (q.^2 - s); 
            alpha = (s.^2) ./ (q.^2 - sigma2*s);
            absA  = abs(alpha);
            
            % Estimate number of atoms
            n_tg0 = sum(theta>0);
            if n_tg0 == 0; n_tg0=1; end    
            t = sum( absA(That) <= max_alpha_bb );    
            k = min(n_tg0 + t,length(absA));            

            % Select atoms corresponding to k smallest values of alpha 
            [~, Tadd] = sort(absA,'ascend');
            Tadd      = Tadd(1:k);
            Tacc      = union(That, Tadd); 
            
            % Backtracking
            phi_acc = Phi(:,Tacc);
            Sigma   = inv(sigma2*diag(absA(Tacc)) + phi_acc'*Bm1*phi_acc);
            x_p     = Sigma*phi_acc'*yt;         
                        
            [~,max_i] = sort(abs(x_p),'descend');
            That      = Tacc(max_i(1:k)); 
            x_p       = x_p(max_i(1:k));
            x_p       = x_p(alpha(That)>0);
            That      = That(alpha(That)>0);            
        end
    end 
%------------------------------> END BSP <---------------------------------

%------------------> Remove atoms that should not be in I -----------------
%    disp(['That before checks.: ', num2str(That')]);
    [comm_t, comm_I, ~] = intersect(I,That);
    if ~isempty(comm_t)
        to_remove = comm_t( find( alpha_I(comm_I)<0 & alpha(comm_t)>0) );
        if ~isempty(to_remove)
            % remove atom from I           
            [~,who,~]    = intersect(I,to_remove);                                   
            mu_I(who)    = []; alpha_I(who) = [];
            keep         = setdiff(1:length(I),who);
            Sig_I        = Sig_I(keep,keep);
            I(who)       = []; 
             % remove atom from That so it won't be added below
            [~,who,~]    = intersect(That,to_remove);                       
            keep         = setdiff(1:length(That),who);
            Sigma        = Sigma(keep,keep);            
            That(who)    = [];          
        end
    end  
% ---------------------- Kalman filter steps ------------------------------
    alpha_I  = (alpha(I));
    Sig_I    = Sig_I + diag(1./abs(alpha_I));
         
    Phi_temp = Phi(:,I);
    K        = Sig_I*Phi_temp'*inv(eyeN*sigma2 + Phi_temp*Sig_I*Phi_temp');
    Sig_I    = (eye(length(I)) - K*Phi_temp)*Sig_I;    
    mu_I     = mu_I + K*yt;      
% ---------------------- Add statistics for new atoms ---------------------         
    if ~isempty(That)   
        [comm_I, ~, ~]       = intersect(I,That);
        [diff_I, which_diff] = setdiff(That,comm_I);
        if ~isempty(diff_I)
            I       = [I; diff_I];
            alpha_I = [alpha_I; (alpha(diff_I))];           
            Sig_I   = [Sig_I, zeros(size(Sig_I,1),length(diff_I));...
                       zeros(length(diff_I),size(Sig_I,2)), ...
                       Sigma(which_diff,which_diff) ];  
            mu_I    = [mu_I; x_p(which_diff)];   
        end          
    end
    w_recovered(I,j) = mu_I;
% --------------------- Display some information --------------------------
%     disp(['That after checks..: ', num2str(That')]);  
%     disp('-----------------------');
%     disp('alpha_I              I');
%     disp('-----------------------');
%    disp([num2str([alpha_I, I])]);      
    %e_hb_sp(j) = mean((y{j} - Phi(:,I)*mu_I).^2);    
end