% SP_TIMESTEP_FXN 	Function for updating forwards- and
% backwards-propagating messages between frames of a multi-timestep sparse
% signal reconstruction problem.  The specific messages updated include
% those passed between the signal support indicators, {s_n}, of neighboring
% timesteps, and those of the "active" coefficient amplitude parameters,
% {theta_n}, of neighboring timesteps.  Signal support is assumed to evolve
% according to a Markov chain, while amplitude parameters evolve according
% to a discrete Gauss-Markov process.
%
% SYNTAX:
% [lambda, eta, kappa] = sp_timestep_fxn(Support, Amplitude, Msg)
%
% INPUTS:
% Support          	A structure containing support evolution parameters
%   .pz1            Pr{s_n^(t) = 0 | s_n^(t-1) = 1}.  Can be scalar, or
%                   N-by-1 in dimension.
%   .p1z            Pr{s_n^(t) = 1 | s_n^(t-1) = 0}  Can be scalar, or
%                   N-by-1 in dimension.
%   .lambda     	N-by-1 vector of incoming activity probabilities from
%                   h-factor nodes to s-variable nodes.  Choice of which
%                   lambdas constitute incoming probabilities will depend
%                   on whether time is forwards- or backwards-marching
%   .pi             N-by-1 vector of incoming activity probabilities from
%                   f-factor nodes to s-variable nodes generated by
%                   sp_frame_fxn
% Amplitude     	A structure containing amplitude evolution parameters
%   .alpha          Scalar "innovation rate" of the Gauss-Markov process,
%                   or N-by-1 vector of innovation rates if coefficients
%                   are being treated as non-i.i.d.
%   .rho            Variance of the circular complex driving noise of the
%                   Gauss-Markov process or N-by-1 vector of variances if
%                   coefficients are being treated as non-i.i.d.
%   .eta            N-by-1 vector of incoming means from d-factor nodes.
%                   Which etas constitute incoming means will depend on
%                   whether time is forwards- or backwards-marching
%   .kappa          N-by-1 vector of incoming variances from d-factor
%                   nodes.  Which kappas constitute incoming variances will
%                   depend on whether time is forwards- or
%                   backwards-marching
%   .xi             N-by-1 vector of incoming means from f-factor nodes,
%                   generated by sp_frame_fxn
%   .psi            N-by-1 vector of incoming variances from f-factor
%                   nodes, generated by sp_frame_fxn
%   .eta_0          N-by-1 vector of the means of the amplitude evolution
%                   process, i.e., E[theta(n,t)] = eta_0(n)
% Msg               A structure containing message direction and temporal
%                   location information
%   .direction      A character string equal to: 'forward', for forward
%                   time-marching message passing, or 'backward', for
%                   backward time-marching message passing
%   .terminal       Set equal to 1 if the current timestep is either 0 or
%                   T (i.e. a terminal timestep), or set to 0 otherwise
%
% OUTPUTS:
% lambda            N-by-1 vector of updated activity probabilities for
%                   messages leaving d-factor nodes in the desired
%                   time-marching direction
% eta               N-by-1 vector of updated amplitude means for messages
%                   leaving h-factor nodes in the desired time-marching
%                   direction
% kappa             N-by-1 vector of update amplitude variances for
%                   messages leaving h-factor nodes in the desired
%                   time-marching direction
%

%
% Coded by: Justin Ziniel, The Ohio State Univ.
% E-mail: zinielj@ece.osu.edu
% Last change: 12/29/11
% Change summary: 
%		- Created from sp_timestep_fxn v1.0 (12/29/11; JAZ)
% Version 1.0
%

function [lambda, eta, kappa] = sp_timestep_fxn(Support, Amplitude, Msg)


%% Check to ensure that there are a proper number of inputs

if nargin < 3
    error('sp_timestep_fxn: Function requires 3 input structures')
end

if ~isfield(Support, 'pz1') || ~isfield(Support, 'p1z') || ...
        ~isfield(Support, 'lambda') || ~isfield(Support, 'pi')
    error('sp_timestep_fxn: ''Support'' is missing req''d fields')
end

if numel(Support.pz1) ~= numel(Support.p1z)
    error('sp_timestep_fxn: Dimension mismatch between ''pz1'' and ''p1z''')
end

if ~isfield(Amplitude, 'alpha') || ~isfield(Amplitude, 'rho') || ...
        ~isfield(Amplitude, 'eta') || ~isfield(Amplitude, 'kappa') || ...
        ~isfield(Amplitude, 'xi') || ~isfield(Amplitude, 'psi') || ...
        ~isfield(Amplitude, 'eta_0')
    error('sp_timestep_fxn: ''Amplitude'' is missing req''d fields')
end

if numel(Amplitude.alpha) ~= numel(Amplitude.rho)
    error('sp_timestep_fxn: Dimension mismatch between ''alpha'' and ''rho''')
end

if ~isfield(Msg, 'direction') || ~isfield(Msg, 'terminal')
    error('sp_timestep_fxn: ''Msg'' is missing req''d fields')
end

% Unpack the Support and Amplitude structures
pz1 = Support.pz1; p1z = Support.p1z; lambda = Support.lambda;
pi = Support.pi; alpha = Amplitude.alpha; rho = Amplitude.rho;
eta = Amplitude.eta; kappa = Amplitude.kappa; xi = Amplitude.xi;
psi = Amplitude.psi; eta_0 = Amplitude.eta_0;


%% Update the inter-frame message quantities

if strcmpi(Msg.direction, 'forward')
    % We are computing the forward time-marching message updates
    
    % First update lambda, the parameter for the message from h_n(t+1) to
    % s_n(t+1)
    if Msg.terminal
        disp('sp_timestep_fxn: Warning! No update of messages req''d')
        Output.lambda = NaN*ones(size(lambda));     % Dummy msg
    else
        % Standard update for lambda
        Output.lambda = [p1z.*(1 - lambda).*(1 - pi) + ...
            (1 - pz1).*lambda.*pi] ./ [(1 - lambda).*(1 - pi) + ...
            lambda.*pi];        % Eqn. xx
    end
    
    % Next update eta and kappa, the parameters for the message from
    % d_n(t+1) to theta_n(t+1)
    if Msg.terminal
        % No message update required
        Output.eta = NaN*ones(size(eta));       	% Dummy msg
        Output.kappa = NaN*ones(size(kappa));       % Dummy msg
    else
        % Standard update message
        temp_var = (1./kappa + 1./psi).^(-1);  	% Temporary variance quantity
        Output.eta = (1 - alpha).*temp_var.*((eta./kappa) + (xi./psi)) + ...
            alpha.*eta_0;	% Eqn. (A12)
        Output.kappa = (1 - alpha).^2.*temp_var + alpha.^2.*rho;    % Eqn. (A13)
    end
    
elseif strcmpi(Msg.direction, 'backward')
    % We are computing the backward time-marching message updates
    
    % First update lambda, the parameter for the message from h_n(t) to
    % s_n(t-1)
    if Msg.terminal
        % Special update message for the terminal node
        Output.lambda = [pz1.*(1 - pi) + (1 - pz1).*pi] ./ ...
            [(1 - p1z + pz1).*(1 - pi) + (1 - pz1 + p1z).*pi];    % Eqn. xx
    else
        % Standard update for lambda
        Output.lambda = [pz1.*(1 - lambda).*(1 - pi) + ...
            (1 - pz1).*lambda.*pi] ./ [(1 - p1z + pz1).*(1 - lambda).* ...
            (1 - pi) + (1 - pz1 + p1z).*lambda.*pi];             % Eqn. xx
    end
    
    % Next update eta and kappa, the parameters for the message from d_n(t)
    % to theta_n(t-1)
    if Msg.terminal
        % Special update message for the terminal node
        Output.eta = (1./(1 - alpha)).*(xi - alpha.*eta_0);     % Eqn. xx
        Output.kappa = ((1 - alpha).^-2) .* (alpha.^2.*rho + psi);    % Eqn. xx
    else
        % Standard update message
        temp_var = (1./kappa + 1./psi).^(-1);  	% Temporary variance quantity
        Output.eta = (1./(1 - alpha)).*(temp_var.*((xi./psi) + (eta./kappa)) - ...
            alpha.*eta_0);  % Eqn. xx
        Output.eta(isnan(Output.eta)) = 0;
        Output.kappa = ((1 - alpha).^-2) .* (temp_var + alpha.^2.*rho);     % Eqn. xx
    end
    
else
    error('sp_timestep_fxn: Unrecognized direction in ''Msg.direction''')
end

% Remove final output variables from holding structure
lambda = Output.lambda;
eta = Output.eta;
kappa = Output.kappa;