/**************************************************************
 * Utility functions for managing the packet queue
 *
 * @author: Hongwei zhang
 *                    hzhang@motorola.com, zhangho@cse.ohio-state.edu
 *
 * @modified: August 1, 2005
 *
 **************************************************************/

bool queueSanity();


/**
 * tries to find an unpacked packet to store an upcoming payload element
 *
 * Used in: enqueue()  
 *
 * @param len The length of payload element (in bytes), including PE_OVERHEAD
 *
 * @return Pointer to an unpacked buffer, or QUEUE_SIZE if unable to find one
 *
 */
uint8_t placeHolder(uint8_t len) {
  uint8_t i, ptr;
  ptr = QUEUE_SIZE;

  atomic {
    if (busySpareQHead >= QUEUE_SIZE) { //not spare busy Q
      ptr = QUEUE_SIZE;
      //dbg(DBG_TEMP, "no packing \n");
      goto done;
    }

    i = busySpareQHead;
    while (i != QUEUE_SIZE)
      if (Q[i].length + len > MAX_APS_PAYLOAD_LEN)
	i = Q[i].next;
      else {
	ptr = i;
	dbg("DBG_USR1", "placeHolder: PACKING ...  in Q[%d],   existing Len = %d\n", i, Q[i].length);
	goto done;
      }
    if (i >= QUEUE_SIZE) {
      //dbg(DBG_TEMP, "no packing \n");
      ptr = QUEUE_SIZE;
    }
  done:
    ;
  } //atomic

  return ptr;
} //end of placeHolder



/**
 * Order fully packed packet according to their urgency (via maxLatency)
 *
 * Used in: enqueue()
 *
 * @param ptr Pointer to the packet buffer
 *
 */
void putInOrderFull(uint8_t ptr) {
  int i, prevQ;

  atomic {
    //special case: this is the first packed in queue
    if (busyQHead >= QUEUE_SIZE) {
      busySpareQHead = QUEUE_SIZE;
      busyQHead = busyQTail = ptr;
      Q[ptr].prev = Q[ptr].next = QUEUE_SIZE;
      goto done;
    }

    //Otherwise, find the position
    i = busyQHead;
    while (i != QUEUE_SIZE && i != busySpareQHead)
      if (Q[i].maxLatency <= Q[ptr].maxLatency)
	i = Q[i].next;
      else 
	break;

    //put it in front
    if (i >= QUEUE_SIZE) { //reached the tail
      Q[busyQTail].next = ptr;
      Q[ptr].prev = busyQTail;
      Q[ptr].next = QUEUE_SIZE;
      busyQTail = ptr;
    }
    else if (i == busyQHead) { //at the head
      Q[ptr].prev = QUEUE_SIZE;
      Q[ptr].next = busyQHead;
      Q[busyQHead].prev = ptr;
      busyQHead = ptr;
    }
    else { //in the middle
      prevQ = Q[i].prev;
      Q[prevQ].next = ptr;
      Q[i].prev = ptr;
      Q[ptr].prev = prevQ;
      Q[ptr].next = i;
    }
  done:
    ;
  } //atomic

#ifdef DEBUG_APS_ONLY
  if (!queueSanity()) 
    dbg("DBG_ERROR", "APS ---> putInOrderFull(): QUEUE ERROR\n");
#endif

} //end of putInOrderFull()

/**
 * Order fully packed packet according to their SLatency (SL)
 *                            By Jinhong Xu
 * Used in: enqueue()
 *
 * @param ptr Pointer to the packet buffer
 *
 */

void putInSLOrderFull(uint8_t ptr){
	int i, prevQ;
	
	atomic {
		//special case: this is the first packed in queue
		if (busyQHead >= QUEUE_SIZE) {
			busySpareQHead = QUEUE_SIZE;
			busyQHead = busyQTail = ptr;
			Q[ptr].prev = Q[ptr].next = QUEUE_SIZE;
			goto done;
		}
		
		//Otherwise, find the position
		i = busyQHead;
		while (i != QUEUE_SIZE && i != busySpareQHead)
			if (Q[i].SLatency <= Q[ptr].SLatency)
				i = Q[i].next;
			else 
				break;
		
		//put it in front
		if (i >= QUEUE_SIZE) { //reached the tail
			Q[busyQTail].next = ptr;
			Q[ptr].prev = busyQTail;
			Q[ptr].next = QUEUE_SIZE;
			busyQTail = ptr;
		}
		else if (i == busyQHead) { //at the head
			Q[ptr].prev = QUEUE_SIZE;
			Q[ptr].next = busyQHead;
			Q[busyQHead].prev = ptr;
			busyQHead = ptr;
		}
		else { //in the middle
			prevQ = Q[i].prev;
			Q[prevQ].next = ptr;
			Q[i].prev = ptr;
			Q[ptr].prev = prevQ;
			Q[ptr].next = i;
		}
	done:
		;
	} //atomic
	
#ifdef DEBUG_APS_ONLY
	if (!queueSanity()) 
		dbg("DBG_ERROR", "APS ---> putInOrderFull(): QUEUE ERROR\n");
#endif
  }//putInSLOrderFull


/**
 * Order packets not fully packed according to their occupancy
 *
 * Used in: enqueue()
 *
 * @param ptr Pointer to the packet buffer
 *
 */
void putInOrderNotFull(uint8_t ptr) {
  int i, prevQ;

  atomic {
    //special cases
    if (busyQHead >= QUEUE_SIZE) { //this is the first packed in queue
      busySpareQHead = ptr;
      busyQHead = busyQTail = ptr;
      Q[ptr].prev = Q[ptr].next = QUEUE_SIZE;
      goto done;
    }
    else if (busySpareQHead >= QUEUE_SIZE) { //this is the first busySpare
      busySpareQHead = ptr;
      Q[busyQTail].next = ptr;
      Q[ptr].prev = busyQTail;
      Q[ptr].next = QUEUE_SIZE;
      busyQTail = ptr;
      goto done;
    }

    //Otherwise, find the position
    i = busySpareQHead;
    while (i != QUEUE_SIZE)
      if (Q[i].length >= Q[ptr].length)
	i = Q[i].next;
      else
	break;

    //put it in front of the spotted location
    if (i >= QUEUE_SIZE) { //reached the tail
      Q[busyQTail].next = ptr;
      Q[ptr].prev = busyQTail;
      Q[ptr].next = QUEUE_SIZE;
      busyQTail = ptr;
    }
    else { //otherwise
      if (i == busySpareQHead)
	busySpareQHead = ptr;
      //put in place
      if (i == busyQHead) { //at the head
	Q[ptr].prev = QUEUE_SIZE;
	Q[ptr].next = i;
	Q[i].prev = ptr;
	busyQHead = ptr;
      }
      else { //in the middle
	prevQ = Q[i].prev;
	Q[prevQ].next = ptr;
	Q[i].prev = ptr;
	Q[ptr].prev = prevQ;
	Q[ptr].next = i;
      }
    }
  done:
    ;
  } //atomic

#ifdef DEBUG_APS_ONLY
  if (!queueSanity()) 
    dbg("DBG_ERROR", "APS ---> putInOrderNotFull(): QUEUE ERROR\n");
#endif

} //end of putInOrderNotFull()



/**
 * Try to put an payload element in the packet queue
 *
 * Used in: APSSend.send(), forward()
 *
 * @param id AM message type
 *
 * @param data Pointer to the data of a payload element
 * 
 * @param payloadLen Length of the data
 *
 * @param maxLatency Maximum allowable latency, in milliseconds
 * 
 * @param SLatency  SLatency of
 * @return SUCCESS or FAIL: whether the data has been queued successfully
 *
 */
bool enqueue(uint8_t id, uint8_t * data, uint8_t payloadLen, uint16_t maxLatency, uint16_t SLatency) {
    uint8_t i;
    uint8_t packed, tp, prevQ, nextQ;
    TOS_APSPE * thisPE;
    TOS_APSPayload * thisPayload;
    int thisElementLen;
    uint32_t curTime, thisInterval;
    bool returnStatus = SUCCESS;
	 dbg("DBG_ERROR", " data::  %d\n", data[0]);

   if (qLock == TRUE) {
     returnStatus = FALSE;
     goto done;
   }
   else 
     atomic qLock = TRUE;

    /* for parameter estimation: meanPELen */
    if (meanPELen == 0xff) //firt time to receive a PE
      meanPELen = payloadLen + PE_OVERHEAD;
    else 
      meanPELen = meanPELen - (meanPELen >> compPastW) + ((payloadLen + PE_OVERHEAD) >> compPastW);
    //dbg(DBG_TEMP, "meanPELen = %d\n", meanPELen);

    /* for parameter estimation: reception */
#ifdef TOSSIM
    curTime = (call SysTime.get());//TICKS_PER_MILLISECOND;
    //dbg(DBG_TEMP, "curTime = %d\n", curTime);
    //dbg (DBG_TEMP, "tos_state.tos_time = %lld\n", tos_state.tos_time);
#else
    curTime = (call LocalTime.get());// TICKS_PER_MILLISECOND;
#endif
    //EWMA style estimation
    atomic {
      if (lastRcvTime ==0) {//the first time to receive an packet to process
	lastRcvTime = curTime;
	expRcvSize = payloadLen;
      }
      else {                                //otherwise
	//interval
	thisInterval = curTime - lastRcvTime;
	if (thisInterval < maxPktInterval) {//"reasonable"
	  if (!passedSecondReception) {
	    passedSecondReception = TRUE;
	    expRcvInterval = thisInterval;
	  }
	  else
	    expRcvInterval = (expRcvInterval - (expRcvInterval >> compPastW)) + (thisInterval >> compPastW);	
	}
	lastRcvTime = curTime;
	//size
	expRcvSize = (expRcvSize - (expRcvSize >> compPastW)) + (payloadLen >> compPastW);
      }
    }//atomic


    /* try to find a place to pack the payload, instead of creating a new packet */
#ifndef NO_PACKING_AT_ALL 
    packed = placeHolder(payloadLen + PE_OVERHEAD);
    /*
    if (maxLatency > NO_PACKING_INTERVAL_THRESHOLD) //simulate NO-PACKING 
      packed = QUEUE_SIZE;
      */
#else
    packed = QUEUE_SIZE;
#endif

    /* check whether there is buffer space and whether payload length does not exceed the limit */
    if ((freeQHead >= QUEUE_SIZE && packed >= QUEUE_SIZE) //queue overflow
          || payloadLen > MAX_APS_PE_LEN   //payload element too large
       ) {
      dbg("DBG_ERROR", "queue overflows: numBusyBuffers = %d,   freeQHead = %d,  payloadLen =%d  pending = %d\n", numBusyBuffers, freeQHead, payloadLen, pending);
      atomic apsStats.numQueueOverflows++;
      returnStatus = FAIL;
      goto done;
      //return FAIL;
    }

    /* put in buffer */
    atomic {
      //compose this information element
      if (packed != QUEUE_SIZE)  //to pack
	thisPE = (TOS_APSPE *)(&(Q[packed].msg.data[ALR_OVERHEAD+Q[packed].length]));
      else {                                       //to allocate a new buffer
	thisPE = (TOS_APSPE *)(&(Q[freeQHead].msg.data[ALR_OVERHEAD+PAYLOAD_OVERHEAD]));
      }
      thisPE->type = id;
      thisPE->length = payloadLen;
      thisPE->maxLatency = maxLatency;
	  thisPE->SLatency = SLatency;         //modified by jinhong
      for (i = 0; i < payloadLen; i++){
            if(i==0)
            dbg("info","loop in in enqueue()!");
			thisPE->data[i] = data[i];
			}
		if(i>payloadLen)
			dbg("info","loop out in enqueue()!");
		
      //update the "length" & "maxLatency" fields

	  dbg("DBG_ERROR", "Sending. thisPE->length= %d, thisPE->type=%d, thisPE->maxLatency = %d, thisPE->SLatency = %d, thisPE->data =%x\n", thisPE->length,thisPE->type,thisPE->maxLatency,thisPE->SLatency, thisPE->data[0]);
//#ifndef SL_PACKING      
	  if (maxLatency > 0)
			toDecreaseLatency = TRUE;
/*
#else
	  if (SLatency > 0)
		  toDecreaseSLatency = TRUE;
*/ 
//#endif
	  thisElementLen = PE_OVERHEAD + payloadLen;
      if (packed != QUEUE_SIZE) { //packed
			thisPayload = (TOS_APSPayload *)(&(Q[packed].msg.data[ALR_OVERHEAD]));
	//dbg(DBG_TEMP, "enqueue: thisElementLen = %d, PayloadLength before packing = %d,%d \n", thisElementLen, thisPayload->length, Q[packed].length);
			thisPayload->length += thisElementLen;
	//dbg(DBG_TEMP, "enqueue: PayloadLength after packing = %d \n", thisPayload->length);
	/*
	if (thisPayload->minElementLen > thisElementLen)
	  thisPayload->minElementLen = thisElementLen;
	if (thisPayload->maxLatency > maxLatency)
	  thisPayload->maxLatency = maxLatency;
	*/
		   Q[packed].length += thisElementLen;
#ifndef SL_PACKING
		  if (Q[packed].maxLatency > maxLatency)
			  Q[packed].maxLatency = maxLatency;
#else
		  if (Q[packed].SLatency > SLatency)
			  Q[packed].SLatency = SLatency;
#endif
      }
      else {                                        //allocated a new buffer
		  thisPayload = (TOS_APSPayload *)(&(Q[freeQHead].msg.data[ALR_OVERHEAD]));
		  thisPayload->length = thisElementLen;
	/*
	thisPayload->minElementLen = thisElementLen;
	thisPayload->maxLatency = maxLatency;
	*/
		  Q[freeQHead].length = PAYLOAD_OVERHEAD + thisElementLen;
		  Q[freeQHead].maxLatency = maxLatency;
		  Q[freeQHead].SLatency = SLatency;
	//dbg(DBG_TEMP, "enqueue: new buffer Q[%d], thisElementLen = %d, payloadLen = %d, %d \n", freeQHead, thisElementLen, thisPayload->length, Q[freeQHead].length);
      }
    } //atomic
	
	
    /* update pointers/variables related to queue management */
    if (packed != QUEUE_SIZE) { //packed
      if ((busySpareQHead != packed && Q[Q[packed].prev].length >= Q[packed].length) //stay at its original place
            || (busySpareQHead == packed && ALR_OVERHEAD+Q[packed].length+MIN_PAYLOAD_ELEMENT_LEN <= TOSH_DATA_LENGTH)
          )
					goto done;
      else {
					if (busySpareQHead == packed) { //must already be fully packed now
							 atomic busySpareQHead = Q[packed].next;
							 goto done;
							}
				atomic { //i.e., busySpareQHead != packed
						//get this packet out of the queue
						 prevQ = Q[packed].prev;
						nextQ = Q[packed].next;
								if (nextQ >= QUEUE_SIZE) {
										Q[prevQ].next = QUEUE_SIZE;
										busyQTail = prevQ;
									 }
								else {
										Q[prevQ].next = nextQ;
										Q[nextQ].prev = prevQ;
										}
							//reset pointers of itself
						Q[packed].prev = Q[packed].next = QUEUE_SIZE;
							} //atomic
	//put in the right place in order
					if (ALR_OVERHEAD+Q[packed].length + MIN_PAYLOAD_ELEMENT_LEN > TOSH_DATA_LENGTH) //becomes full
								{
								 
											#ifndef SL_PACKING		  
														putInOrderFull(packed);
											#else 
														 putInSLOrderFull(packed);
											#endif
	
									}
	//#ifdef SIMPLE_PACKING
						else  //not full yet: sort busySpareQ
										 putInOrderNotFull(packed);
	//#endif		
			}
	
    }

    else { //allocated a new buffer
      atomic {
	if (numBusyBuffers < QUEUE_SIZE)
	  numBusyBuffers++;
	else
	  dbg("DBG_ERROR", "error: counting numBusyBuffers (=%d)\n", numBusyBuffers);
	tp = freeQHead;
	//get the buffer out of the freeQ
	if (freeQHead != freeQTail) {
	  freeQHead = Q[freeQHead].next;
	  Q[freeQHead].prev = QUEUE_SIZE;
	}
	else
	  freeQHead = freeQTail = QUEUE_SIZE;
	//reset pointers of itself
	Q[tp].prev = Q[tp].next = QUEUE_SIZE;
      }
      //put in the righ place in order

      if (ALR_OVERHEAD+Q[tp].length + MIN_PAYLOAD_ELEMENT_LEN > TOSH_DATA_LENGTH) //becomes full
				{
											#ifndef SL_PACKING		  
														putInOrderFull(tp);
											#else 
														 putInSLOrderFull(tp);
											#endif
				}
//#ifdef SIMPLE_PACKING     
      else  //not full yet: sort busySpareQ
		  putInOrderNotFull(tp);
//#endif
    }

#ifdef DEBUG_APS_ONLY
    if (!queueSanity()) 
      dbg("DBG_ERROR", "APS ---> enqueue(): QUEUE ERROR\n");
#endif

    //dbg(DBG_TEMP, "busyQHead = %d, Q[%d].length = %d\n", busyQHead, busyQHead, Q[busyQHead].length);
 done:
    atomic qLock = FALSE;

	 /*******************Jinhong Xu******************************************/
               dbg("APS", "busyQHead = %d, Q[%d].length = %d\n", busyQHead, busyQHead, Q[busyQHead].maxLatency);

       /*********************************************************************/
    return returnStatus;
} //end of enqueue()


/**
 * Decrease the maximum allowable MaxLatency for a packet and the information elements in it
 *
 * Used in: Timer.fired(), decreaseAllMaxLatency()
 *
 * @param pkt The pointer to Q[...]
 * @param duration The amount (in milliseconds) of change
 *
 * @return Whether maxLatency is still > 0
 */
bool decreasePktMaxLatency(uint8_t pkt, uint16_t duration) {
  TOS_APSPE * thisPE;
  uint8_t j;
  bool tmp;

  tmp = FALSE;

  //sanity check
  if (pkt >= QUEUE_SIZE) {
    dbg("DBG_ERROR", "error: decreasePktMaxLatency() ---> pkt =%d, and is >= QUEUE_SIZE\n", pkt);
    return FALSE;
  }


  //decrease max. allowable latency
  atomic {
    if (Q[pkt].maxLatency > duration) {
      Q[pkt].maxLatency -= duration;
      tmp = TRUE;
    }
    else 
      Q[pkt].maxLatency = 0;
    //
    j = ALR_OVERHEAD + PAYLOAD_OVERHEAD;
    while (j < Q[pkt].length + ALR_OVERHEAD) {
      thisPE = (TOS_APSPE *)(&(Q[pkt].msg.data[j]));
      if (thisPE->length > MAX_APS_PE_LEN - PE_OVERHEAD || thisPE->length < MIN_PAYLOAD_ELEMENT_LEN - PE_OVERHEAD) {
	dbg("DBG_ERROR", "error: corrupted packet. thisPE->length = %d\n", thisPE->length);
	break;
      }
      if (thisPE->maxLatency > duration) {
	thisPE->maxLatency -= duration;
	tmp = TRUE;
      }
      else 
	thisPE->maxLatency = 0;

      j += (thisPE->length + PE_OVERHEAD);
    }
  }//atomic

#ifdef DEBUG_APS_ONLY
  if (!queueSanity()) 
    dbg("DBG_ERROR", "APS ---> decreasePktMaxLatency(): QUEUE ERROR\n");
#endif

  return tmp;
} //end of decreasePktMaxLatency()



/**
 * Increase the maximum allowable MaxLatency for a packet and the information elements in it
 *
 * Used in: Timer.fired()
 *
 * @param pkt The pointer to Q[...]
 * @param duration The amount (in milliseconds) of change
 */
void increasePktMaxLatency(uint8_t pkt, uint16_t duration) {
  TOS_APSPE * thisPE;
  uint8_t j;

  //sanity check
  if (pkt >= QUEUE_SIZE) {
    dbg("DBG_ERROR", "error: increasePktMaxLatency() ---> pkt =%d, and is >= QUEUE_SIZE\n", pkt);
    return;
  }


  //increase max. allowable latency
  atomic {
    Q[pkt].maxLatency += duration;
    //
    j = ALR_OVERHEAD + PAYLOAD_OVERHEAD;
    while (j < Q[pkt].length+ ALR_OVERHEAD) {
      thisPE = (TOS_APSPE *)(&(Q[pkt].msg.data[j]));
      if (thisPE->length > MAX_APS_PE_LEN - PE_OVERHEAD || thisPE->length < MIN_PAYLOAD_ELEMENT_LEN - PE_OVERHEAD) {
	dbg("DBG_ERROR", "error: corrupted packet. thisPE->length = %d\n", thisPE->length);
	break;
      }
      thisPE->maxLatency += duration;

      j += (thisPE->length + PE_OVERHEAD);
    }
  }//atomic

#ifdef DEBUG_APS_ONLY
  if (!queueSanity()) 
    dbg("DBG_ERROR", "APS ---> increasePktMaxLatency(): QUEUE ERROR\n");
#endif

} //end of increasePktMaxLatency()



/**
 * Decrease the max. allowable latency for all queued packets and information elements
 *
 * Used in: Timer.fired()
 *
 * @param duration Amount of reduction
 */
void decreaseAllMaxLatency(uint16_t duration) {
  uint8_t i;
  bool tmp;

  atomic toDecreaseLatency = FALSE;

  if (busyQHead >= QUEUE_SIZE)
    return;

  i = busyQHead;
  while (i < QUEUE_SIZE) {
    tmp = decreasePktMaxLatency(i, duration);
    atomic {
      if (!toDecreaseLatency && tmp)
	toDecreaseLatency = tmp;
    }

    i = Q[i].next;
  }

}//end of decreaseAllMaxLatency()

/**
 * Decrease the SLfor all queued packets 
 *
 * Used in: Timer.fired()
 *
 * @param duration Amount of reduction
 */
void decreaseAllSLatency(uint16_t duration) {
	uint8_t i;
	if (busyQHead >= QUEUE_SIZE)
		return;
	
	i = busyQHead;
	while (i < QUEUE_SIZE) {
		if (Q[i].SLatency <= duration)
			Q[i].SLatency = 0;
			else
			Q[i].SLatency -= duration;	
		
		i = Q[i].next;
	}
	
}//end of decreaseAllSLatency()



/**
 * Put an urgent but non-fully-packed packet (to be late in delivery) to the head of busyQ 
 * 
 * Used in: Timer.fired()
 *
 * @param ett ALR delivery latency
 */
void pickUrgentPkt(uint32_t ett) {
  uint8_t i, prevQ, nextQ;

  if (qLock == TRUE) {
    goto done;
  }
  else 
    atomic qLock = TRUE;


  if (busySpareQHead >= QUEUE_SIZE)
    goto done;

  atomic {
    //search for the first such non-fully-packed packet
    i = busySpareQHead;
    while (i < QUEUE_SIZE)
      if (ett + APS_TIMER_INTERVAL >= Q[i].maxLatency)
	break;
      else 
	i = Q[i].next;

    //move this to the head of busyQ, if ever found one
    if (i < QUEUE_SIZE) {
      //remove it from its original position
      if (busySpareQHead == i)
	busySpareQHead = Q[i].next;
      if (busyQHead == i) { //is head: actually will not be TRUE at all; for sanity, in case the
       	                         //precondition of this component is not met
	busyQHead = Q[i].next;
	if (busyQHead >= QUEUE_SIZE)
	  busyQTail = QUEUE_SIZE;
	else 
	  Q[busyQHead].prev = QUEUE_SIZE;
      }
      else {                  //otherwise
	prevQ = Q[i].prev;
	nextQ = Q[i].next;
	if (nextQ >= QUEUE_SIZE) {
	  Q[prevQ].next = QUEUE_SIZE;
	  busyQTail = prevQ;
	}
	else {
	  Q[prevQ].next = nextQ;
	  Q[nextQ].prev = prevQ;
	}
      }
      //reset pointers of itself
      Q[i].prev = Q[i].next = QUEUE_SIZE;
      //put it to the head of the queue
      if (busyQHead >= QUEUE_SIZE) //will not hold in normal scenarios
	busyQHead = busyQTail = i;
      else {
	Q[i].next = busyQHead;
	Q[busyQHead].prev = i;
	busyQHead = i;
      }
    }
  } //atomic


#ifdef DEBUG_APS_ONLY
  if (!queueSanity()) 
    dbg("DBG_ERROR", "APS ---> pickUrgentPkt(): QUEUE ERROR\n");
#endif

 done:
  atomic qLock = FALSE;

} //end of pickUrgentPkt()



/**
 * Check the sanity of packet queue: pointers, fields of each queued packet
 *
 * Used in: Timer.fired()
 *
 */
bool queueSanity() {
  uint8_t i, j;
  uint8_t busyBufs, freeBufs;
  TOS_APSPayload * thisPayload;
  TOS_APSPE * thisPE;
  bool sanity;

  sanity = TRUE;

  busyBufs = freeBufs = 0;

  //check busyQ
  if (busyQHead != QUEUE_SIZE) {
    busyBufs = 1;
    //pointers
    if (Q[busyQHead].prev != QUEUE_SIZE || Q[busyQTail].next != QUEUE_SIZE) {
      dbg("DBG_ERROR", "queue error 0: ?????? \n");
      sanity = FALSE;
    }
    j = busyQHead;
    i = Q[j].next;
    while (i != QUEUE_SIZE) {
      busyBufs++;
      if (Q[i].prev != j) {
	dbg("DBG_ERROR", "queue error 1: ?????? \n");
	sanity = FALSE;
      }
      j = i;
      i = Q[i].next;
    }
    if (busyQTail != j) {
      dbg("DBG_ERROR", "queue error 2: ?????? \n");
      sanity = FALSE;
    }
    if (busyBufs != numBusyBuffers) {
      dbg("DBG_ERROR", "queue error 2.1: ????\n");
      sanity = FALSE;
    }
    //content
    i = busyQHead;
    while (i != QUEUE_SIZE) {
      if (i > QUEUE_SIZE) {
	dbg("DBG_ERROR", "queue error 3: ??????\n");
	sanity = FALSE;
      }
      else if (Q[i].length > MAX_APS_PAYLOAD_LEN) {
	dbg("DBG_ERROR", "queue error 4: ??????\n");
	sanity = FALSE;
      }
      thisPayload = (TOS_APSPayload *)(&Q[i].msg.data[ALR_OVERHEAD]);
      if (thisPayload->length + PAYLOAD_OVERHEAD != Q[i].length) {
	dbg("DBG_ERROR", "queue error 5: ?????? \n");
	sanity = FALSE;
      }
      j = ALR_OVERHEAD + PAYLOAD_OVERHEAD;
      while (j < Q[i].length+ ALR_OVERHEAD) {
	thisPE = (TOS_APSPE *)(&Q[i].msg.data[j]);
	if (thisPE->length > MAX_APS_PE_LEN - PE_OVERHEAD || 
                      thisPE->length < MIN_PAYLOAD_ELEMENT_LEN - PE_OVERHEAD
	    ) {
	  dbg("DBG_ERROR", "queue error 5-2: corrupted packet. thisPE->length = %d ??????\n", thisPE->length);
	  sanity = FALSE;
	}
	j += (thisPE->length + PE_OVERHEAD);
      }
      if (j != Q[i].length + ALR_OVERHEAD) {
	dbg("DBG_ERROR", "queue error 6: (j = %d, Q[i].length) = %d ?????? \n", j, Q[i].length);
	sanity = FALSE;
      }

      i = Q[i].next;
    }
  }

  //check freeQ
  if (freeQHead != QUEUE_SIZE) {
    freeBufs = 1;
    //pointers
    if (Q[freeQHead].prev != QUEUE_SIZE || Q[freeQTail].next != QUEUE_SIZE) {
      dbg("DBG_ERROR", "queue error 7: freeQHead = %d, freeQTail = %d, Q[%d].prev = %d, Q[%d].next = %d ?????? \n", freeQHead, freeQTail, freeQHead, Q[freeQHead].prev, freeQTail, Q[freeQTail].next);
      sanity = FALSE;
    }
    j = freeQHead;
    i = Q[j].next;
    while (i != QUEUE_SIZE) {
      freeBufs++;
      if (Q[i].prev != j) {
	dbg("DBG_ERROR", "queue error 8: ?????? \n");
	sanity = FALSE;
      }
      j = i;
      i = Q[i].next;
    }
    if (freeQTail != j) {
      dbg("DBG_ERROR", "queue error 9: ?????? \n");
      sanity = FALSE;
    }
    //content
    i = freeQHead;
    while (i != QUEUE_SIZE) {
      if (Q[i].length != 0) {
	dbg("DBG_ERROR", "queue error 10: i = %d, Q[i].length = %d ?????? \n", i, Q[i].length);
	sanity = FALSE;
      }

      i = Q[i].next;
    }
  }

  return sanity;
} //end of queueSanity()
