/**
 * Copyright (c) 2003 - The Ohio State University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs, and the author attribution appear in all copies of this
 * software.
 *
 * IN NO EVENT SHALL THE OHIO STATE UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE OHIO STATE
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE OHIO STATE UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE OHIO STATE UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */


//-----  [START] -----
//
//MODULE:                                             TestReliableCommM.nc
//KEY WORDS DESCRIPTION:       implementation for TestReliableComm
//FUNCTIONAL DESCRIPTION:    test file for ReliableComm
//AUTHOR:                                             Hongwei Zhang
//AUTHOR CONTACT INFO:          zhangho@cse.ohio-state.edu
//LICENSING:                                         non-profit use only
//
//-----  [END] -----


includes AM; 
includes ReliableComm; 

#ifndef DEST_STATION_ADDR 
#define DEST_STATION_ADDR 0          //the ID of the base station
#define RELAY_STATION_ADDR 1      //the ID of the relaying node, who always forwards received packets to the base station
#define ORIGIN_STATION_ADDR0 2   //lower bound on sender id
#define ORIGIN_STATION_ADDR1 6   //upper bound on sender id
#define ONE_HOP 1                                      //send packets to the base station or the replaying node, depending on whether
                                                                             //this constant is defined or not

#define SOURCE_FREQUENCY 100       //inter-packet interval; milliseconds 


#define TIME_SCALE 1
#define DATA_LENGTH 3
#define DeadThreshold 3
#endif

module TestReliableCommM {
  provides interface StdControl;
  uses {
    interface StdControl as SubControl;
    //"to use ReliableComm for UART communication": interface StdControl as SubControl2;

    interface ReliableCommControl;
    interface ReliableSendMsg as ReliableSend;
    interface ReliableReceiveMsg as ReliableReceive;

    //interface SendMsg;

    interface Timer;
    interface Leds;
  }
}

implementation {

  TOS_Msg testMsg;
  uint8_t enq_nxt;
  uint8_t counter;

  bool pending;
  uint16_t pendingDeadCount;

  command result_t StdControl.init() {
    enq_nxt = 0;
    pendingDeadCount = 0;
    pending = FALSE;
    //return rcombine3(call SubControl.init(), call SubControl2.init(), call Leds.init()); 
    return rcombine(call SubControl.init(), call Leds.init());
  }

  command result_t StdControl.start() {
    if (TOS_LOCAL_ADDRESS == DEST_STATION_ADDR) {
      call ReliableCommControl.setSnooping(FALSE);
#ifndef EXPT_ONLY
      call ReliableCommControl.setBase(TRUE);
#endif
    }
    else if (TOS_LOCAL_ADDRESS == RELAY_STATION_ADDR)
#ifndef EXPT_ONLY
      call ReliableCommControl.setBaseChildren(TRUE);
#endif
#ifdef ONE_HOP
#ifndef EXPT_ONLY
    if (TOS_LOCAL_ADDRESS != DEST_STATION_ADDR)
      call ReliableCommControl.setBaseChildren(TRUE);
#endif
#endif

     call Leds.greenOff();
     call Leds.redOff();
     call Leds.yellowOff();
     call Timer.start(TIMER_REPEAT, SOURCE_FREQUENCY);
     //return rcombine(call SubControl.start(), call SubControl2.start()); 
     return (call SubControl.start());
  }

  command result_t StdControl.stop() {
    //return rcombine(call SubControl.stop(), call SubControl2.stop());
    return (call SubControl.stop());
  }

  event result_t Timer.fired() 
  {    
    atomic{
	enq_nxt++;
	enq_nxt %= SEND_QUEUE_SIZE;
	counter++;
	testMsg.data[0] = TOS_LOCAL_ADDRESS;
	testMsg.data[1] = TOS_LOCAL_ADDRESS>>8;
	testMsg.data[2] = enq_nxt;
	testMsg.data[3] = testMsg.data[4] = 0xff;
      }
    //dbg(DBG_SWITCH, "timer fired at node %d \n", TOS_LOCAL_ADDRESS); 

    if ((TOS_LOCAL_ADDRESS >= ORIGIN_STATION_ADDR0 &&
           TOS_LOCAL_ADDRESS <= ORIGIN_STATION_ADDR1
         ) && 
        counter%TIME_SCALE == 0) {
      dbg(DBG_SWITCH, "Is going to send packet # %d\n", enq_nxt);
      call Leds.greenToggle();
#ifdef ONE_HOP
      call ReliableSend.send(DEST_STATION_ADDR, DATA_LENGTH, &testMsg);
#else
      call ReliableSend.send(RELAY_STATION_ADDR, DATA_LENGTH, &testMsg);
#endif
    }

    return SUCCESS;
  } //end of Timer.fired()

  event result_t ReliableSend.sendDone(TOS_MsgPtr msg, result_t success){
    //dbg(DBG_SWITCH, "ReliableSendMsg.sendDone at node %d \n", TOS_LOCAL_ADDRESS);

    ///*debug
    if (success)
      call Leds.yellowToggle();
    else
      call Leds.redToggle();
    //*/

    return SUCCESS;
  }

  /* -----------------*/
  event TOS_MsgPtr ReliableReceive.receive(TOS_MsgPtr m) {
    uint8_t i;

    call Leds.greenToggle();

    for (i=0; i < (m->length + sizeof(ReliableComm_ProtocolMsg)); i++)
      testMsg.data[i] = m->data[i];

    //for the base station
    if (TOS_LOCAL_ADDRESS == DEST_STATION_ADDR) {
      dbg(DBG_SWITCH, "receive packet # %d\n", testMsg.data[0]);
      call ReliableSend.send(TOS_UART_ADDR, m->length, &testMsg);

      /* use ReliableComm for UART communication
      if (!pending) {
        pending = TRUE;
        if (!(call SendMsg.send(TOS_UART_ADDR, m->length + sizeof(ReliableComm_ProtocolMsg), &testMsg))) {
          call Leds.redToggle();
          pending = FALSE;
        }
      }
      else {
	call Leds.redToggle();
	pendingDeadCount++;
      }

      if (pendingDeadCount > DeadThreshold)
	pending = FALSE;
      */
    }
    else //is a relay
      call ReliableSend.send(DEST_STATION_ADDR, m->length, m);

    return m;
  }

  /* to use ReliableComm for UART comm.
 event  result_t SendMsg.sendDone(TOS_MsgPtr msg, result_t success)
  {
    atomic{
      pendingDeadCount = 0;
      pending = FALSE;
    }

    if (success) 
      call Leds.yellowToggle();
    else 
      call Leds.redToggle();

    return SUCCESS;
  }
  */

}
