/**
 * Copyright (c) 2004 - The Ohio State University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs, and the author attribution appear in all copies of this
 * software.
 *
 * IN NO EVENT SHALL THE OHIO STATE UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE OHIO STATE
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE OHIO STATE UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE OHIO STATE UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */


//-----  [START] -----
//
//MODULE:                                             ReliableComm.h
//KEY WORDS DESCRIPTION:       header file for ReliableComm
//FUNCTIONAL DESCRIPTION:    define constants used in ReliableComm
//AUTHOR:                                             Hongwei Zhang
//AUTHOR CONTACT INFO:          zhangho@cse.ohio-state.edu
//LICENSING:                                         non-profit use only
//
//-----  [END] -----



#ifndef _RELIABLE_COMM_H
#define _RELIABLE_COMM_H  1


/*********************************************************
 ***                                  Configurable parameters                                   ***
 ********************************************************/
#define SEND_QUEUE_SIZE 10    //1) check whether L1 and L2 are set appropriately whenever 
                                                                  //     changing this parameter;  
                                                                  //2)  should be no less than NUM_BASE_ACK_BUFFERS for
                                                                  //      a base station; should be no more than 127
                                                                  //3) this parameter affects RAM consumption: the larger the 
                                                                  //     parameter is, the more RAM is required (for buffer space). 

#define L1 4                                          //threshold for "starting congestion control"
#define L2 2                                          //threshold for "surely congested"; L2 <= L1

#define NUM_BASE_ACK_BUFFERS 8     //1) Number of buffers for ACK packets from the base station, and
                                                                                   //     should be no greater than SEND_QUEUE_SIZE; 
                                                                                  //2) Then, (SEND_QUEUE_SIZE - NUM_BASE_ACK_BUFFERS)  buffers
                                                                                  //     are used for UART communication at the base station

#define firstMaxTransmitCount 3                  //the default maximum number of transmissions at each hop
#define MAX_TRANSMIT_COUNT 3        //no more than (32-CONTENTION_TOTAL_WEITGHT_part-1) 
#define DefaultTransmissionPower 255        //default transmission power level; the range is 1 - 255

#define MAX_NUM_CHILDREN  5             //Maximum number of children a node could have


#define TOSSIM_SYSTIME 1                      //to be commented out in deployment; only for 
                                                                               //debugging in TOSSIM and in TestReliableComm


/*********************************************************
 ***                          Non-configurable parameters                                   ***
 ***                    unless you know the implementation well                   ***
 *********************************************************/
#define Timer_Interval                 128                  //milliseconds

//#define EXPT_ONLY 1    //!!!to comment out in the demo integration

#define CONGESTION_CONTROL 1
#define CONTENTION_CONTROL 1
#define RESEND_CONTROL 1

//v1.2

#define PLAN_TO_RESET_RESEND 5  //if the # of fresh packets is less than PLAN_TO_RESET_RESEND, monitor the 
                                                                             //state of the lastDest so that resetting .resend when necessary 

//v1.3
#define USE_PER_PACKET_SEND_TIME 1 // not use mtts (instead use per-packet-send-time) for
                                                                                      //  contention control
#ifdef USE_PER_PACKET_SEND_TIME
#define DEFAULT_PER_PACKET_TIME  36  //milliseconds; consider 29-byte payload and 16-byte on average initial backoff
                                                                                         //for MICA2
#endif

//v1.4
//#define NOT_USE_CONGESTION_CONTENTION_WAIT_THRESHOLD 1 
//#define NOT_USE_RESENT_CHANNEL_UTIL_CONTROL 1 

//v1.5 
#define USE_perPacketTimeDev 1   //in calculating contentionWait 
#define compDEV_RATIO 2 //perPacketTimeDev initialized to perPacketTime>>compDEV_RATIO
#define USE_perPacketTime_for_ChannelUtil 1

//v1.6
#define MULTI_HOP_CONTENTION_WAIT 1   //double the contention wait, considering multi-hop scheduling 
                                                                                               //(to ameliorate hidden-terminal effect)
#define NORMAL_RESEND 1     //.resend does not have to be calculated in a conservative manner, 
                                                               //since we have "?effective?" contention-control 

//v1.7
#define NOT_USE_ChannelUtilGuard_FOR_RESEND_CONTROL 1 //to make contention control work for retransmissions

//v1.8
#define CompConsrvContentionWait 2  //perPackettime&Dev<<compConsrvContentionWait: * 4
#ifdef EXPT_ONLY
#define MAX_CONSRV 5
#endif

//for tossim
#define USE_MacControl 1 //to use interface MacControl 
#ifdef TOSSIM_SYSTIME
#define DBG_SWITCH DBG_USR1
#define DBG_SWITCH2 DBG_USR1
#define DBG_SWITCH3 DBG_USR1
//#define INTEGRITY_CHECKING 1 //to be removed in deployment; only for checking the integrity of the multi-tier queuing
#endif
//#define SIMPLE_ACK 1   //use simple explicit ack: send an ack upong each packet reception; not using MAC-layer ACK feedback
#ifdef SIMPLE_ACK 
#define TIME_OUT 450 //milliseconds
#endif
//#define EXPLICIT_ACK 1 //to use the explicit ack at the MAC layer as an indicator of whether a packet has been received
//#define LOG_STATE 1 //to be uncommented if write log state into flash
//#define CALCULATE_LOG 1 //whether to calculate logging information
//#define REPORT_LOG_WHILE_ALIVE 1

#define INIT_PARENT_SPACE (SEND_QUEUE_SIZE>>1)
#define STABILIZE_PARENT_SPACE 2
#define PARENT_TO_SEND_DEV 2 
//#define CONTENTION_TOTAL_WEITGHT_part 11 //actual total weight is exp(2, deFactoMaxTransmitCount+CONTENTION_TOTAL_WEITGHT_sub)
#define channelUtilizationGuard 3  //about 1.17% probability not hearing packets in channel 
//#define consrvFactor 2 //??? to be randomized
#define compConsrvFactor 1 //<<compConsrvFactor
//#define pastW 7/8
#define compPastW 3 //>>compPastW: should be less than or equal to 5
//#define devPastW 3/4
#define compDevPastW 2 //>>compDevPastW: 
//#define DEV_WEIGHT 4
#define compDevWeight 2 //<<compDevWeight
#define MTTE 180   //millisecond
#define MTTS 90   //millisecond
#define MTTE_DEV 64  //millisecond
#define MTTS_DEV 32  //millisecond

/* For LiTes
//#define JIFFIES_PER_MILLISECOND  32
#define compJIFFIES_PER_MILLISECOND 5 //exp(2, 5)
*/
//For echelon: mtts, mtte, etc, are in terms of milliseconds
#define compJIFFIES_PER_MILLISECOND 0 //each unit is a millisecond
//#define SYSTIME_UNITS_PER_MILLISECOND 921.6
#define compSYSTIME_UNITS_PER_MILLISECOND 10 //approx. expt(2, 10)

#define HANDLER_ID_LOWER_BOUND    80
#define HANDLER_ID_UPPER_BOUND   90

#define BASE_STATION_ID  0  //to be commented out for echelon
#define BASE_ACK_DEST_ID 0xfffe 
#define UART_HANDLER_ID (HANDLER_ID_UPPER_BOUND+1)
#define NULL8 0xff
#define NULL16 0xffff
#define NULL32 0xffffffff

typedef struct {
  uint16_t myAddr;
#ifndef SIMPLE_ACK
  uint8_t myPos;
  uint8_t mySeq;
#ifndef EXPLICIT_ACK
  uint8_t np0; //next "sent" queue postion
  uint8_t np1; //next "to be queued" position

  uint16_t frAddr; //the last forwarder/sender of the packet
  uint8_t frPos;
  uint8_t frSeq;

  uint8_t cumulLeft; //for block ack
  uint8_t cumulRight; 
#endif //EXPLICIT_ACK

  uint8_t emptyElements; //# of empty queue positions
  uint8_t cvq; //the highest ranked virtual queue having at least one packet
  uint8_t cvqSize; //size the VQ[cvq]
  uint16_t mtte; //mean time to empty a queue position
  uint16_t mtts; //mean time to send out a packet to air
#endif //SIMPLE_ACK
} ReliableComm_ProtocolMsg; 
//Note: the size of ReliableComm_Control is 19 bytes; in LiTes, routing payload was 10-14 bytes; thus, need to modify TOSH_DATA_LENGTH (defined in AM.h) to 28-32. 

#ifndef EXPLICIT_ACK
#define aggregatedACK 4 //note: we should guarantee that TOSH_DATA_LENGTH >= aggregatedACK * sizeof(Base_Ack_Msg)
#define baseAckWait 0
typedef struct {
  uint16_t frAddr;
#ifndef SIMPLE_ACK
  uint8_t frPos;
  uint8_t frSeq;
  uint8_t cumulLeft; 
  uint8_t cumulRight; 
#endif
} Base_Ack_Msg;  //for aggregated ack from base stations
#endif

//for self-stabilization
#define MAX_BASE_ACK_DEAD_COUNT  3 //# of times
#define NonBasePendingDeadThreshold  1000 //millisecond
#define MIN_MTTE 46 //millisecond (or 46)
#define MAX_MTTE_THRESHOLD 1600    //millisecond 
#define MIN_MTTS 24 //millisecond (or 23)
#define MAX_MTTS_THRESHOLD 800     //millisecond 
#define CONGESTION_WAIT_THRESHOLD ((L1 * MAX_MTTE_THRESHOLD) <<(compConsrvFactor+1))
#define CONTENTION_WAIT_THRESHOLD ((channelUtilizationGuard * MAX_MTTS_THRESHOLD) <<(compConsrvFactor+1))

//for tuning parameters
#define ReliableComm_Tuning_Handler 150
#define ReliableComm_Tuning_Addr 286 
typedef struct {
  uint8_t transmissionPower; 
  uint8_t maxTransmitCount;
#ifdef EXPT_ONLY
  uint8_t compConsrvContWaiting;
  uint8_t compChannelUtilControl;
  uint8_t compResendConsrv;
#endif
} ReliableComm_Tuning_Msg; 

//for state logging
#define ReliableComm_Log_Handler 160
#define ReliableComm_Log_Addr 296
#define Min_Log_Interval 30000  //milliseconds
#define MAX_NUM_PARENTS  3
#define LOG_LENGTH 3
#define MAX_WRITING_LOG_DEAD_COUNT 10
#define FIRST_LOG_SECTION 32
#define SECOND_LOG_SECTION 30
#define LOG_RECORD_SIZE 62
typedef struct {
  uint8_t type1; //1: marker1

  //routing structure
  uint8_t myID;
  //uint8_t myParentID;

  //queuing condition
  uint8_t queueLength;
  uint16_t queueOverflowCount;
  uint16_t otherQueueOverflowCount;

 //sending 
  uint16_t totalSendsCalled;  //for ReliableSendMsg.send(...)
  uint16_t totalPacketsSent; 
  uint16_t twiceTried;  //at least tried twice 
  uint16_t tripleTried;  //at least three times
  uint16_t delayedACK; 
  //uint16_t maxRetransmitFailureCount;  //???
  //uint16_t otherMaxRetransmitFailureCount; //???
  struct link_sending {
    uint8_t node_id;
    uint16_t totalSendsCalled;
    uint16_t totalPacketsSent;
  } linkSending[MAX_NUM_PARENTS];

  uint8_t type2; //2: marker2

  //receiving
  uint16_t totalPacketsReceived;
  uint16_t receivedDuplicates;
  struct link_reception {
    uint8_t node_id;
    uint16_t totalPacketsReceived;
    uint16_t receivedDuplicates;
  } linkReception[MAX_NUM_CHILDREN];

  /*
  uint8_t transmissionPower; 
  uint8_t maxRetrasmitCount; 
  uint16_t baseRetransmitWaiting;
  uint16_t randomWaitingRange; 
  uint16_t additiveFlowControlWaiting;
  */
} ReliableComm_Reflector; 

#endif
