/* *
 * @ author:    Xiaohui Liu (whulxh@gmail.com) 
 * @ updated:   12/31/12 12:10 PM
 * @ description: migrate to CC2420X stack for time sync to work

 * packet format: from back to front
 * header: ActiveMessageLayerC, TinyosNetworkLayerC, Ieee154PacketLayerC, CC2420XDriverLayerC
	nxle_uint8_t length;
	
	nxle_uint16_t fcf;
	nxle_uint8_t dsn;
	nxle_uint16_t destpan;
	nxle_uint16_t dest;
	nxle_uint16_t src;
	
	// CC2420 802.15.4 header ends here
	nxle_uint8_t network;
	
	nxle_uint8_t type;

 * meta:  CC2420XDriverLayerC, MetadataFlagsLayerC, TimestampingLayerC
 */
#include <Ieee154PacketLayer.h>
#include <TinyosNetworkLayer.h>
#include <TimeStampingLayer.h>
#include <MetadataFlagsLayer.h>
#include <ActiveMessageLayer.h>
#include <AsyncCC2420Transceiver.h>
#include <CC2420XRadio.h>
#include <RadioConfig.h>

#include <Tasklet.h>

module AsyncCC2420TransceiverP {
	provides {
		interface Init;
		interface SplitControl;
		
		interface AsyncAMSend as AMSend[am_id_t id];
		interface AsyncReceive as Receive[am_id_t id];
		interface AsyncReceive as Snoop[am_id_t id];
		interface AsyncAMPacket as AMPacket;
		interface AsyncPacket as Packet;
		// internal use
		interface RadioPacket;
		interface CC2420XDriverConfig;
		interface PacketTimeStamp<TRadio, uint32_t> as PacketTimeStampRadio;
		interface PacketFlag[uint8_t bit];
		interface CC2420Packet;
		
		interface SoftwareAckConfig;
	}
	
	uses {
		interface RadioSend as SubSend;
		interface RadioReceive as SubReceive;
		interface RadioPacket as SubPacket;
		interface RadioState;
		interface Ieee154PacketLayer;
		interface ActiveMessageAddress;
		// hold the resource and never release
		interface Resource;
		
		interface PacketField<uint8_t> as PacketRSSI;
		interface PacketField<uint16_t> as PacketRSSIIdle;
		interface PacketFlag as TimeStampFlag;

		interface UartLog;
	}
}

implementation {
uint8_t seqno;
message_t *p_msg;

command error_t Init.init() {
	seqno = TOS_NODE_ID;
	if ((call Resource.immediateRequest()) != SUCCESS)
		call Resource.request();
	return SUCCESS;
}
event void Resource.granted() {
}

// AM header
activemessage_header_t* getHeader(message_t* msg) {
//	return ((void*)msg) + call SubPacket.headerLength(msg);
	return ((void*)msg) + call SubPacket.headerLength(msg) + sizeof(ieee154_simple_header_t) + sizeof(network_header_t);
}

network_header_t* getNetworkHeader(message_t* msg) {
//	return ((void*)msg) + call SubPacket.headerLength(msg);
	return ((void*)msg) + call SubPacket.headerLength(msg) + sizeof(ieee154_simple_header_t);
}

//void* getPayload(message_t* msg) {
//	return ((void*)msg) + call RadioPacket.headerLength(msg);
//}


//--------------------------------------------------------------------------------------
// SplitControl interface
//--------------------------------------------------------------------------------------
bool is_on = TRUE;

task void startTask() {
	signal SplitControl.startDone(SUCCESS);
}
task void stopTask() {
	signal SplitControl.stopDone(SUCCESS);
}

// fan out at IMACForwarderP as well; does not have to differentiate RadioState.Done() event signalled there bcoz modified RadioState.setChannel() does not signal it
tasklet_async event void RadioState.done() {
atomic {
	if (is_on) {
		post startTask();
	} else {
		post stopTask();
	}
}
}

command error_t SplitControl.start() {
	atomic is_on = TRUE;
	return call RadioState.turnOn();
}

command error_t SplitControl.stop() {
	atomic is_on = FALSE;
	return call RadioState.turnOff();
}


//--------------------------------------------------------------------------------------
// AMSend interface
//--------------------------------------------------------------------------------------
uint16_t tx_cnt = 0;
bool pending = FALSE;
async command error_t AMSend.send[am_id_t id](am_addr_t addr, message_t* msg, uint8_t len) {
	error_t ret;
	// need this protection; otherwise ongoing packet can be overwritten and wrong packet is signalled Done
atomic {
	if (pending) {
		return EBUSY;
	}
}	
	if (len > call Packet.maxPayloadLength())
		return ESIZE;
	call Packet.setPayloadLength(msg, len);
//	if( call Config.checkFrame(msg) != SUCCESS )
//		return FAIL;
	if (!call Ieee154PacketLayer.isDataFrame(msg))
		call Ieee154PacketLayer.createDataFrame(msg);

	atomic call Ieee154PacketLayer.setDSN(msg, seqno);
	call AMPacket.setGroup(msg, call AMPacket.localGroup());
	call AMPacket.setDestination(msg, addr);
	call AMPacket.setSource(msg, call AMPacket.address());
	getNetworkHeader(msg)->network = TINYOS_6LOWPAN_NETWORK_ID;
	call AMPacket.setType(msg, id);

	atomic p_msg = msg;
	ret = call SubSend.send(msg);
	if (SUCCESS == ret) {
		atomic pending = TRUE;
	}
	return ret;
}

tasklet_async event void SubSend.ready() {}

async event void SubSend.sendDone(error_t error) {
	message_t *p_msg_;
	atomic {
		pending = FALSE;
		p_msg_ = p_msg;
	}
//	call UartLog.logEntry(DBG_FLAG, call AMPacket.type(p_msg), __LINE__, tx_cnt);
	signal AMSend.sendDone[call AMPacket.type(p_msg_)](p_msg_, error);
}

inline async command error_t AMSend.cancel[am_id_t id](message_t* msg) {
	// TODO: dummy cancel
	return SUCCESS;
}

inline async command uint8_t AMSend.maxPayloadLength[am_id_t id]() {
	return call Packet.maxPayloadLength();
}

inline async command void* AMSend.getPayload[am_id_t id](message_t* msg, uint8_t len) {
	return call Packet.getPayload(msg, len);
}

default async event void AMSend.sendDone[am_id_t id](message_t *msg, error_t error) {
}

//--------------------------------------------------------------------------------------
// Receive interface
//--------------------------------------------------------------------------------------
tasklet_async event bool SubReceive.header(message_t *msg) {
	return TRUE;
}

tasklet_async event message_t* SubReceive.receive(message_t* msg) {
	am_id_t id = call AMPacket.type(msg);
	uint8_t len = call Packet.payloadLength(msg);
	//void* payload = getPayload(msg);
	void *payload = call Packet.getPayload(msg, len);
	
	// address recognition disabled by default
	if (call AMPacket.isForMe(msg)) {
		return signal Receive.receive[id](msg, payload, len);
	} else {
		//call UartLog.logEntry(DBG_FLAG, DBG_LOSS_FLAG, __LINE__, id);
		return signal Snoop.receive[id](msg, payload, len);
	}
}

default async event message_t *Receive.receive[am_id_t id](message_t *msg, void *payload, uint8_t len) {
	return msg;
}
default async event message_t *Snoop.receive[am_id_t id](message_t *msg, void *payload, uint8_t len) {
	return msg;
}

//--------------------------------------------------------------------------------------
// AMPacket interface
//--------------------------------------------------------------------------------------
inline async command am_addr_t AMPacket.address() {
	return call ActiveMessageAddress.amAddress();
}

inline async command am_group_t AMPacket.localGroup() {
	return call ActiveMessageAddress.amGroup();
}

inline async command bool AMPacket.isForMe(message_t* msg) {
	am_addr_t addr = call AMPacket.destination(msg);
	return addr == call AMPacket.address() || addr == AM_BROADCAST_ADDR;
}

inline async command am_addr_t AMPacket.destination(message_t* msg) {
	//return call Config.destination(msg);
	return call Ieee154PacketLayer.getDestAddr(msg);
}

inline async command void AMPacket.setDestination(message_t* msg, am_addr_t addr) {
//	call Config.setDestination(msg, addr);
	call Ieee154PacketLayer.setDestAddr(msg, addr);
}

inline async command am_addr_t AMPacket.source(message_t* msg) {
//	return call Config.source(msg);
	return call Ieee154PacketLayer.getSrcAddr(msg);
}

inline async command void AMPacket.setSource(message_t* msg, am_addr_t addr) {
//	call Config.setSource(msg, addr);
	call Ieee154PacketLayer.setSrcAddr(msg, addr);
}

inline async command am_id_t AMPacket.type(message_t* msg) {
	return getHeader(msg)->type;
}

inline async command void AMPacket.setType(message_t* msg, am_id_t type) {
	getHeader(msg)->type = type;
}

inline async command am_group_t AMPacket.group(message_t* msg)  {
//	return call Config.group(msg);
	return call Ieee154PacketLayer.getDestPan(msg);
}

inline async command void AMPacket.setGroup(message_t* msg, am_group_t grp) {
//	call Config.setGroup(msg, grp);
	call Ieee154PacketLayer.setDestPan(msg, grp);
}

inline async event void ActiveMessageAddress.changed() {
}

//--------------------------------------------------------------------------------------
// RadioPacket interface
// TODO: to be skipped later
//--------------------------------------------------------------------------------------
async command uint8_t RadioPacket.headerLength(message_t* msg) {
//	return call SubPacket.headerLength(msg) + sizeof(activemessage_header_t);
	return call SubPacket.headerLength(msg) + sizeof(activemessage_header_t) + sizeof(network_header_t) + sizeof(ieee154_simple_header_t);
}

async command uint8_t RadioPacket.payloadLength(message_t* msg) {
//	return call SubPacket.payloadLength(msg) - sizeof(activemessage_header_t);
	return call SubPacket.payloadLength(msg) - sizeof(activemessage_header_t) - sizeof(network_header_t) - sizeof(ieee154_simple_header_t);
}

async command void RadioPacket.setPayloadLength(message_t* msg, uint8_t length) {
//	call SubPacket.setPayloadLength(msg, length + sizeof(activemessage_header_t));
	call SubPacket.setPayloadLength(msg, length + sizeof(activemessage_header_t) + sizeof(network_header_t) + sizeof(ieee154_simple_header_t));
}

async command uint8_t RadioPacket.maxPayloadLength() {
//	return call SubPacket.maxPayloadLength() - sizeof(activemessage_header_t);
	return call SubPacket.maxPayloadLength() - sizeof(activemessage_header_t) - sizeof(network_header_t) - sizeof(ieee154_simple_header_t);
}

async command uint8_t RadioPacket.metadataLength(message_t* msg) {
//	return call SubPacket.metadataLength(msg);
	return call SubPacket.metadataLength(msg) + sizeof(timestamp_metadata_t) + sizeof(flags_metadata_t);
}

// get flags meta
flags_metadata_t* getMeta(message_t* msg) {
	return ((void*)msg) + sizeof(message_t) - call SubPacket.metadataLength(msg) - sizeof(flags_metadata_t);
}

async command void RadioPacket.clear(message_t* msg) {
	call Ieee154PacketLayer.createDataFrame(msg);
	getMeta(msg)->flags = 0;
	call SubPacket.clear(msg);
}

//--------------------------------------------------------------------------------------
// Packet interface
//--------------------------------------------------------------------------------------
async command void Packet.clear(message_t* msg) {
	call RadioPacket.clear(msg);
}

async command uint8_t Packet.payloadLength(message_t* msg) {
	return call RadioPacket.payloadLength(msg);
}

async command void Packet.setPayloadLength(message_t* msg, uint8_t len) {
	call RadioPacket.setPayloadLength(msg, len);
}

async command uint8_t Packet.maxPayloadLength() {
	return call RadioPacket.maxPayloadLength();
}

async command void* Packet.getPayload(message_t* msg, uint8_t len) {
	if (len > call RadioPacket.maxPayloadLength())
		return NULL;

	return ((void*)msg) + call RadioPacket.headerLength(msg);
}

//--------------------------------------------------------------------------------------
// CC2420XDriverConfig interface
//--------------------------------------------------------------------------------------
async command uint8_t CC2420XDriverConfig.headerLength(message_t* msg) {
	return offsetof(message_t, data) - sizeof(cc2420xpacket_header_t);
}

async command uint8_t CC2420XDriverConfig.maxPayloadLength() {
	return sizeof(cc2420xpacket_header_t) + TOSH_DATA_LENGTH;
}

async command uint8_t CC2420XDriverConfig.metadataLength(message_t* msg) {
	return 0;
}

async command uint8_t CC2420XDriverConfig.headerPreloadLength() {
	// we need the fcf, dsn, destpan and dest
	return 7;
}

async command bool CC2420XDriverConfig.requiresRssiCca(message_t* msg) {
	//return call Ieee154PacketLayer.isDataFrame(msg);
	// DATA packet not CCA prior to tx, CONTROL & FTSP do; ER replaces CCA function already
	return ((call AMPacket.type(msg)) != AM_IMAC_LE);
}

//--------------------------------------------------------------------------------------
// PacketFlag interface
//--------------------------------------------------------------------------------------
async command bool PacketFlag.get[uint8_t bit](message_t* msg) {
	return getMeta(msg)->flags & (1<<bit);
}

async command void PacketFlag.set[uint8_t bit](message_t* msg) {
	getMeta(msg)->flags |= (1<<bit);
}

async command void PacketFlag.clear[uint8_t bit](message_t* msg) {
	getMeta(msg)->flags &= ~(1<<bit);
}

async command void PacketFlag.setValue[uint8_t bit](message_t* msg, bool value) {
	if (value)
		call PacketFlag.set[bit](msg);
	else
		call PacketFlag.clear[bit](msg);
}

//--------------------------------------------------------------------------------------
// PacketTimeStampRadio interface
//--------------------------------------------------------------------------------------
timestamp_metadata_t* getMetaTimestamp(message_t* msg) {
	return ((void*)msg) + sizeof(message_t) - call SubPacket.metadataLength(msg) - sizeof(flags_metadata_t) - sizeof(timestamp_metadata_t);
}

async command bool PacketTimeStampRadio.isValid(message_t* msg) {
	return call TimeStampFlag.get(msg);
}

async command uint32_t PacketTimeStampRadio.timestamp(message_t* msg) {
	return getMetaTimestamp(msg)->timestamp;
}

async command void PacketTimeStampRadio.clear(message_t* msg) {
	call TimeStampFlag.clear(msg);
}

async command void PacketTimeStampRadio.set(message_t* msg, uint32_t value) {
	call TimeStampFlag.set(msg);
	getMetaTimestamp(msg)->timestamp = value;
}

//--------------------------------------------------------------------------------------
// CC2420Packet interface
//--------------------------------------------------------------------------------------
// access rssi sampled right before or after the reception of a packet
// int16_t, not int8_t, to prevent underflow in calcInboundGain
async command int16_t CC2420Packet.getRssi(message_t* msg) {
	return ((int8_t)call PacketRSSI.get(msg) - (int16_t)RSSI_OFFSET);
}

async command int16_t CC2420Packet.getRssiIdle(message_t* msg) {
	uint16_t rssi = call PacketRSSIIdle.get(msg);
	return ((int8_t)(rssi & 0x00FF) - (int16_t)RSSI_OFFSET);
}

//--------------------------------------------------------------------------------------
// SoftwareAckConfig interface
//--------------------------------------------------------------------------------------
async command bool SoftwareAckConfig.requiresAckWait(message_t* msg)
{
    return call Ieee154PacketLayer.requiresAckWait(msg);
}

async command bool SoftwareAckConfig.isAckPacket(message_t* msg)
{
    return call Ieee154PacketLayer.isAckFrame(msg);
}

async command bool SoftwareAckConfig.verifyAckPacket(message_t* data, message_t* ack)
{
    return call Ieee154PacketLayer.verifyAckReply(data, ack);
}

async command void SoftwareAckConfig.setAckRequired(message_t* msg, bool ack)
{
    call Ieee154PacketLayer.setAckRequired(msg, ack);
}

async command bool SoftwareAckConfig.requiresAckReply(message_t* msg)
{
    return call Ieee154PacketLayer.requiresAckReply(msg);
}

async command void SoftwareAckConfig.createAckPacket(message_t* data, message_t* ack)
{
    call Ieee154PacketLayer.createAckReply(data, ack);
}

//#warning to turn as 256 jiffies as in cc2420 and choose h/w or s/w Ack
#ifndef SOFTWAREACK_TIMEOUT
#define SOFTWAREACK_TIMEOUT 8000    // 1000
#endif

async command uint16_t SoftwareAckConfig.getAckTimeout()
{
    return (uint16_t)(SOFTWAREACK_TIMEOUT * RADIO_ALARM_MICROSEC);
}

tasklet_async command void SoftwareAckConfig.reportChannelError()
{
#ifdef TRAFFIC_MONITOR
//      signal TrafficMonitorConfig.channelError();
#endif
}

}
