
module QueuedSendM {
  provides {
    interface StdControl;
    interface SendMsg as QueueSendMsg[uint8_t id];
    interface QueueControl;
  }

  uses {
    interface StdControl as CommControl;

    interface SendMsg as SerialSendMsg[uint8_t id];
    
#ifndef TOSSIM
#ifndef PLATFORM_XSM
    interface CC2420Control as RadioControl;
#else
    interface CC1000Control as RadioControl;
#endif
    interface MacControl;	
#endif

    interface ReceiveMsg;

    interface StdControl as TimerControl;
    interface Timer;

    interface Leds;
  }
}

implementation {

  uint8_t unicastSeqNo; //increased by one for each unicast call to GenericComm or similar 

  enum {
    MESSAGE_QUEUE_SIZE = SEND_QUEUE_SIZE
  };

  //Hongwei
  uint8_t deFactoMaxTransmitCount; 

  msgq_entry_t msgqueue[MESSAGE_QUEUE_SIZE];
  uint16_t enqueue_next, dequeue_next;

  bool retransmit;

  bool pending;  
  uint16_t pendingPeriod;  

  queue_link_status LinkState;
  
#ifdef DEBUG_ONLY
  uint16_t numUnicastTxes;
#endif

  uint16_t numPendingBeenStabilized;


  command result_t StdControl.init() 
  {
    int i;

    atomic {
      retransmit = TRUE;

      enqueue_next = dequeue_next = 0;

      pending = FALSE;
      pendingPeriod = 0;

      deFactoMaxTransmitCount = MAX_TX_COUNT;

      for (i = 0; i < MESSAGE_QUEUE_SIZE; i++)
	msgqueue[i].length = 0;

      LinkState.queueLen = 0;
      LinkState.num_pkt_dropped = 0;
      LinkState.num_self_pkt_dropped = 0;
    }

#ifdef DEBUG_ONLY
    numUnicastTxes = 0;
#endif

    numPendingBeenStabilized = 0;

#ifndef TOSSIM         
    call MacControl.enableAck();
#endif

#ifndef IN_KANSEI
    call TimerControl.init();
#endif

    unicastSeqNo = 0;

    return (call CommControl.init());
  } //end of StdControl.init() 


  command result_t StdControl.start() 
  {
#ifndef IN_KANSEI
    call TimerControl.start();
#endif

#ifndef TOSSIM   
    call RadioControl.SetRFPower(DefaultPowerLevel);
#endif

    call Timer.start(TIMER_REPEAT, QS_TIMER_INTERVAL);
    
    return (call CommControl.start());   
  }

  command result_t StdControl.stop() 
  {
    call Timer.stop();

#ifndef IN_KANSEI
    call TimerControl.stop();
#endif

    return (call CommControl.stop());
  }


  /* Queue data structure
     Circular Buffer
     enqueue_next indexes first empty entry
     buffer full if incrementing enqueue_next would wrap to dequeue
     empty if dequeue_next == enqueue_next
     or msgqueue[dequeue_next].length == 0
  */
#ifdef QS_USE_TASK_IN_TX
  task void QueueServiceTask() 
#else
  void QueueServiceTask() 
#endif
  {
    link_pkt_head* link_hd = (link_pkt_head*)(msgqueue[dequeue_next].pMsg.data);
    uint8_t id = msgqueue[dequeue_next].id;

    atomic ++(msgqueue[dequeue_next].xmit_count);
    if (msgqueue[dequeue_next].address != TOS_BCAST_ADDR &&
         msgqueue[dequeue_next].address != TOS_UART_ADDR 
         //&& id == AM_APP_DATA_CHANNEL
       ) //encode seqNo
      link_hd->tx_count = unicastSeqNo;

    if (!(call SerialSendMsg.send[id](msgqueue[dequeue_next].address,
                                                                 msgqueue[dequeue_next].length, 
                                                                 &(msgqueue[dequeue_next].pMsg)
			   )
           )
       ) {
      atomic {
	pending = FALSE;
	--(msgqueue[dequeue_next].xmit_count);
      }
    }
    else if (msgqueue[dequeue_next].address != TOS_BCAST_ADDR && 
                  msgqueue[dequeue_next].address != TOS_UART_ADDR 
                  //&& id == AM_APP_DATA_CHANNEL
                ) //success and is for data transmission
      unicastSeqNo++;

#ifdef DEBUG_ONLY
    if (msgqueue[dequeue_next].address != TOS_BCAST_ADDR && msgqueue[dequeue_next].address != TOS_UART_ADDR)
      numUnicastTxes++;
#endif

  } //end of QueueServiceTask()


  command result_t QueueSendMsg.send[uint8_t id](uint16_t address, uint8_t length, TOS_MsgPtr msg)
  {
    uint8_t i;
    link_pkt_head* link_hd=(link_pkt_head*)msg->data;
		
    //dbg(DBG_USR2, "QueuedSend: queue msg enq %d deq %d\n", enqueue_next, dequeue_next);

    //safety
    if (length <= 0)
      return FAIL;

    if (((enqueue_next + 1) % MESSAGE_QUEUE_SIZE) == dequeue_next) { // Fail if queue is full 
      if (id == AM_APP_DATA_CHANNEL) {
	atomic LinkState.num_pkt_dropped++;
	if (link_hd->source == TOS_LOCAL_ADDRESS)
	  atomic LinkState.num_self_pkt_dropped++;
      }
      else if (TOS_LOCAL_ADDRESS == BASE_STATION_ADDR && address == TOS_UART_ADDR)
	atomic LinkState.num_pkt_dropped++;
      
      return FAIL;
    }

    /* enqueue the packet */
    atomic{
      msgqueue[enqueue_next].address = address;
      msgqueue[enqueue_next].length = length;
      msgqueue[enqueue_next].id = id;

      msgqueue[enqueue_next].xmit_count = 0;
      msgqueue[enqueue_next].mac_latency = 0;

      if (address != TOS_UART_ADDR) {
	msgqueue[enqueue_next].last_hop = link_hd->source;
     		
	link_hd->delay = 0;
	link_hd->tx_count = 0;
	link_hd->source = TOS_LOCAL_ADDRESS;
      }
     
      //Hongwei: copy packet & set ack
      for (i = 0; i < length; i++)
	msgqueue[enqueue_next].pMsg.data[i] = msg->data[i];			
      msgqueue[enqueue_next].pMsg.ack = 0;

      //points to the next empty spot
      enqueue_next++; enqueue_next %= MESSAGE_QUEUE_SIZE;

      LinkState.queueLen++;
    }//atomic

    //next packet to send
    if (!pending && msgqueue[dequeue_next].length != 0) {
      atomic	{
	pending = TRUE;
	pendingPeriod = 0;
      }
#ifdef QS_USE_TASK_IN_TX
      if (!(post QueueServiceTask()))
	atomic pending = FALSE;
#else
      QueueServiceTask();
#endif
    }   
   
    return SUCCESS;
  } //end of .send[]()


  /* after one transmission */
  event result_t SerialSendMsg.sendDone[uint8_t id](TOS_MsgPtr msg, result_t success) {

    link_pkt_head* link_hd=(link_pkt_head*)msg->data;

    //call Leds.greenToggle();

    if (msg != &(msgqueue[dequeue_next].pMsg)) {
      return FAIL;		//filter out non-queuesend msg; this would be internal error
    }

    //calculate latency    
    if (success == SUCCESS && 
          msgqueue[dequeue_next].address != TOS_UART_ADDR && 
          msgqueue[dequeue_next].address != TOS_BCAST_ADDR
       ) 
      atomic msgqueue[dequeue_next].mac_latency += link_hd->delay;

    link_hd->delay = 0;

    if ((msg->ack != 0) ||
          (!retransmit) || 
          (msgqueue[dequeue_next].address == TOS_UART_ADDR) || 
          (msgqueue[dequeue_next].address == TOS_BCAST_ADDR)
        ) { //success or do not need retransmit 
      atomic {
      	link_hd->delay = msgqueue[dequeue_next].mac_latency;
	link_hd->tx_count = msgqueue[dequeue_next].xmit_count;//Telos/CC2420RadioM.nc does NOT retransmit at MAC layer either :-) 

      	if (msgqueue[dequeue_next].address != TOS_UART_ADDR || 
                          msgqueue[dequeue_next].address != TOS_BCAST_ADDR
                        )
	  link_hd->source = msgqueue[dequeue_next].last_hop;
      	      	
	msgqueue[dequeue_next].length = 0;
	dequeue_next++; dequeue_next %= MESSAGE_QUEUE_SIZE;
	if (LinkState.queueLen > 0)
	  LinkState.queueLen--;
      }//atomic
      signal QueueSendMsg.sendDone[id](msg, success);
    }
    else { //need to retry
      if (msgqueue[dequeue_next].xmit_count%deFactoMaxTransmitCount == 0) { //Hongwei: has failed badly
	atomic{
	  link_hd->delay = msgqueue[dequeue_next].mac_latency;
	  link_hd->tx_count = msgqueue[dequeue_next].xmit_count; //deFactoMaxTransmitCount;

	  if (msgqueue[dequeue_next].address != TOS_UART_ADDR || 
                          msgqueue[dequeue_next].address != TOS_BCAST_ADDR
                        )
	    link_hd->source = msgqueue[dequeue_next].last_hop;

#ifndef REDIRECT_FAIL_PKT
	  msgqueue[dequeue_next].length = 0;
	  dequeue_next++; dequeue_next %= MESSAGE_QUEUE_SIZE;
	  if (LinkState.queueLen > 0)
	    LinkState.queueLen--;
#else
	  if (msgqueue[dequeue_next].xmit_count >= MAX_TX_IF_REDIRECT) {//something is seriously wrong 
        	                                                                                                                                           //with this next-hop 
	    msgqueue[dequeue_next].length = 0;
	    dequeue_next++; dequeue_next %= MESSAGE_QUEUE_SIZE;
	    if (LinkState.queueLen > 0)
	      LinkState.queueLen--;
	  } 
	  /*
	  else  
	    msgqueue[dequeue_next].mac_latency = 0;
	  */
#endif
	} //atomic

	signal QueueSendMsg.sendDone[id](msg, FAIL);
      }
    }

    atomic pending = FALSE;

    // Send next
    if (msgqueue[dequeue_next].length != 0) {
      atomic {
	pending = TRUE;
	pendingPeriod = 0;
      }
#ifdef QS_USE_TASK_IN_TX
      if (!(post QueueServiceTask()))
	atomic pending = FALSE;
#else
      QueueServiceTask();
#endif
    }
    else if  (dequeue_next != enqueue_next) //stabilization
      atomic dequeue_next=enqueue_next;
    
    return SUCCESS;
  } //end of sendDone() 


 /* Timer fired: event run */
 event result_t Timer.fired()
 {  
   //stabilization of pending 
   if (pending) {  
     atomic {	  	
       pendingPeriod += QS_TIMER_INTERVAL;

       if (pendingPeriod > pendingDeadThreshold) {
	 pending = FALSE;
	 numPendingBeenStabilized++;
       }
     }
   }

   //check to see if need to send the next packet
   if (!pending && msgqueue[dequeue_next].length != 0) {
     atomic {
       pending = TRUE;
       pendingPeriod = 0;
     }
#ifdef QS_USE_TASK_IN_TX
     if (!(post QueueServiceTask()))
       atomic pending = FALSE;
#else
     QueueServiceTask();  
#endif
   }

   return SUCCESS;  	
 }//end of Timer.fired()
  
  
  command uint8_t QueueControl.getXmitCount() 
 {
    if (msgqueue[dequeue_next].length != 0)
      return msgqueue[dequeue_next].xmit_count;
    return 0;
  }

  
 default event result_t QueueSendMsg.sendDone[uint8_t id](TOS_MsgPtr msg, result_t success) 
 {
   return SUCCESS;
 }


 default command result_t SerialSendMsg.send[uint8_t id](uint16_t address, uint8_t length, TOS_MsgPtr msg)
   {
     return SUCCESS;
   }


 //Hongwei
 command result_t QueueControl.parameterTuning(QueuedSend_Tuning_Msg * tuningMsgPtr)
 {
   deFactoMaxTransmitCount = tuningMsgPtr->maxTransmitCount;
  
#ifndef TOSSIM 
   call RadioControl.SetRFPower(tuningMsgPtr->transmissionPower);
#endif
 
   return SUCCESS;
 }

  
 /* Change the destination of buffered packets */
 command result_t QueueControl.change_destination(uint8_t dst) 
 {
   uint8_t i;

   //self-stabilization
   if (enqueue_next >= MESSAGE_QUEUE_SIZE) 
     atomic enqueue_next = (dequeue_next + 1) % MESSAGE_QUEUE_SIZE;

   i = dequeue_next;
   while (i != enqueue_next) {
     //change corresponding field 
     if (msgqueue[i].length > 0 &&  
          msgqueue[i].address != TOS_UART_ADDR && 
          msgqueue[i].address != TOS_BCAST_ADDR && 
          msgqueue[i].address != dst) {
       msgqueue[i].address = dst;

       msgqueue[i].xmit_count = 0;
       msgqueue[i].mac_latency = 0;
     }

     //next element
     i = (i + 1)%MESSAGE_QUEUE_SIZE;
   } //while()

   return SUCCESS;
 } //end of change_destination()


 command uint8_t QueueControl.getOccupancy() {     
   return LinkState.queueLen;
 }
  
 command uint16_t QueueControl.get_num_pkt_dropped()
 {	
   return LinkState.num_pkt_dropped;
 }

 command uint8_t QueueControl.get_num_self_pkt_dropped()
 {
   return LinkState.num_self_pkt_dropped;
 }


#ifdef IN_KANSEI
  command result_t QueueControl.restartTimer()
  {
    call Timer.stop();
    return call Timer.start(TIMER_REPEAT, QS_TIMER_INTERVAL);
  }
#endif

#ifdef DEBUG_ONLY
  command uint16_t QueueControl.getNumUnicastTxes()
  {
    return numUnicastTxes;
  }
#endif


  command uint16_t QueueControl.getNumPendingBeenStabilized()
  {
    return numPendingBeenStabilized;
  }


 //Hongwei: tuning parameters
 event TOS_MsgPtr ReceiveMsg.receive(TOS_MsgPtr m)
 {
   QueuedSend_Tuning_Msg * tuningMsgPtr;

   tuningMsgPtr = (QueuedSend_Tuning_Msg *)(m->data);
   if (tuningMsgPtr->transmissionPower > 0 &&
       tuningMsgPtr->maxTransmitCount > 0) {

#ifndef TOSSIM
     call RadioControl.SetRFPower(tuningMsgPtr->transmissionPower);
#endif
      
     //deFactoMaxTransmitCount = tuningMsgPtr->maxTransmitCount;
   }
   else {
     ;
     //call Leds.redOn();
     //call Leds.greenToggle();
     //call Leds.yellowToggle();
   }

   return m;
 } //end of ReceiveMsg.receive()

}
