
/*
 * 07/20/2006 - : Added logic for dealing with temporal changes in network condition, specifically for the case where 
 *                           network condidtion (e.g., link reliability) improves. (Hongwei Zhang, hzhang@cs.wayne.edu)
 */

module LOFM
{
  provides {
    interface StdControl as Control; 
    interface Routing;
  }

  uses {
    interface QueueControl;   

    interface SendMsg as CONTROL_Send;
    interface ReceiveMsg as CONTROL_Receive;
#if defined(D_V_SNOOPING) || defined(SNOOPING_BASED_RCVER_EST)
    interface ReceiveMsg as snoop;
#endif
    
    interface SendMsg as DATA_Send;
    interface ReceiveMsg as DATA_Receive;

    interface StdControl as TimerControl;
    interface Timer;
    interface Random;

    interface Leds;
  }
}
implementation
{

  uint8_t initEstFbDelay; 

#if defined(BEACON_BASED_ESTIMATION) || defined(SNOOPING_BASED_RCVER_EST)
  uint8_t to_send_index; //for controlling sending linkReliability info via beacons/piggybacked-data
#endif

#ifdef SNOOPING_BASED_RCVER_EST
  snooping_link_est_t snoop_link_estimator[MAX_NUM_NODES_TO_SNOOP];
                                    //assume that one item-space is reserved for every node in the network; otherwise need table management
  uint8_t round;
#endif

#ifdef BEACON_BASED_ESTIMATION
  uint8_t hb_seqNo;
#endif

#if defined(QUICK_BEACON) || defined(SNOOPING_BASED_RCVER_EST)
  bool sendQuickBeacon;
#endif

#ifdef SUPPRESS_DUPLICATE_PKT
  duplicate_detector_t dup_detector[MAX_NUM_CHILDREN];
#endif

  bool toStartLOF; //whether to start the routing/LOF process now; for Kansei based experiment control 

  lof_routing_state_t lof; 
  loc_t my_loc; 

  ngbr_table_t ngbrTable[MAX_NGBR_TABLE_SIZE];
  ngbr_table_index_t ngbrTableIndex[MAX_NGBR_TABLE_SIZE];  //maintaining the ordered/ranked routing table; 
                                                                                                                                  //and points to ngbrTable 

#ifndef BEACON_BASED_ESTIMATION
  ngbr_metric_index_t yetToInitEstList[MAX_NGBR_TO_CONSIDER+1]; //initial-estimation is yet to be done
#endif
  //  uint8_t served_list[MAX_NETWORK_SIZE+1];  //for probe request; TODO: can be modified to serve unlimited network size 
                                                                                                    //                                  (and only with constraint on max. neighborhood size)

#ifdef USE_NGBR_SWITCH 
  ngbr_pairprob_t ngbrPairProb[MAX_TABLE_SIZE_DB];
#endif

#ifdef USE_DETECTOR
#ifdef OPP_TRACKER
  uint8_t oppTrackingState; 
  uint8_t oppTrackingPotential;
#endif

  uint8_t detectorState;	//state indicator

  uint32_t detector_threshold;        //threshold of the detector
  uint32_t last_entry_threshold;
  uint32_t threshold_positive;        //averaged threshold that leads to positive feedback
  bool probIntoSH;                            //whether try to enter "special handling" probabilistically when TxDecrease exceeds "weighted average of threshold-with-positive-feedback

  uint8_t improvedNetConditionHoldWindow;
  uint8_t holdWindow_positive;
  bool probIntoHoldWindow; 

  uint16_t preSpecialHandlingParent; //for positive/negative  threshold & temporal feedback
  uint8_t expectedNumPktsToSendAfterSH; //for temporal feedback

  uint8_t oldPosition;
  ngbr_table_index_t* tried;            //the neighbor sampled the last time
  ngbr_table_index_t* toTry;          //pointer to the neighbor to be sampled
  //uint32_t detector_feedback;           //detection feedback

  uint8_t detector_initiator; //whether and type of initiator

  uint8_t beliefSrc;	              //source of propogation process
  uint32_t timeToNextSeqNo; //milliseconds
  uint32_t beliefSeqNo;            //sequence number of the current belief propagation

  uint8_t whetherToPropagate;	//whether to propagate or not
  uint16_t propHoldTime;
  uint8_t numPktsRcvdDuringHoldTime;
  uint8_t propagatedHops;
#endif //USE_DETECTOR

#ifdef USE_AGE_FACTOR
  uint8_t age_factor[30]={88,83,78,73,68,64,60,56,53,50,47,44,41,38,36,34,32,30,28,26,25,23,22,20,19,18,17,16,15,14}; //%
#endif
#ifdef USE_DETECTOR
  uint8_t age_factor_detector[10] = {56, 42, 31, 23, 17, 13, 10, 7, 5, 4}; //%
#endif

#ifdef D_V_LOF
  uint8_t beaconCounter;
#endif

  TOS_Msg OutMsg, ReportMsg;

  uint32_t hb_timer_interval, timeElapsedSinceLastTimerFiring;
  uint8_t hb_alternate; //alternate in sending packets for initial link estimation and 
                                          //other purpose (such as probe-request, -reply, -withdrawal)

  uint8_t ngbr_switch_count;
  uint8_t num_tb_regeneration;

#ifdef DEBUG_ONLY 
  uint16_t numProbeReplySent;
  uint16_t numRouteWithdrawalSent;
  uint16_t numReplyBasedNgbrRevitalization; 
  uint16_t numSendButNoParent;
  uint16_t numFwdButNoParent;
  uint8_t numChildren;
  uint8_t numSH;
  uint8_t numSH2;
#endif


  //===================== component functions =========================//
#ifdef LOG_LOF_DETECTOR
  void log_lof_detector(uint8_t flag)
  {
    uint8_t data[13];

#ifdef USE_DETECTOR
    uint32_t* p32=(uint32_t*)data;

    p32[0] = beliefSeqNo;
    data[4] = beliefSrc;
    data[5] = propagatedHops; 
    data[6] = detectorState; 

    p32 = (uint32_t *)(&(data[7]));   //thresholds
    p32[0] = detector_threshold;
    data[11] = improvedNetConditionHoldWindow;
#endif

    signal Routing.reportValue(data, flag, 12);
  }
#endif

#ifdef LOG_MEAN_VARIANCE
  void log_mean_var(uint32_t ov)
  {
    uint8_t data[32];
    uint32_t* p32 = (uint32_t*)data;
    ngbr_table_index_t * q = lof.ngbr_table_head;
    uint8_t i, count=0;

    while(q!=NULL && count<3) {
	p32[count*2]=ngbrTable[q->index].deliveryLatency;
	p32[count*2+1]=ngbrTable[q->index].devDeliveryLatency;

	count++;
	q=q->next;
    }
    for(i=count; i<3; i++) {
	p32[i*2]=0;
	p32[i*2+1]=0;
    }
    p32[6]=ov;
    data[28]=0;
#ifdef USE_DETECTOR
    data[28]=detectorState;
#endif
			
    signal Routing.reportValue(data, 3, 29);
  } //end of log_mean_var()
#endif


  //-----------------------------------------------------------------------------//
#ifdef SNOOPING_BASED_RCVER_EST
  /*
   * Used in: receiver_side_link_estimator(), hb_receive_beacon()
   */
  void amIBetter(bool isSnoop, uint8_t i, uint32_t weight
#ifdef MAXIMIZE_HOP_COUNT
	            , uint8_t numHops
#elif defined(MAXIMIZE_RELIABILITY)
 	            , uint16_t pathReliability
#endif
	         ) {

    uint32_t newWeight;
#ifdef LOG_LOF_DETECTOR
    bool logDetector = FALSE;
#endif 

    //whether needs to send quick-beacon or immediately piggyback to a data packet
    if (isSnoop == TRUE && //not from a child
          snoop_link_estimator[i].linkReliability > 0 &&
         (TOS_LOCAL_ADDRESS == BASE_STATION_ID || 
           (lof.parent_id != TOS_LOCAL_ADDRESS //self is connected
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
             && weight + ngbrTable[lof.ngbr_table_head->index].CI_bound > ngbrTable[lof.ngbr_table_head->index].weight
                                                                                                                      //provides a potentially better route
#else
             && weight > ngbrTable[lof.ngbr_table_head->index].weight
#endif
            )
          )
        ) {//detector enabled
      if (TOS_LOCAL_ADDRESS == BASE_STATION_ID){
	//newWeight = (((uint32_t)NT_SCALE_FACTOR)<<(LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)) * SCALE_TEN / snoop_link_estimator[i].linkReliability;
	//newWeight = (((uint32_t)NT_SCALE_FACTOR)<<(LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT+3)) / snoop_link_estimator[i].linkReliability; //only consider unidirectional reliability
	newWeight = ((uint32_t)((((uint32_t)NT_SCALE_FACTOR)<<LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / snoop_link_estimator[i].linkReliability) << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT+3)) / snoop_link_estimator[i].linkReliability; //being conservative: consider directional reliability and assume ack-reliability is the same as data reliability

      }
      else {
	//newWeight = ((((uint32_t)NT_SCALE_FACTOR)<<(LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)) * SCALE_TEN / snoop_link_estimator[i].linkReliability) + ngbrTable[lof.ngbr_table_head->index].weight;
	//newWeight = ((((uint32_t)NT_SCALE_FACTOR)<<(LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT+3)) / snoop_link_estimator[i].linkReliability) + ngbrTable[lof.ngbr_table_head->index].weight; //only forward reliability
	newWeight = (((uint32_t)((((uint32_t)NT_SCALE_FACTOR)<<LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / snoop_link_estimator[i].linkReliability) << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT+3)) / snoop_link_estimator[i].linkReliability) + ngbrTable[lof.ngbr_table_head->index].weight; //both directions
      }

      if (weight > newWeight
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
            || (TOS_LOCAL_ADDRESS != BASE_STATION_ID
                 && weight + ngbrTable[lof.ngbr_table_head->index].CI_bound > newWeight
#ifdef MAXIMIZE_HOP_COUNT
                 && numHops - 1 < ngbrTable[lof.ngbr_table_head->index].numHops + 1
#else
                 && pathReliability < ngbrTable[lof.ngbr_table_head->index].pathReliability
#endif
              )
#endif
          ) {
	to_send_index = i;
	sendQuickBeacon = TRUE;
#ifdef LOG_LOF_DETECTOR
	logDetector = TRUE;
#endif
      }
    } //detector enabled
	  
    //table management: potentialChild
    if (snoop_link_estimator[i].potentialChild == FALSE &&
          (TOS_LOCAL_ADDRESS == BASE_STATION_ID ||
            (lof.parent_id != TOS_LOCAL_ADDRESS &&
#if !defined(MAXIMIZE_HOP_COUNT) && !defined(MAXIMIZE_RELIABILITY)
              weight > ngbrTable[lof.ngbr_table_head->index].weight
#else
              weight + ngbrTable[lof.ngbr_table_head->index].CI_bound > ngbrTable[lof.ngbr_table_head->index].weight
#endif
             )
           )
        )
      snoop_link_estimator[i].potentialChild = TRUE;
    else if (snoop_link_estimator[i].potentialChild == TRUE &&
                  TOS_LOCAL_ADDRESS != BASE_STATION_ID &&
                  (lof.parent_id == TOS_LOCAL_ADDRESS ||
#if !defined(MAXIMIZE_HOP_COUNT) && !defined(MAXIMIZE_RELIABILITY)
                     weight <= ngbrTable[lof.ngbr_table_head->index].weight
#else
                     weight + ngbrTable[lof.ngbr_table_head->index].CI_bound <= ngbrTable[lof.ngbr_table_head->index].weight
#endif
                   )
                 )
      snoop_link_estimator[i].potentialChild = FALSE;

#ifdef LOG_LOF_DETECTOR
    if (logDetector == TRUE)
      log_lof_detector(8);
#endif
  } //amIBetter()


  /*
   * Used in: data_receive(), snoop.receive()
   */
  void receiver_side_link_estimator(uint8_t src, uint8_t seqNo, bool isSnoop, uint32_t weight
#ifdef MAXIMIZE_HOP_COUNT
			      , uint8_t numHops
#elif defined(MAXIMIZE_RELIABILITY)
			     , uint16_t pathReliability
#endif
                                                                 )
  {
    uint8_t i, thisTxCount;
    uint16_t thisReliability;

    /*
    //??????
    if (TOS_LOCAL_ADDRESS == BASE_STATION_ID)
      return;
    */

    if (src >= MAX_NUM_NODES_TO_SNOOP)
      return;
    i = src; //when no table management

    /* check whether this node has already been in the table */
    /*table management: 
    for (i=0; i < MAX_NUM_NODES_TO_SNOOP && snoop_link_estimator[i].src != FLAG_NOT_INITIALIZED; i++){
     if (snoop_link_estimator[i].src == src) { //already in table
    */
    if (snoop_link_estimator[i].src != FLAG_NOT_INITIALIZED) {
      atomic {//for link reliability
	  thisTxCount = (seqNo >= snoop_link_estimator[i].last_tx_seqNo) ?
 	                               (seqNo - snoop_link_estimator[i].last_tx_seqNo):
            	                               (128 - snoop_link_estimator[i].last_tx_seqNo + seqNo);
	  if (thisTxCount > 0) {
	    snoop_link_estimator[i].numSucc++;
	    snoop_link_estimator[i].numSend += thisTxCount;
	  }
	  snoop_link_estimator[i].last_tx_seqNo = seqNo;
	  if (snoop_link_estimator[i].numSend < snoop_link_estimator[i].numSucc)   //stabilizing 
	    snoop_link_estimator[i].numSend = snoop_link_estimator[i].numSucc = 0;

	  //dbg(KEY_DEBUG, "ngbr-id = %d, numSucc = %d, numSend = %d\n", src, snoop_link_estimator[i].numSucc, snoop_link_estimator[i].numSend);

	  if ((snoop_link_estimator[i].numSend >= LINK_RELIABILITY_CALC_BASE_NUM || //another period 
	       (snoop_link_estimator[i].numSend >= CONSECUTIVE_SUCCESS_CALC_BASE && snoop_link_estimator[i].numSend == snoop_link_estimator[i].numSucc) || //very good link
	         thisTxCount > MAX_TX_COUNT //failure
	       ) && snoop_link_estimator[i].numSend > 0 
	      ) {
	    //weighted average
	    thisReliability = (snoop_link_estimator[i].numSucc << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / 
	                                   snoop_link_estimator[i].numSend;
	    if (snoop_link_estimator[i].linkReliability > 0)
	      snoop_link_estimator[i].linkReliability = (snoop_link_estimator[i].linkReliability - 
					    (snoop_link_estimator[i].linkReliability >> compPastW)
					  ) + (thisReliability >> compPastW);
	    else
	      snoop_link_estimator[i].linkReliability = thisReliability;
	    //for very low reliability
	    if (snoop_link_estimator[i].linkReliability < MIN_RELIABILITY) 
	      snoop_link_estimator[i].linkReliability = MIN_RELIABILITY; //set it as a 0.1% 
	    //start another calculation period
	    snoop_link_estimator[i].numSend = snoop_link_estimator[i].numSucc = 0;
	  } //another period

      } //atomic 

      //0) whether needs to send quick-beacon or immediately piggyback to a data packet
      //1) part of table management: potentialChild
      if (isSnoop == TRUE && weight == 0) //has overheard initEst pkt
	return;
      else 
	amIBetter(isSnoop, i, weight
#ifdef MAXIMIZE_HOP_COUNT
		  , numHops
#elif defined(MAXIMIZE_RELIABILITY)
		  , pathReliability
#endif
		);

      return;
  } //already here
 //table managment: } //for()

    /* a newly heard node */
    atomic {
      /* table management
	 if (i >= MAX_NUM_NODES_TO_SNOOP) { //table is already full; need to find a node to replace 
	 for (i=0; i < MAX_NUM_NODES_TO_SNOOP-1; i++) 
	 if (snoop_link_estimator[i].potentialChild == FALSE)
	 break;
	 }
      */
      //fill in information
      snoop_link_estimator[i].src = src;
      snoop_link_estimator[i].last_tx_seqNo = seqNo;
      snoop_link_estimator[i].numSucc = snoop_link_estimator[i].numSend = 1;
      snoop_link_estimator[i].linkReliability = 0;
      //
      if (snoop_link_estimator[i].potentialChild == FALSE &&
            (TOS_LOCAL_ADDRESS == BASE_STATION_ID ||
              (lof.parent_id != TOS_LOCAL_ADDRESS &&
#if !defined(MAXIMIZE_HOP_COUNT) && !defined(MAXIMIZE_RELIABILITY)
                weight > ngbrTable[lof.ngbr_table_head->index].weight
#else
                weight + ngbrTable[lof.ngbr_table_head->index].CI_bound > ngbrTable[lof.ngbr_table_head->index].weight
#endif
               )
             )
          )
	snoop_link_estimator[i].potentialChild = TRUE;
      else if (snoop_link_estimator[i].potentialChild == TRUE &&
                    TOS_LOCAL_ADDRESS != BASE_STATION_ID &&
                    (lof.parent_id == TOS_LOCAL_ADDRESS || 
#if !defined(MAXIMIZE_HOP_COUNT) && !defined(MAXIMIZE_RELIABILITY)
                       weight <= ngbrTable[lof.ngbr_table_head->index].weight
#else
                       weight + ngbrTable[lof.ngbr_table_head->index].CI_bound <= ngbrTable[lof.ngbr_table_head->index].weight
#endif
                     )
                   )
	snoop_link_estimator[i].potentialChild = FALSE;
    } //atomic

} //end of receiver_side_link_estimator()
#endif //SNOOPING_BASED_RCVER_EST


#ifdef SNOOPING_BASED_RCVER_EST
void attachLinkReliabilityList_snoop(linkReliability_list_t *lrList)
{
  uint8_t count = 0;
  uint8_t entriesAttached = 0;

  while (count < MAX_NUM_NODES_TO_SNOOP && //have not finished one round yet
               entriesAttached <= (MAX_ENTRIES_CARRIED_PER_HB-1)  //still room available
       ) {
    if (snoop_link_estimator[to_send_index].src != FLAG_NOT_INITIALIZED && //contains info
         snoop_link_estimator[to_send_index].linkReliability > 0 &&
         (snoop_link_estimator[to_send_index].potentialChild == TRUE 
           //|| round%FREQ_RATIO_FOR_POTENTIAL_CHILD == 0
         )
        ) {
      lrList[entriesAttached].src = (uint8_t)snoop_link_estimator[to_send_index].src;
      lrList[entriesAttached].linkReliability = snoop_link_estimator[to_send_index].linkReliability;
      entriesAttached++;
    }

    to_send_index = (to_send_index + 1)%MAX_NUM_NODES_TO_SNOOP;
    if (to_send_index == 0)
      round = (round + 1) % FREQ_RATIO_FOR_POTENTIAL_CHILD;
    count++;
  }//while()

  /*
  //if not filled up yet, try to fill it up with info not related to children (it is fine to have duplicate)
  if (count >= MAX_NUM_NODES_TO_SNOOP && entriesAttached <= (MAX_ENTRIES_CARRIED_PER_HB-1)) {
    count = 0;
    while (count < MAX_NUM_NODES_TO_SNOOP && //have not finished one round yet
                 entriesAttached <= (MAX_ENTRIES_CARRIED_PER_HB-1)  //still room available
               ) {
      if (snoop_link_estimator[to_send_index].src != FLAG_NOT_INITIALIZED && //contains info
           snoop_link_estimator[to_send_index].linkReliability > 0 &&
           snoop_link_estimator[to_send_index].potentialChild == FALSE
          ) {
	lrList[entriesAttached].src = (uint8_t)snoop_link_estimator[to_send_index].src;
	lrList[entriesAttached].linkReliability = snoop_link_estimator[to_send_index].linkReliability;
	entriesAttached++;
      }

      to_send_index = (to_send_index + 1)%MAX_NUM_NODES_TO_SNOOP;
      count++;
    }
  }
  */

  //if still not filled up yet
  while (entriesAttached <= MAX_ENTRIES_CARRIED_PER_HB-1) { //clear fields
    lrList[entriesAttached].src = 0;
    lrList[entriesAttached].linkReliability = 0;
    entriesAttached++;
  }

 } //end of attachLinkReliabilityList_snoop() 
#endif


#ifdef CONSIDER_FORWARD_RELIABILITY
void attachLinkReliabilityList(linkReliability_list_t *lrList)
{
  uint8_t count = 0;
  uint8_t entriesAttached = 0;

  while (count < MAX_NGBR_TABLE_SIZE && //have not finished one round yet
               entriesAttached <= (MAX_ENTRIES_CARRIED_PER_HB-1)  //still room available
       ) {
    if (ngbrTableIndex[to_send_index].flag == 1) {//contains ngbr info
      lrList[entriesAttached].src = (uint8_t)ngbrTable[ngbrTableIndex[to_send_index].index].node_id;
      lrList[entriesAttached].linkReliability = ngbrTable[ngbrTableIndex[to_send_index].index].link_reliability;
      entriesAttached++;
    }

    to_send_index = (to_send_index + 1)%MAX_NGBR_TABLE_SIZE;
    count++;
  }

  while (entriesAttached <= MAX_ENTRIES_CARRIED_PER_HB-1) { //clear fields
    lrList[entriesAttached].src = 0;
    lrList[entriesAttached].linkReliability = 0;
    entriesAttached++;
  }

 } //end of attachLinkReliabilityList() 
#endif


  /* Compose a probe packet to be sent 
   *
   * Used in: SendBeacon(), hb_timer_task()
   */
  uint8_t compose_probe_packet(int8_t pkt_type, uint8_t num_remaining, uint8_t* buf)
  {
    lof_heartbeat_t* lof_hb_pkt = (lof_heartbeat_t*)buf;

    atomic{
      lof_hb_pkt->type = pkt_type;
      lof_hb_pkt->src = TOS_LOCAL_ADDRESS;
      lof_hb_pkt->num_probe_to_send=num_remaining;
    }

#ifdef D_V_LOF
    if (pkt_type == PROBE_REPLY || pkt_type == PROBE_BEACON) {
      if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
	lof_hb_pkt->weight = 0;
#ifdef MAXIMIZE_HOP_COUNT
	lof_hb_pkt->numHops = 0;
#elif defined(MAXIMIZE_RELIABILITY)
	lof_hb_pkt->pathReliability = PERFECT_RELIABILITY;
#endif
#ifdef SNOOPING_BASED_RCVER_EST
	lof_hb_pkt->parent_id = TOS_LOCAL_ADDRESS;
#endif
      }
      else {
	lof_hb_pkt->weight = ngbrTable[lof.ngbr_table_head->index].weight;
#ifdef MAXIMIZE_HOP_COUNT
	lof_hb_pkt->numHops = (uint8_t)(ngbrTable[lof.ngbr_table_head->index].numHops + 1);
#elif defined(MAXIMIZE_RELIABILITY)
	lof_hb_pkt->pathReliability = ngbrTable[lof.ngbr_table_head->index].pathReliability;
#endif
#ifdef SNOOPING_BASED_RCVER_EST
	lof_hb_pkt->parent_id = lof.parent_id;
#endif
      }

#ifdef BEACON_BASED_ESTIMATION
      lof_hb_pkt->hb_seqNo = hb_seqNo;
      hb_seqNo++;
#ifdef CONSIDER_FORWARD_RELIABILITY
      attachLinkReliabilityList(lof_hb_pkt->lrList);
#endif
#endif

#ifdef SNOOPING_BASED_RCVER_EST
      attachLinkReliabilityList_snoop(lof_hb_pkt->lrList);
#endif

    } //PROBE_REPLY ||  PROBE_BEACON
#endif //D_V_LOF

    return sizeof(lof_heartbeat_t);
  } //end of compose_probe_packet(...)


#ifdef D_V_LOF
  /* send a beacon packet
   *
   * Used in: link_estimation(), hb_timer_task(), Timer.fired()
   */
  void SendBeacon()
  {
    uint8_t pkt_size; 

    //dbg(KEY_DEBUG, "enter SendBeacon()");

    pkt_size = compose_probe_packet(PROBE_BEACON, 0, OutMsg.data+QUEUE_HEAD_LEN);

    call CONTROL_Send.send(TOS_BCAST_ADDR, pkt_size + QUEUE_HEAD_LEN, &OutMsg);

    //dbg(KEY_DEBUG, "exit SendBeacon()");
  } //end of SendBeacon()
#endif	


  /* Change the destination addresses for packets in buffers; and log necessary stats information
   *
   * Used in: route_maintenance(), ngbr_switch()
   */
  void change_parent(uint8_t parent_id)
  {
#ifdef LOG_PROBBEINGBEST
    uint8_t data1[29];
    uint32_t* p32=(uint32_t*)data1;
    ngbr_table_index_t * q=lof.ngbr_table_head;
    uint8_t i,count=0;
#endif
#ifdef LOG_ORDERING
    /*log the order*/
    uint8_t data[MAX_NGBR_TABLE_SIZE];
    uint8_t size=0;
    ngbr_table_index_t* ptr = lof.ngbr_table_head;
		
    if(ptr!=NULL) {
      while(ptr!=NULL) {
	data[size]=ngbrTable[ptr->index].node_id;
	size++;
	ptr = ptr->next;			
      }		
      signal Routing.reportValue(data, 2, size);
    }
#endif

    call QueueControl.change_destination(parent_id);

#ifdef LOG_PROBBEINGBEST			
    while(q!=NULL && count<7) {
      p32[count]=ngbrTable[q->index].probBeingBest;
      count++;
      q=q->next;
    }
    for(i=count;i<7;i++)			
      p32[i]=0;		
    signal Routing.reportValue(data1,4,28);	
#endif
  } //end of change_parent()


  //--------------------------------------------------------------------------------------------------------------//
#ifdef USE_NGBR_SWITCH 
  /* Update the relative-goodness-probability for a pair of nodes
   *
   * Used in: new_pairprob(), update_head_prob() 
   */
  void update_prob(ngbr_table_index_t* p1, ngbr_table_index_t* p2, uint32_t prob)
  {
    uint8_t i;
		
    if(p1==NULL || p2==NULL)
      return;

    atomic {		
      for(i=0; i<MAX_TABLE_SIZE_DB; i++) {
	if(ngbrPairProb[i].ptr1==p1 && ngbrPairProb[i].ptr2==p2) {
	  ngbrPairProb[i].prob = prob; 
	  break; //Hongwei added
	}
	else if(ngbrPairProb[i].ptr1==p2 && ngbrPairProb[i].ptr2==p1) {
	  ngbrPairProb[i].prob = SCALE_THOUSAND - prob;
	  break; //Hongwei added
	}
      }
    } //atomic
  } //end of update_prob()


  /* Get the probability for a pair of nodes 
   * 
   * Used in: init_ngbr_switch()
   */
  uint32_t get_prob(ngbr_table_index_t* p1,ngbr_table_index_t* p2)
  {
    uint8_t i;
		
    if(p1==NULL || p2==NULL)
      return 0;
		
    for(i=0;i<MAX_TABLE_SIZE_DB;i++) {
      if(ngbrPairProb[i].ptr1==p1 && ngbrPairProb[i].ptr2==p2)
	return ngbrPairProb[i].prob;
      else if (ngbrPairProb[i].ptr1==p2 && ngbrPairProb[i].ptr2==p1) 
	return SCALE_THOUSAND-ngbrPairProb[i].prob;
    }
    return 0;	
  }


  /* Calculate new probability introduced by ptr2
   *
   * Used in: wrapUpOneInitialEst() <<which is called in initialLinkEstimation()>>
   */
  void new_pairprob(ngbr_table_index_t* ptr2)
  {		
    uint32_t thisProb;
    ngbr_table_index_t* ptr1 = lof.ngbr_table_head;

    if (ptr2 == NULL)
      return;

#ifndef SWITCH_DEAD_NGBR
    if(ngbrTable[ptr2->index].liveness == IS_DEAD)
      return;
#endif

    while(ptr1 != NULL && ptr1 != ptr2) {
#ifndef SWITCH_DEAD_NGBR
      if(ngbrTable[ptr1->index].liveness == IS_ALIVE) {
#endif
	thisProb = prob_greater(ngbrTable[ptr1->index].logWeight, ngbrTable[ptr1->index].devLogWeight, 
			      ngbrTable[ptr2->index].logWeight, ngbrTable[ptr2->index].devLogWeight
			    );
	update_prob(ptr1, ptr2, thisProb);
#ifndef SWITCH_DEAD_NGBR
      } 
#endif
      ptr1=ptr1->next;
    }			
  } //end of new_pairprob()

	
  /* Update the pairwise probability of being-better-than-one-another after each adaptation of the estimation of the head neighbor 
   *
   * Used in: putInOrder(), ngbr_switch() 
   */
  void update_head_prob()   
  {
    uint32_t thisProb;
    ngbr_table_index_t* ptr1 = lof.ngbr_table_head;
    ngbr_table_index_t* ptr2;
		
    if (ptr1 == NULL)
      return;
	
    ptr2 = ptr1->next;

#ifndef SWITCH_DEAD_NGBR
    if (ngbrTable[ptr1->index].liveness == IS_DEAD)
      return;
#endif

    while (ptr2 != NULL) {
#ifndef SWITCH_DEAD_NGBR
      if(ngbrTable[ptr2->index].liveness == IS_ALIVE) {
#endif
	thisProb = prob_greater(ngbrTable[ptr1->index].logWeight, ngbrTable[ptr1->index].devLogWeight,
                                                                  ngbrTable[ptr2->index].logWeight, ngbrTable[ptr2->index].devLogWeight
			    );
	update_prob(ptr1, ptr2, thisProb);
#ifndef SWITCH_DEAD_NGBR
      }
#endif
      ptr2=ptr2->next;
    }
  } //end of update_head_prob()


  /* To initialize/reset information related to neighbor-switching, when the current
   *  next-hop is changed due to initial link estimation or process_tx/txexc, or when the  
   *  regenerated node does perform well.
   *
   * called by route_maintenance(...)
   */
  task void init_ngbr_switch()
  {
    ngbr_table_index_t * head = lof.ngbr_table_head;
    ngbr_table_index_t * ptr1, * ptr2;
    ngbr_table_index_t * tail;
    uint8_t ptr1_counter, ptr2_counter; 
    uint16_t chainProb[MAX_NGBR_TABLE_SIZE+1]; 
    uint16_t sumProbs = 0;
    uint16_t tailProbs = SCALE_THOUSAND;
    uint16_t curProbs = 0;
    uint16_t prePos = 0;
    uint16_t curPos = 0;

    //dbg(FLOW_DEBUG, "entering init_ngbr_switch()\n");
	 
#ifdef USE_DETECTOR
    if (detectorState == STATE_SPECIAL_HANDLING || detectorState == STATE_INFORMED_DETECTION
#ifdef OPP_TRACKER
          || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
        ) {
      lof.pktsBeforeSwitching = 0xffff;
      return;
    }
#endif 

    /* check whether there is more than one neighbor; if no, simply set pktsBeforeSwitching to -1 and return */
    if (head == NULL || head->next == NULL 
#ifndef SWITCH_DEAD_NGBR
          || ngbrTable[head->index].liveness == IS_DEAD || ngbrTable[head->next->index].liveness == IS_DEAD 
#endif
	) {//no or only one neighbor
      lof.pktsBeforeSwitching = 0xffff;
      return;
    }

    /*For each neighbor, compute the probability that it should be at the head of the neighbor list */	  
    //0) init probBeingBest to 1
    ptr1 = head;
    while (ptr1 != NULL 
#ifndef SWITCH_DEAD_NGBR
                 && ngbrTable[ptr1->index].liveness == IS_ALIVE
#endif
	   ) {
      ngbrTable[ptr1->index].probBeingBest = SCALE_THOUSAND;
      ptr1 = ptr1->next; 
    }  
    //A) compute the probBeingBest for each non-parent neighbor: prob. that each node should be at the head position
    //       if only considering nodes in front of a node itself in the neighbor list	  
    ptr1 = head;
    ptr1_counter = 1;
    tail = ptr1;
    while (ptr1 != NULL && ptr1_counter <= lof.ngbr_table_size
#ifndef SWITCH_DEAD_NGBR
	   && ngbrTable[ptr1->index].liveness == IS_ALIVE
#endif
	   ) {	             
      if (ptr1_counter == 1) {
	ngbrTable[ptr1->index].probBeingBest = ngbrTable[ptr1->index].probBeingBest* 
                                                                                                  get_prob(head, head->next) / SCALE_THOUSAND; 
                                                                                                                                                      //for numerical reasons, do not format "*="
	goto next_ngbr;
      }
      else 
	ngbrTable[ptr1->index].probBeingBest = ngbrTable[ptr1->index].probBeingBest*
                                                                                                  get_prob(ptr1, head)/SCALE_THOUSAND;

      ptr2 = head->next;
      ptr2_counter = 2;
      //compare to each other neighbor in front of self in the neighbor list
      while (ptr2->next != NULL && ptr2_counter < ptr1_counter && ptr2_counter < lof.ngbr_table_size
#ifndef SWITCH_DEAD_NGBR
	     && ngbrTable[ptr2->index].liveness == IS_ALIVE
#endif
	     ) {
	ngbrTable[ptr1->index].probBeingBest = ngbrTable[ptr1->index].probBeingBest * 
                                                                                                 (SCALE_THOUSAND - get_prob(ptr2, ptr1)
				                  + get_prob(head,ptr1) * 
                                                                                                     (SCALE_THOUSAND - ngbrTable[ptr2->index].probBeingBest)/
                                                                                                     SCALE_THOUSAND
				                 ) / SCALE_THOUSAND;

	ptr2 = ptr2->next;
	ptr2_counter++;
      }//ptr2
    next_ngbr:	    
      chainProb[ptr1_counter] = ngbrTable[ptr1->index].probBeingBest;

      tail = ptr1; 
      ptr1 = ptr1->next; 
      ptr1_counter++; 
    } //ptr1
    //B) compute the probability by considering the nodes behind a node itself in the neighbor list too; going backwards from the tail
    sumProbs = ngbrTable[tail->index].probBeingBest; 
    ptr1_counter = ptr1_counter - 2; 
    while (ptr1_counter > 0) { 
      if (ptr1_counter > 1)
	tailProbs = tailProbs*(SCALE_THOUSAND - chainProb[ptr1_counter+1])/SCALE_THOUSAND;

      chainProb[ptr1_counter] = chainProb[ptr1_counter] * tailProbs / SCALE_THOUSAND;

      sumProbs += chainProb[ptr1_counter];
      ptr1_counter--;  
    }

    //C) asign the probability to neighbor table array
    ptr1_counter=1;
    ptr1 = head;
    while (ptr1 != NULL && ptr1_counter <= lof.ngbr_table_size
#ifndef SWITCH_DEAD_NGBR
                 && ngbrTable[ptr1->index].liveness == IS_ALIVE
#endif 
               )  {
      ngbrTable[ptr1->index].probBeingBest = chainProb[ptr1_counter];

      //stabilizing
      if (ngbrTable[ptr1->index].probBeingBest > SCALE_THOUSAND)
	ngbrTable[ptr1->index].probBeingBest = SCALE_THOUSAND;

      ptr1_counter++;
      ptr1 = ptr1->next;
    }

    //dbg(KEY_DEBUG, "sumProbs = %d\n", sumProbs);

    /* ? stabilization */
    if (sumProbs <= 1) //i.e., 0.001
      sumProbs = SCALE_THOUSAND;
	
    /* allocate the slots according to the calculated probabilities */
    ptr1 = head;
    ptr1_counter = 0;
    while (ptr1 != NULL && ptr1_counter < lof.ngbr_table_size
#ifndef SWITCH_DEAD_NGBR
	   && ngbrTable[ptr1->index].liveness == IS_ALIVE
#endif 
               )  {
      /*
      //? stabilization 
      if (ngbrTable[ptr1->index].probBeingBest < MIN_RELIABILITY) 
	ngbrTable[ptr1->index].probBeingBest = MIN_RELIABILITY;
      */

      ngbrTable[ptr1->index].rangeLower = prePos;
      //compute rangeUpper
      curProbs += (ngbrTable[ptr1->index].probBeingBest*SCALE_THOUSAND / sumProbs); //normalize the probability
      if (curProbs > SCALE_THOUSAND) //stabilization
	curProbs = SCALE_THOUSAND; 

      curPos = (RANDOM_NUM_MAX  / SCALE_THOUSAND) * curProbs + 1; //semantically, should be: 
                                                                                                                         //RANDOM_NUM_MAX * curProbs / SCALE_THOUSAND + 1
      ngbrTable[ptr1->index].rangeUpper = curPos;
      //move to the next neighbor
      prePos = curPos+1;
      ptr1 = ptr1->next;
      ptr1_counter++;
    } //allocate slots

    /* Set pktsBeforeSwitching to head->probBeingBest * MAX_PKTS_BEFORE_SWITCH */	    
    if (ngbrTable[head->index].probBeingBest <= 0)  //stabilization
      lof.pktsBeforeSwitching = 1;
    else {
      lof.pktsBeforeSwitching = (ngbrTable[head->index].probBeingBest * 
	                                     (ptr1_counter * NUM_PER_NODE_PKTS_BETWEEN_NET_CONDITION_CHANGE)
                                                         /SCALE_THOUSAND
		               ) + 1;

      dbg(KEY_DEBUG, "pktsBeforeSwitching = %d, probBeingBest = %d, [lower, upper] = [%d, %d], # of neighbors = %d \n", lof.pktsBeforeSwitching, ngbrTable[head->index].probBeingBest, ngbrTable[head->index].rangeLower, ngbrTable[head->index].rangeUpper, lof.ngbr_table_size); 

      if (lof.pktsBeforeSwitching <= 0)
	lof.pktsBeforeSwitching = 1;
    }

    //dbg(FLOW_DEBUG, "exiting init_ngbr_switch()\n");
  } //end of init_ngbr_switch(...)


  /* To perform ngbr-switching 
   * 
   * Used in: process_tx(...)
   */
  void ngbr_switch() 
  {	  
    ngbr_table_index_t *switched,*pre;
    uint16_t randInt;
    ngbr_table_index_t * head = lof.ngbr_table_head;	  
	  	  
    dbg(FLOW_DEBUG, "entering ngbr_switch()\n");

    randInt = (call Random.rand()) % RANDOM_NUM_MAX;

    if (head == NULL || head->next == NULL 
#ifndef SWITCH_DEAD_NGBR
         || ngbrTable[head->next->index].liveness == IS_DEAD
#endif
       )
      goto done;

    /* if the current neighbor remains to be chosen */
    if (ngbrTable[head->index].rangeLower <= randInt && randInt <= ngbrTable[head->index].rangeUpper)
      goto done;
		
    dbg(FLOW_DEBUG, "range lower=%d,randInt=%d,upper=%d ngbr_switch()\n",ngbrTable[head->index].rangeLower,randInt,ngbrTable[head->index].rangeUpper);
	
    /* otherwise, check which non-current-next-hop neighbor to switch to */
    pre=head;
    switched = head->next;
    while (switched != NULL 
#ifndef SWITCH_DEAD_NGBR
	   && ngbrTable[switched->index].liveness == IS_ALIVE
#endif
	   ) {
      //check whether to regenerate this node
      if (ngbrTable[switched->index].rangeLower <= randInt && randInt <= ngbrTable[switched->index].rangeUpper)
	break;
      //check the next one
      pre=switched;
      switched = switched->next;
    }
    /* regenerate the neighbor by making it the current next-hop */
    if (switched != NULL 
#ifndef SWITCH_DEAD_NGBR
	&& ngbrTable[switched->index].liveness == IS_ALIVE
#endif
	) {  //refresh the current next-hop

      update_head_prob(); //must not have been done in putInOrder, 
                                             //otherwise would have changed parent, and thus will not ngbr-switch immediately
      atomic {
	lof.parent_id = ngbrTable[switched->index].node_id;

#ifdef USE_DETECTOR
	preSpecialHandlingParent = 0xFFFF;
	expectedNumPktsToSendAfterSH = 0;
#endif
	   	
	/* move switched to the head*/
	pre->next=switched->next;
	switched->next = head;
	lof.ngbr_table_head = switched;

	ngbr_switch_count++;
      }//atomic 

      //update destination address of buffered packets
      change_parent(lof.parent_id);
    } //end of "regenerate the neighbor"

    /* No need to regenerate dead neighbor; 
     * Or do not initialize the ngbr-regeneration yet, wait to see whether the switched node performs well, which is checked in route-maintenane. */ 
  done: 
    lof.pktsBeforeSwitching = 0xffff; 
  } //end of ngbr_switch(...)

#endif //USE_NGBR_SWITCH 


  //-----------------------------------------------------------------------------------------//
  /* Set node location based on node id */  
  void set_loc(uint8_t id, loc_t* nloc)
  {
    atomic{
      nloc->x = id%GRID_SIZE;
      nloc->y = id/GRID_SIZE;
    }
  } //end of set_loc()


  /* Reset some fields of certain neighbor table items 
  *
  * Used in: new_ngbr(), init_state()
  */
  void reset_neighbor_partially(ngbr_table_t * ngbr)
  {
    atomic {
      ngbr->numSend = ngbr->numSucc = 0;
      ngbr->link_reliability = 0;
#ifdef MAXIMIZE_RELIABILITY
      ngbr->numDataSend = ngbr->numDataSucc = 0;
      ngbr->data_reliability = 0;
#endif
#ifdef SNOOPING_BASED_RCVER_EST
      ngbr->ackReliability = (2 << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT); //set it as 2 (to mark that it has not been initialized yet)
#endif
      /*Hongwei
	ngbr->deliveryLatency = 0;
	ngbr->deliveryLatency_Log = 0;
	ngbr->devDeliveryLatency = 0; 
	ngbr->devLog = 0;
      */
      ngbr->weight = 0; 
#ifndef USE_NGBR_SWITCH
      ngbr->devWeight = 0; 
#else
      ngbr->logWeight = 0;
      ngbr->devLogWeight = 0;
#endif

#ifdef USE_DETECTOR
      ngbr->candidatePotential = 0;
      ngbr->detectorLQI = 0;
      ngbr->devDetectorLQI = 0;
      ngbr->last_level_of_entry = 0;
#ifdef OPP_TRACKER
      ngbr->oppTrackerLQI = 0;
      ngbr->oppNode = 0;
#endif
#endif

    } //atomic
  } //end of reset_neighbor_partially()


  /* Tries to find a routing table entry for this neighbor
   *
   * Used in: move_to_initialEstNgbr(), handleANewlyHeardNgbr()
   */
  ngbr_table_index_t* new_ngbr(uint8_t node_id, uint32_t ngbrWeight, uint8_t thisSeqNo)
  {
      ngbr_table_t * ngbr;
      ngbr_table_index_t *prePtr, *ptr;
      uint8_t new_index;

#ifndef NON_GEO_DV
      loc_t ngbr_loc;
      uint32_t Lsr;
      uint32_t Lsd;
      uint32_t Lrd;
      uint32_t Le;
#endif 

      for (new_index=0; new_index < MAX_NGBR_TABLE_SIZE && ngbrTableIndex[new_index].flag == 1; new_index++)
	;
      if (new_index == MAX_NGBR_TABLE_SIZE) {//neighbor table entries have been used up; 
  	                                                                               //flash out the least ranked node, (including this current node in D-V-LOF)
	prePtr = NULL;
	ptr = lof.ngbr_table_head;
	while (ptr->next != NULL) {
	  prePtr = ptr;
	  ptr = ptr->next;
	}
	if (prePtr != NULL
#ifdef D_V_LOF
                         &&(ngbrTable[ptr->index].liveness == IS_DEAD || 
	             (ngbrTable[ptr->index].weight > ngbrWeight  //consider this new neighbor only if it is better than the current one
#ifdef BEACON_BASED_ESTIMATION
	              && ngbrTable[ptr->index].link_reliability > 0  //has finished at least one round of estimation
#endif
	              )
	            )
#endif
	    ) {
	  atomic {
	    prePtr->next = NULL;
	    lof.ngbr_table_size--;
	    ptr->flag = 0;
	    new_index = ptr->index;
	  }
	}
	else {//cannot find a neighbor table entry
	  return NULL;
	} 
      }

      atomic {
	ngbr = &ngbrTable[new_index];
	/* initialize information about the neighbor */
	ngbr->node_id = node_id;
      }

#if !defined(D_V_LOF) || !defined(NON_GEO_DV)
      set_loc(node_id, &ngbr_loc); 
      Lsr = distance(&my_loc, &ngbr_loc);  //has been scaled up by 10
      Lsd = distance(&my_loc, &lof.base_loc);  //has been scaled up by 10
      Lrd = distance(&lof.base_loc, &ngbr_loc);  //has been scaled up by 10

      //dbg(KEY_DEBUG, "my-id = %d, ngbr-id = %d \n", TOS_LOCAL_ADDRESS, node_id);

      //new geo. metric
      Le = Lsd - Lrd; //has been scaled up by 10 
      if (Lsd > Lrd) {
#ifndef D_V_LOF
	atomic ngbr->numHops = (uint16_t) (SCALE_THOUSAND / Le);  //been scaled up by a factor of 100
#ifdef MAXIMIZE_RELIABILITY
	ngbr->estGeoHops = Lsd % Le == 0? Lsd / Le : Lsd / Le + 1;
#endif
#else
	;
#endif
	//dbg(KEY_DEBUG, "Le = %d,  ngbr->numHops = %d \n", Le, ngbr->numHops); 
      }
      else
	return NULL; 
#endif //NON_GEO_DV

      atomic {
	ngbr->numInitSamplesToTake = NUM_INIT_SAMPLES; 
	ngbr->liveness = IS_ALIVE;
	ngbr->consecutive_tx_failure = 0;
#if defined(SUPPRESS_DUPLICATE_PKT) || defined(USE_AGE_FACTOR)
	ngbr->seqNo = 0;
#endif
	ngbrTableIndex[new_index].flag=1;
	ngbrTableIndex[new_index].index = new_index;
	ngbrTableIndex[new_index].next=NULL;
      }

      reset_neighbor_partially(ngbr);

#ifdef BEACON_BASED_ESTIMATION
      ngbr->ttl = TTL_BEFORE_HEARING_ANOTHER_BEACON;
      ngbr->last_hb_seqNo = thisSeqNo;
      ngbr->numSucc++; 
      ngbr->numSend++;
      ngbr->ngbr_weight = ngbrWeight;
      ngbr->forwardReliability = PERFECT_RELIABILITY;
      //ngbr->weight = ngbrWeight + NT_SCALE_FACTOR * SCALE_THOUSAND; //expected # of tx scaled; set it to be high (i.e., 100 txes) before estimation
      ngbr->weight = ngbrWeight + (3000 << 13);
#endif

#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
      ngbr->n = ngbr->CI_bound = 0;
#endif

      return &ngbrTableIndex[new_index];
    }  //end new_ngbr(...)


  /* Schedule another neighbor for initial link estimation, if no one else is in process 
   *
   * Used in: initialLinkEstimation(), handleANewlyHeardNgbr(), blacklist_if_exists()
   */
  void move_to_initialEstNgbr()
  { 
    ngbr_table_index_t * ptr;

    if (lof.initialEstNgbr != NULL || lof.initialEst_head == NULL) {//is perform initial estimation for other neighbor 
                                                                                                                         //or has finished initial estimation
      //stabilization
      if (lof.initialEst_head == NULL)
	lof.initialEstNgbr = NULL;

      return;
    }
  
    //first, tries to create a table entry for performing init. estimation for this node
    if((ptr = new_ngbr(lof.initialEst_head->node_id, 
#ifdef D_V_LOF
  	                    lof.initialEst_head->ngbr_weight, 
	                    0 //dummy
#else 
	                    0, 0 //dummy
#endif
                                       )
         ) == NULL
       )
      return; 

    atomic {
#ifdef D_V_LOF
      ngbrTable[ptr->index].ngbr_weight = lof.initialEst_head->ngbr_weight; //D-V diffusion computation
#ifdef MAXIMIZE_HOP_COUNT
      ngbrTable[ptr->index].numHops = lof.initialEst_head->ngbr_numHops;
#elif defined(MAXIMIZE_RELIABILITY)
      ngbrTable[ptr->index].ngbrPathReliability = lof.initialEst_head->ngbr_pathReliability;
#endif
#endif
      lof.initialEstNgbr = ptr;
      lof.initialEst_head = lof.initialEst_head->next;

      if (lof.initialEst_head == NULL)
	lof.initialEst_tail=NULL;
    }
  }//end of move_to_initialEstNgbr()


  /* Used in: putInOrder() */
  uint8_t distanceCmpID(uint8_t id1, uint8_t id2)
  {
    loc_t loc1, loc2;

    set_loc(id1, &loc1);
    set_loc(id2, &loc2);
		
    return distanceCmp(&lof.base_loc, &loc1, &loc2);
  } //end of distanceCmpID()


  /* Put an item in the right order of the neighbor table 
  *
  * Used in:   route_maintenance(), refresh_ngbr_table(), refresh_ngbr_table_beacon(), 
  *                   initialLinkEstimation(), process_tx(), process_txexc()
  */
  void putInOrder(ngbr_table_index_t* head, ngbr_table_index_t* item) 
  {
      ngbr_table_index_t * tp_pre, *tp, *table_head;
      ngbr_table_index_t * item_pre;

      table_head = lof.ngbr_table_head;

      /* handle special cases */
      if (table_head == NULL) { //item is the first neighbor
	dbg(FLOW_DEBUG,"head is null \n"); 

	atomic {
	  item->next = NULL; //self-stabilization
	  lof.ngbr_table_head = item;
	} 
	return;
      }
      else if (item==NULL) {
	return;
      }

      /* search for the position "tp" in front of which "item" should be placed */
      tp_pre = tp = table_head;
      while (tp != NULL) {
	if (ngbrTable[tp->index].node_id == ngbrTable[item->index].node_id //is "item" itself; not use "tp == item", 
	                                                                                                           //to d/w dynamic memory movement by the system
	    || ngbrTable[tp->index].liveness > ngbrTable[item->index].liveness  //level 1: liveness
	    || (ngbrTable[tp->index].liveness == ngbrTable[item->index].liveness &&
#if (!defined(MAXIMIZE_HOP_COUNT) && !defined(MAXIMIZE_RELIABILITY)) || defined(USE_NGBR_SWITCH)
	        (ngbrTable[tp->index].weight < ngbrTable[item->index].weight    //level 2: weight 
#else
	        (ngbrTable[tp->index].weight + ngbrTable[item->index].CI_bound < ngbrTable[item->index].weight
#endif
	          || ((ngbrTable[tp->index].weight <= ngbrTable[item->index].weight
#ifdef MAXIMIZE_HOP_COUNT
		       //#ifndef USE_NGBR_SWITCH
	               || ngbrTable[tp->index].weight <= ngbrTable[item->index].weight + ngbrTable[item->index].CI_bound
		       //#else 
		       //	               || ngbrTable[tp->index].logWeight <= ngbrTable[item->index].logWeight + ngbrTable[item->index].CI_bound
		       //#endif
#endif
	              )&& 
#ifdef MAXIMIZE_HOP_COUNT 
	              (ngbrTable[tp->index].numHops > ngbrTable[item->index].numHops //level 3: hop count
                                    || (ngbrTable[tp->index].numHops == ngbrTable[item->index].numHops &&
#elif defined(MAXIMIZE_RELIABILITY)
	             (ngbrTable[tp->index].pathReliability > ngbrTable[item->index].pathReliability //level 3': path reliability
 	               || (ngbrTable[tp->index].pathReliability == ngbrTable[item->index].pathReliability &&
#endif
#ifndef USE_NGBR_SWITCH 
	                   (ngbrTable[tp->index].devWeight < ngbrTable[item->index].devWeight //level 4: variance in weight
	                     || (ngbrTable[tp->index].devWeight == ngbrTable[item->index].devWeight && 
#else 
	                   (ngbrTable[tp->index].devLogWeight < ngbrTable[item->index].devLogWeight
	                     || (ngbrTable[tp->index].devLogWeight == ngbrTable[item->index].devLogWeight && 
#endif
		     distanceCmpID(ngbrTable[tp->index].node_id, ngbrTable[item->index].node_id) > 1  //level 5: distance
	                        )
	                   )
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
	                 )
	             ) 
#endif
	           )
	         )
	      )
	   ) { 
	  tp_pre = tp; 
	  tp = tp->next;
	}
	else
	  break;
      }

      /* to put "item" in front of "tp", if tp != NULL; otherwise, move "item" to the end of the list */
      //need not to do anything  
      if (item->next == tp) {
#ifdef OPP_TRACKER
	if (oppTrackingState == STATE_OPP_TRACKING_SH)
	  atomic oppTrackingState = STATE_OPP_TRACKING_NO;
#endif
	return;
      }
      //otherwise, to put item in order
#ifdef USE_NGBR_SWITCH
      if (item == lof.ngbr_table_head || tp == lof.ngbr_table_head) //will need to change parent
	update_head_prob(); //update probability related to head (delayed)
#endif
      atomic {
	//find the node preceding "item"
	item_pre = table_head;
	if (item != table_head)
	  while (item_pre != NULL && item_pre->next != item)
	    item_pre = item_pre->next;
	//remove "item" first
	if (item == table_head) {
#if defined(USE_DETECTOR) && defined(OPP_TRACKER)
	  if (detectorState != STATE_SPECIAL_HANDLING && detectorState != STATE_INFORMED_DETECTION &&
                            oppTrackingState != STATE_OPP_TRACKING_SH 
	      ) {
	    ngbrTable[item->next->index].oppTrackerLQI = ngbrTable[item->next->index].detectorLQI;
	    ngbrTable[item->next->index].oppNode = ngbrTable[table_head->index].node_id;
	  }
	  else if (oppTrackingState == STATE_OPP_TRACKING_SH) {
	    oppTrackingState = STATE_OPP_TRACKING_NO;
	    ngbrTable[item->next->index].oppTrackerLQI = 0; //reset
	  }
#endif
	  lof.ngbr_table_head = item->next;
	}
	else
	  item_pre->next = item->next;
	//then insert item between tp_pre and tp
	if (tp == table_head) {
	  lof.ngbr_table_head = item;
	  item->next = table_head;
	}
	else {
	  tp_pre->next = item;
	  item->next = tp;
	}
      } //atomic

  } //end of putInOrder(...)


#ifdef USE_DETECTOR
  /* Get the current rank/order of certain neighbor in the routing table list
  *
  *Used in: route_maintenance()
  */
  ngbr_table_index_t* getRank(uint8_t ngbIndex, uint8_t* pos)
  {
    ngbr_table_index_t * ptr = lof.ngbr_table_head;
    uint8_t ret = 0;
			
    while (ptr != NULL) {
      if (ptr->index == ngbIndex) {
	*pos = ret;
	return ptr;
      }				
      ret++;
      ptr = ptr->next;
    }
 
    *pos = ret;

    return ptr;
  }//end of getRank()
#endif


  /* 1) Adapt route selection based on data-driven link estimation;
   * 2) Perform special handling in the case of significant change (i.e., bad => good) in network condition.
   *
   * Used in: initialLinkEstimation(), process_tx(), process_txexc(), ... 
   */
  void route_maintenance()
  {
    ngbr_table_index_t* ptr;
#ifdef USE_DETECTOR
    bool hasReachedTableEnd;
    uint8_t i, newPos;
    uint32_t old_weight;
#endif

    dbg(FLOW_DEBUG,"enter route_maintenance \n");

#ifdef BEACON_BASED_ESTIMATION 
    if (TOS_LOCAL_ADDRESS == BASE_STATION_ID)
      return;
#endif

#ifdef USE_DETECTOR
    /* Perform actions related to special handling */ 
    if (detectorState == STATE_SPECIAL_HANDLING //is in special handling 
         && (lof.initialEstNgbr == NULL || ngbrTable[lof.initialEstNgbr->index].numInitSamplesToTake > 0)  //is not trying to put a 
	                                                                                                                                                   //newly initialized neighbor to the table 
        ) {
      //has tried a neighbor
      if (tried != NULL) {
	getRank(tried->index, &newPos);
	atomic {
	  if (newPos >= oldPosition) {
	    if (ngbrTable[tried->index].candidatePotential > 0)
	      ngbrTable[tried->index].candidatePotential--;
	  }
	  tried = NULL;
	}
      }
      //find the next sampling candidate; skip dead node
      atomic {
	if (toTry == NULL) {
	  toTry = lof.ngbr_table_head;
	  hasReachedTableEnd = TRUE;
	}
	else
	  hasReachedTableEnd = FALSE;
	for (i = 0; i < lof.ngbr_table_size && //MAX_NGBR_TABLE_SIZE &&
	                  (ngbrTable[toTry->index].candidatePotential < 1 ||
                                        ngbrTable[toTry->index].liveness == IS_DEAD  ||
                                        toTry->flag == 0 
	                 );
                             i++
	     ) {
	  toTry = toTry->next;
	  if (toTry == NULL) {
	    if (!hasReachedTableEnd) {
	      hasReachedTableEnd = TRUE;
	      if (lof.ngbr_table_head != NULL)
		toTry = lof.ngbr_table_head;
	      else //self-stabilization
		break;
	    }
	    else {
	      i = lof.ngbr_table_size;
	      break;
	    }
	  }
	}
      } //atomic

      if (i >= lof.ngbr_table_size || toTry == NULL) { //termination of this special handling
	atomic {
	  detectorState = STATE_NORMAL;
	  tried = toTry = NULL;
	}

	//if(detector_feedback > 0) { //positive feedback
	if (preSpecialHandlingParent != ngbrTable[lof.ngbr_table_head->index].node_id) {//positive feedback; i.e., parent has changed
#ifdef LOG_LOF_DETECTOR
	  log_lof_detector(7);
#endif

	  //expectedNumPktsToSendAfterSH *= 2;

	  atomic { 
	    //threshold_positive
	    if (threshold_positive <= 0)
	      threshold_positive = detector_threshold;
	    else if (!probIntoSH) 
	      threshold_positive = threshold_positive - (threshold_positive>>compPastW) + (last_entry_threshold>>compPastW);

	    if (probIntoSH) { //detector_threshold (decrease)
	      probIntoSH = FALSE;
	      detector_threshold = detector_threshold - (detector_threshold>>compPastW) + (threshold_positive>>compPastW);
	    }
#ifdef NO_BELIEF_PROPAGATION
	    if (detector_initiator == DETECTOR_INITIATOR_SELF) {//initiate belief propagation 
	      whetherToPropagate = TO_PROPAGATE;
	      beliefSrc = TOS_LOCAL_ADDRESS;
	      if (timeToNextSeqNo <= 0) {
		beliefSeqNo++;
		timeToNextSeqNo = DETECTOR_SEQ_NO_INC_INTERVAL;
	      }
	      propagatedHops = 0;
	    }
	    else if (detector_initiator == DETECTOR_INITIATOR_OTHERS_THRESHOLD_TOO_HIGH) {
	                                                                                           //i.e., Should have detected, therefore decrease detector threshold 
	      if (detector_threshold > MIN_THRESHOLD + THRESHOLD_ADAPT_STEP)
		detector_threshold -= THRESHOLD_ADAPT_STEP;
	      else 
		detector_threshold = MIN_THRESHOLD;
	    }
#endif
	  } //atomic
	} //positive feedback
	else if (detector_initiator == DETECTOR_INITIATOR_SELF) { //negative feedback
#ifdef LOG_LOF_DETECTOR
	  log_lof_detector(9);
#endif  
	  atomic {
	    expectedNumPktsToSendAfterSH = 0;

	    detector_threshold += THRESHOLD_ADAPT_STEP; //detector_threshold
	    if (detector_threshold > MAX_THRESHOLD)
	      detector_threshold = MAX_THRESHOLD;

	    if (probIntoSH)
	      probIntoSH = FALSE;
	  } //atomic
	} //negative feedback

	atomic detector_initiator = DETECTOR_INITIATOR_NO;
      } //termination of this SH
      else { //special sampling of another neighbor just found
	getRank(toTry->index, &oldPosition);
	//move this neighbor to the head of the routing table
	atomic {
	  old_weight = ngbrTable[toTry->index].weight;
	  ngbrTable[toTry->index].weight = 0; 
	}
	putInOrder(lof.ngbr_table_head, toTry);
	atomic ngbrTable[toTry->index].weight = old_weight;
      }
    } //detectorState == STATE_SPECIAL_HANDLING
#endif //USE_DETECTOR

    /* update next-hop forwarder and related routing control variables */
    ptr = lof.ngbr_table_head;
    //need to change parent* 
    if (ptr != NULL && lof.parent_id != ngbrTable[ptr->index].node_id && ngbrTable[ptr->index].liveness == IS_ALIVE) {
      dbg(FLOW_DEBUG, "%d chooses %d as its current next-hop\n", TOS_LOCAL_ADDRESS, ngbrTable[ptr->index].node_id);
      //check whether need to send probe-reply/request 
      atomic {
	if (lof.parent_id == TOS_LOCAL_ADDRESS) { //just found a parent (perhaps after losing a parent)
	  lof.numProbeReplyToSend = NUM_PROBES_EACH_ROUND; //for both geographic and d-v routing
	  if (lof.numRouteWithdrawalToSend > 0) { //have lost a parent before
	    lof.numProbeRequestToSend = lof.numRouteWithdrawalToSend;
	    lof.numRouteWithdrawalToSend = 0;
	  }
	}
	lof.parent_id = ngbrTable[ptr->index].node_id;
#ifdef QUICK_BEACON
	sendQuickBeacon = TRUE;
#endif
      }
      change_parent(lof.parent_id);

#ifdef USE_DETECTOR
      if (expectedNumPktsToSendAfterSH > 0 && lof.parent_id == preSpecialHandlingParent) { //negative temporal feedback
	atomic {
	  expectedNumPktsToSendAfterSH = 0;

	  improvedNetConditionHoldWindow += HOLD_WINDOW_ADAPT_STEP; //detector_threshold
	  if (improvedNetConditionHoldWindow > MAX_IMPROVED_NET_CONDITION_HOLD_WINDOW)
	    improvedNetConditionHoldWindow = MAX_IMPROVED_NET_CONDITION_HOLD_WINDOW;

	  if (probIntoHoldWindow)
	    probIntoHoldWindow = FALSE;  
	} //atomic
      }
#endif

      /* initialize/reset the neighbor-regeneration procedure */
#ifdef USE_NGBR_SWITCH 
      post init_ngbr_switch();
#endif
    }//"change parent"

#ifdef USE_NGBR_SWITCH 
    else if (ptr != NULL && ptr->next != NULL && 
                  lof.pktsBeforeSwitching == 0xffff && 
                  ngbrTable[ptr->index].liveness == IS_ALIVE) //the explored node does perform well 
      post init_ngbr_switch();
#endif
    else if (lof.parent_id != TOS_LOCAL_ADDRESS && (ptr == NULL || ngbrTable[ptr->index].liveness == IS_DEAD)) { 
      //lose parent
      atomic {
#ifdef KEEP_PARENT_EVEN_IF_DEAD
	if (ptr == NULL) //NOTE: the reason why parent_id is not reset when "head only becomes dead" is 
	                                 //               so that a node will not lose its parent because of false positive in blacklisting 
#endif
	  lof.parent_id = TOS_LOCAL_ADDRESS;

	lof.numRouteWithdrawalToSend = NUM_PROBES_EACH_ROUND;
	lof.numProbeReplyToSend = lof.numProbeRequestToSend = 0;

	num_tb_regeneration++;
      }
    } //end of "lose parent"

    dbg(FLOW_DEBUG,"leave route_maintenance() \n");

  }//end of route_maintenance()


  /* Blacklist a neighbor
   *
   * Used in: hb_receive(), data_receive()
   */
  void blacklist_if_exists(uint8_t node_id)
  {
    ngbr_table_index_t *ptr;
    ngbr_metric_index_t *preTp, *tp;

    dbg(FLOW_DEBUG, "perform blacklist_if_exists()\n");

    //check to see if it is in the wait-to-be-initialized list; if yest, remove it
    if (lof.initialEst_head != NULL) {
      preTp = NULL;
      tp = lof.initialEst_head;
      while (tp != NULL) {
	if (tp->node_id == node_id) { //found it
	  atomic {
	    if (tp == lof.initialEst_head) { //is head
	      lof.initialEst_head = lof.initialEst_head->next;
	      if (lof.initialEst_head == NULL)
		lof.initialEst_tail = NULL;
	    }
	    else if (tp == lof.initialEst_tail) { //is tail (and not head)
	      lof.initialEst_tail = preTp;
	      preTp->next = NULL;
	    }
	    else { //in the middle
	      preTp->next = tp->next;
	    }
	  } //atomic

	  return;
	}
	else {
	  preTp = tp;
	  tp = tp->next;
	}
      }
    }

    //check to see if it is under initial estimation; if yes, remove it
    if(lof.initialEstNgbr != NULL) {
      if(ngbrTable[lof.initialEstNgbr->index].node_id==node_id) { //is currently under initial estimation
	atomic {
	  lof.initialEstNgbr->flag=0;
	  lof.initialEstNgbr = NULL;
	}
	move_to_initialEstNgbr();
	return;
      }
    }
  
    //check to see if it is in the neighbor table
    if (lof.ngbr_table_head != NULL) {
      ptr = lof.ngbr_table_head;
      while (ptr != NULL) {
	if(ngbrTable[ptr->index].node_id == node_id)
	  break;	
	else 
	  ptr = ptr->next;
      }
      //if it is in existing neighbor table, mark it as dead
      if (ptr != NULL) {
	atomic {
	  ngbrTable[ptr->index].liveness = IS_DEAD;
#ifdef USE_DETECTOR
	  if (ptr == lof.ngbr_table_head) { //to d/w the case of under special handling
	    tried = toTry = NULL;
#ifdef OPP_TRACKER
	    if (oppTrackingState == STATE_OPP_TRACKING_SH)
	      oppTrackingState = STATE_OPP_TRACKING_NO;
#endif
	  }
#endif
	}
	putInOrder(lof.ngbr_table_head, ptr);
	route_maintenance();
      }
    } //lof.ngbr_table_head != NULL

  }//end of blacklist_if_exists()


  /* Process a newly heard ngbr either through probe or if in D-V-LOF, through beacon
   *
   * Used in: refresh_ngbr_table()
   */
  void handleANewlyHeardNgbr(uint8_t node_id, uint32_t ngbrWeight, uint8_t thisSeqNo
#ifdef MAXIMIZE_HOP_COUNT
			 , uint8_t ngbrNumHops
#elif defined(MAXIMIZE_RELIABILITY)
			, uint16_t ngbrPathReliability
#endif
                                                           )
  {
#ifdef BEACON_BASED_ESTIMATION
    ngbr_table_index_t *ptr, *q;

    //first, tries to create a table entry for performing init. estimation for this node
    if ((ptr = new_ngbr(node_id, ngbrWeight, thisSeqNo)) == NULL)
      return; //no ngbr entry any more

    //put in neighbor table
    atomic {
      ptr->next = NULL;
      if (lof.ngbr_table_head == NULL) //first time the forwarder is setup
	lof.ngbr_table_head = ptr;
      else  {
	q = lof.ngbr_table_head;
	while (q->next != NULL)
	  q = q->next;
	q->next = ptr; //attach to the tail
      }
      lof.ngbr_table_size++;
    }

    //put in the right place: sort by rank(liveness, weight, distance)
    if (lof.ngbr_table_head != ptr)
      putInOrder(lof.ngbr_table_head, ptr);
    //check whether need to change parent
    route_maintenance();

    return;

#else  

#ifdef D_V_LOF
    atomic {
      yetToInitEstList[node_id].ngbr_weight = ngbrWeight;
#ifdef MAXIMIZE_HOP_COUNT
      yetToInitEstList[node_id].ngbr_numHops = ngbrNumHops;
#elif defined(MAXIMIZE_RELIABILITY)
      yetToInitEstList[node_id].ngbr_pathReliability = ngbrPathReliability;
#endif
    }
#endif 
    if (lof.initialEst_head == NULL) {
      atomic {
	lof.initialEst_head = lof.initialEst_tail = &yetToInitEstList[node_id];
	yetToInitEstList[node_id].next = NULL;
      }
    }
    else {
      atomic {
	lof.initialEst_tail->next=&yetToInitEstList[node_id];
	lof.initialEst_tail = lof.initialEst_tail->next;
	yetToInitEstList[node_id].next = NULL;
      }
    }

    move_to_initialEstNgbr();

#endif
  } //end of handleANewlyHeardNgbr()


  /* when receive a probe  packet: both in grographic and d-v routing
   *
   * Used in: hb_receive(), hb_receive_beacon(), snoop.receive()
   */
  void refresh_ngbr_table(uint8_t node_id, uint32_t ngbrWeight, uint8_t thisSeqNo, linkReliability_list_t *lrList, 
		        bool by_snoop, uint8_t rcver_est_src, uint16_t rcver_est_linkReliability
#ifdef MAXIMIZE_HOP_COUNT
		        , uint8_t ngbrHops
#elif defined(MAXIMIZE_RELIABILITY)
		       , uint16_t ngbrReliability
#endif
                                              ) 
  {
    ngbr_table_index_t *ptr;
    ngbr_metric_index_t* rp;
    bool needToAdaptRoute = FALSE;
#ifdef SNOOPING_BASED_RCVER_EST
    uint16_t this_rcver_est_linkReliability;
    uint16_t thisAckReliability;
    uint32_t thisWeight;
    //uint16_t thisLogWeight;
#endif
#if defined(BEACON_BASED_ESTIMATION) || defined(SNOOPING_BASED_RCVER_EST)
    uint8_t ind;
    uint32_t localWeight = 0;
#endif
#ifdef BEACON_BASED_ESTIMATION
    ngbr_table_t * ngbr;
    uint16_t thisReliability;
    bool recalculateLocalWeight = FALSE;
#ifdef USE_RNP
    uint16_t thisRNP, tp;
#endif
#endif

    dbg(FLOW_DEBUG, "heard reply/beacon from %d\n",node_id);

#ifndef BEACON_BASED_ESTIMATION
    //whether initial estimation is being performed for the node
    ptr = lof.initialEstNgbr;
    if (ptr != NULL && ngbrTable[ptr->index].node_id==node_id) { 
#ifdef D_V_LOF
      atomic {
	ngbrTable[ptr->index].ngbr_weight = ngbrWeight;
#ifdef MAXIMIZE_HOP_COUNT
	ngbrTable[ptr->index].numHops = ngbrHops;
#elif defined(MAXIMIZE_RELIABILITY)
	ngbrTable[ptr->index].ngbrPathReliability = ngbrReliability;
	if (ngbrTable[ptr->index].ngbrPathReliability < MIN_RELIABILITY)
	  ngbrTable[ptr->index].ngbrPathReliability = MIN_RELIABILITY;
#endif
      } //atomic
#endif //D_V_LOF
      return;
    }
#endif //BEACON_BASED_ESTIMATION

    //whether has already been in the neighbor table
    ptr = lof.ngbr_table_head;
    while (ptr != NULL) { //search the neighbor table
      if (ngbrTable[ptr->index].node_id == node_id) { //in the table
#ifdef D_V_LOF 
	if ((ngbrWeight != ngbrTable[ptr->index].ngbr_weight 
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
	        || (by_snoop != TRUE &&
#ifdef MAXIMIZE_HOP_COUNT
                                ngbrHops != ngbrTable[ptr->index].numHops
#else
                                ngbrReliability != ngbrTable[ptr->index].ngbrPathReliability
#endif
                               )
#endif //MAXIMIZE_HOP_COUNT || MAXIMIZE_RELIABILITY
	     )
#if defined(SNOOPING_BASED_RCVER_EST) || defined(D_V_SNOOPING)
	     && (by_snoop != TRUE || 
                                   ptr == lof.ngbr_table_head || 
	               ngbrWeight + ngbrTable[ptr->index].weight - ngbrTable[ptr->index].ngbr_weight < ngbrTable[lof.ngbr_table_head->index].weight 
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
	               || (ngbrWeight + ngbrTable[ptr->index].weight - ngbrTable[ptr->index].ngbr_weight < ngbrTable[lof.ngbr_table_head->index].weight + ngbrTable[ptr->index].CI_bound &&
#ifdef MAXIMIZE_HOP_COUNT
                                        ngbrHops > ngbrTable[lof.ngbr_table_head->index].numHops
#else
	                    ngbrReliability > ngbrTable[lof.ngbr_table_head->index].pathReliability
#endif
	                  )
#endif //MAXIMIZE_HOP_COUNT || MAXIMIZE_RELIABILITY
                                 )
#endif //SNOOPING_BASED_RCVER_EST || D_V_SNOOPING
                        ) {
	  atomic {
	    if (ngbrWeight > ngbrTable[ptr->index].ngbr_weight)
	      ngbrTable[ptr->index].weight += (ngbrWeight - ngbrTable[ptr->index].ngbr_weight);
	    else
	      ngbrTable[ptr->index].weight -= (ngbrTable[ptr->index].ngbr_weight - ngbrWeight);

	    ngbrTable[ptr->index].ngbr_weight = ngbrWeight;
#ifdef MAXIMIZE_HOP_COUNT
	    ngbrTable[ptr->index].numHops = ngbrHops;
#elif defined(MAXIMIZE_RELIABILITY)
	    ngbrTable[ptr->index].ngbrPathReliability = ngbrReliability;
	    if (ngbrTable[ptr->index].ngbrPathReliability < MIN_RELIABILITY) {
	      ngbrTable[ptr->index].ngbrPathReliability = MIN_RELIABILITY;
	      ngbrTable[ptr->index].pathReliability = MIN_RELIABILITY;
	    }
	    else {
	      ngbrTable[ptr->index].pathReliability = ((uint32_t)ngbrReliability * (uint32_t)ngbrTable[ptr->index].data_reliability)
                                                                                                     >> LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT;
	      if (ngbrTable[ptr->index].pathReliability < MIN_RELIABILITY)
		ngbrTable[ptr->index].pathReliability = MIN_RELIABILITY;
	    }
#endif
	  } //atomic

	  needToAdaptRoute = TRUE; 
	}

#ifdef SNOOPING_BASED_RCVER_EST
	//check to see if receiver-estimated data-reliability is carried
	if (by_snoop == TRUE) { //snooped packet
	  if (rcver_est_src == (uint8_t)TOS_LOCAL_ADDRESS) {
	    this_rcver_est_linkReliability = rcver_est_linkReliability;
	    goto begin_rcver_est;
	  }
	  else
	    goto done_rcver_est;
	}
	else {//regular beacon-control packet
	  for (ind = 0; lrList != NULL && ind < MAX_ENTRIES_CARRIED_PER_HB; ind++) {
	    if (lrList[ind].src == (uint8_t)TOS_LOCAL_ADDRESS) {//find information for myself
	      this_rcver_est_linkReliability = lrList[ind].linkReliability;
	      goto begin_rcver_est;
	    }
	  }
	  goto done_rcver_est;
	}

      begin_rcver_est:
	;
#ifdef DEBUG_ONLY
	    if (numSH2 < 255)
	      numSH2++;
#endif
	    if (node_id == lof.parent_id) { //from current parent
	      //thisAckReliability
	      if (ngbrTable[ptr->index].link_reliability < this_rcver_est_linkReliability)
		thisAckReliability = ((uint32_t)(ngbrTable[ptr->index].link_reliability) << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / this_rcver_est_linkReliability;
	      else 
		thisAckReliability = PERFECT_RELIABILITY;
	      //EWMA
	      if (ngbrTable[ptr->index].ackReliability == (2 << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT))
		ngbrTable[ptr->index].ackReliability = thisAckReliability;
	      else {
		ngbrTable[ptr->index].ackReliability = (ngbrTable[ptr->index].ackReliability - 
 	 			                                  (ngbrTable[ptr->index].ackReliability >> compPastW)
							) + (thisAckReliability >> compPastW);
	      }
	      //for extremely low ackReliability
	      if (ngbrTable[ptr->index].ackReliability < MIN_RELIABILITY)  //set as 0.1%
		ngbrTable[ptr->index].ackReliability = MIN_RELIABILITY;

	      dbg(KEY_DEBUG, "ngbr-id = %d, ackReliability = %d\n", node_id, ngbrTable[ptr->index].ackReliability);
	    }
	    else {//FROM a neighbor that is NOT the current parent 
	      dbg(KEY_DEBUG, "ngbr-id = %d, linkReliability = %d, ackReliability = %d\n", node_id, this_rcver_est_linkReliability, ngbrTable[ptr->index].ackReliability);

	      //integrity check
	      if (this_rcver_est_linkReliability <= 0)
		break;

	      //ackReliability
	      if (ngbrTable[ptr->index].ackReliability == (2 << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)) {
		if (ngbrTable[ptr->index].link_reliability < this_rcver_est_linkReliability)
		  ngbrTable[ptr->index].ackReliability = ((uint32_t)(ngbrTable[ptr->index].link_reliability) << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / this_rcver_est_linkReliability;
		else 
		  ngbrTable[ptr->index].ackReliability = PERFECT_RELIABILITY;
	      }
	      if (ngbrTable[ptr->index].ackReliability < MIN_RELIABILITY)  //set as 0.1%
		ngbrTable[ptr->index].ackReliability = MIN_RELIABILITY;

#ifdef NOT_CONSIDER_ACK_RELIABILITY
	      ngbrTable[ptr->index].ackReliability = PERFECT_RELIABILITY;
#endif

	      //localWeight
	      //localWeight = ((uint32_t)((((uint32_t)NT_SCALE_FACTOR)<<LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / this_rcver_est_linkReliability) << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)) * SCALE_TEN / ngbrTable[ptr->index].ackReliability;
	      localWeight = ((uint32_t)((((uint32_t)NT_SCALE_FACTOR)<<LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / this_rcver_est_linkReliability) << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT+3)) / ngbrTable[ptr->index].ackReliability;
	      //check whether need to accept this new receiver-estimation
	      thisWeight = ngbrWeight + localWeight;
	      if (//thisWeight != ngbrTable[ptr->index].weight //???
              	           (thisWeight < ngbrTable[lof.ngbr_table_head->index].weight  
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
	             || (thisWeight < ngbrTable[lof.ngbr_table_head->index].weight + ngbrTable[ptr->index].CI_bound &&
#ifdef MAXIMIZE_HOP_COUNT
                                     ngbrHops > ngbrTable[lof.ngbr_table_head->index].numHops
#else
                                     ngbrReliability > ngbrTable[lof.ngbr_table_head->index].pathReliability
#endif                                     
	                )
#endif //MAXIMIZE_HOP_COUNT || MAXIMIZE_RELIABILITY
	           ) //better than the current route //???
	            /*old
	           //&& thisWeight < ngbrTable[ptr->index].weight && //improved route quality	           
	           //&& (ngbrTable[ptr->index].logWeight - thisLogWeight) > 3 * ngbrTable[ptr->index].devLogWeight /improved significantly
	           //&& (ngbrTable[lof.ngbr_table_head->index].weight - thisWeight) > ngbrTable[lof.ngbr_table_head->index].devWeight //significantly better than the current route
	           */
	         ) {
		ngbrTable[ptr->index].weight = thisWeight;
		ngbrTable[ptr->index].ngbr_weight = ngbrWeight;
#ifdef MAXIMIZE_HOP_COUNT
		ngbrTable[ptr->index].numHops = ngbrHops;
#elif defined(MAXIMIZE_RELIABILITY)
		ngbrTable[ptr->index].ngbrPathReliability = ngbrReliability;
		if (ngbrTable[ptr->index].ngbrPathReliability < MIN_RELIABILITY) {
		  ngbrTable[ptr->index].ngbrPathReliability = MIN_RELIABILITY;
		  ngbrTable[ptr->index].pathReliability = MIN_RELIABILITY;
		}
		else {
		  ngbrTable[ptr->index].pathReliability = ((uint32_t)ngbrReliability * (uint32_t)ngbrTable[ptr->index].data_reliability) >> LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT;
		  if (ngbrTable[ptr->index].pathReliability < MIN_RELIABILITY)
		    ngbrTable[ptr->index].pathReliability = MIN_RELIABILITY;
		}
#endif
		ngbrTable[ptr->index].link_reliability = (((uint32_t)this_rcver_est_linkReliability * 
					                  (uint32_t)ngbrTable[ptr->index].ackReliability
                                                                                                                     ) >> LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT
                                                                                                                   );
		needToAdaptRoute = TRUE;
#ifdef DEBUG_ONLY
		numSH++;
#ifdef LOG_LOF_DETECTOR
		log_lof_detector(6);
#endif
#endif
	      }

	      dbg(KEY_DEBUG, "ngbr-id = %d, weight = %d, localWeight = %d\n", node_id, ngbrTable[ptr->index].weight, localWeight);
	    } //from a non-parent neighbor

      done_rcver_est:
	;
#endif //SNOOPING_BASED_RCVER_EST

#ifdef BEACON_BASED_ESTIMATION
	ngbr = &(ngbrTable[ptr->index]);
	ngbr->ttl = TTL_BEFORE_HEARING_ANOTHER_BEACON;
#ifndef USE_RNP
#ifdef CONSIDER_FORWARD_RELIABILITY
	//check to see if forward reliability is carried
	for (ind = 0; lrList != NULL && ind < MAX_ENTRIES_CARRIED_PER_HB; ind++) {
	  if (lrList[ind].src == (uint8_t)TOS_LOCAL_ADDRESS && ngbr->forwardReliability != lrList[ind].linkReliability){
	    recalculateLocalWeight = TRUE;
	    if (lrList[ind].linkReliability > MIN_RELIABILITY)
	      ngbr->forwardReliability = lrList[ind].linkReliability;
	    else 
	      ngbr->forwardReliability = MIN_RELIABILITY; //set it as a 1%
	    break;
	  }
	}
#endif
	//calculate the link reliability at this instance in the current period
	ngbr->numSucc++;
	ngbr->numSend += (thisSeqNo >= ngbr->last_hb_seqNo)?(thisSeqNo - ngbr->last_hb_seqNo):
	                                                                                                             (128 - ngbr->last_hb_seqNo + thisSeqNo);
	ngbr->last_hb_seqNo = thisSeqNo;
	if (ngbr->numSend < ngbr->numSucc)   //stabilizing 
	  ngbr->numSend = ngbr->numSucc = 0;

	if (//(ngbr->link_reliability == 0 && ngbr->numSend > 3) || //speed up the initial estimation
	      ngbr->numSend >= LINK_RELIABILITY_CALC_BASE_NUM //another period
	    ) {
	  recalculateLocalWeight = TRUE;
	  //weighted average
	  thisReliability = (ngbr->numSucc << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)/ngbr->numSend;
	  if (ngbr->link_reliability > 0)
	    ngbr->link_reliability = (ngbr->link_reliability - (ngbr->link_reliability >> compPastW)) + 
	                                                (thisReliability >> compPastW);
	  else 
	    ngbr->link_reliability = thisReliability;
	  //for very low reliability
	  if (ngbr->link_reliability < MIN_RELIABILITY) 
	    ngbr->link_reliability = MIN_RELIABILITY; 
#ifdef ASSUME_SYMMETRIC_LINK
	  ngbr->forwardReliability = ngbr->link_reliability;
#endif
	  //liveness control
	  if (ngbr->link_reliability < MIN_LIVING_NGBR_LINK_RELIABILITY)
	    ngbr->liveness = IS_DEAD;
	  else if (ngbr->liveness == IS_DEAD)
	    ngbr->liveness = IS_ALIVE;
	  //start another calculation period as need be
	  if (ngbr->numSend >= LINK_RELIABILITY_CALC_BASE_NUM)
	    ngbr->numSend = ngbr->numSucc = 0;
	}

	//calc. weight
	if (recalculateLocalWeight == TRUE && ngbr->link_reliability > 0 && ngbr->forwardReliability > 0) {
	  localWeight = (((((uint32_t)NT_SCALE_FACTOR)<<LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) /
		             ngbr->link_reliability
		           //) << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)) * SCALE_TEN
		           ) << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT+3)
		         ) / ngbr->forwardReliability;
	  ngbr->weight = ngbr->ngbr_weight + localWeight;
	  needToAdaptRoute = TRUE;
	}
#else //USE_RNP
	tp = (thisSeqNo >= ngbr->last_hb_seqNo)?(thisSeqNo - ngbr->last_hb_seqNo):
	                                                                                (128 - ngbr->last_hb_seqNo + thisSeqNo);
	thisRNP = (tp + 1) / 2;
	//weighted average
	if (ngbr->link_reliability > 0)
	  ngbr->link_reliability = (ngbr->link_reliability - (ngbr->link_reliability >> compPastW)) + 
	    (thisRNP >> compPastW);
	else 
	  ngbr->link_reliability = thisRNP;
	//liveness control
	if (ngbr->liveness == IS_DEAD)
	  ngbr->liveness = IS_ALIVE;
	//update weight
	ngbr->weight = ngbr->ngbr_weight + ngbr->link_reliability;
	needToAdaptRoute = TRUE;
#endif //USE_RNP

	dbg(KEY_DEBUG, "ngbr-id= %d, ngbr->link_reliability = %d, ngbr->forwardReliability = %d, localWeight = %d, ngbr->weight = %d\n", ngbr->node_id, ngbr->link_reliability, ngbr->forwardReliability, localWeight, ngbr->weight);
#endif //BEACON_BASED_ESTIMATION

#endif //D_V_LOF

#ifndef BEACON_BASED_ESTIMATION 
	if (ngbrTable[ptr->index].liveness == IS_DEAD)  {//to bring this node up 
	  atomic {
	    ngbrTable[ptr->index].liveness = IS_ALIVE;
	    ngbrTable[ptr->index].consecutive_tx_failure = 0;
	    ngbrTable[ptr->index].numSend = ngbrTable[ptr->index].numSucc = 1;
	    ngbrTable[ptr->index].link_reliability = 0; //re-start a cycle, and forget history
#if defined(KEEP_PARENT_EVEN_IF_DEAD) && !defined(D_V_LOF)
	    if (ptr == lof.ngbr_table_head && lof.numProbeReplyToSend <= 0)
	      lof.numProbeReplyToSend = NUM_PROBES_EACH_ROUND;
#endif
	  }
	  needToAdaptRoute = TRUE;
#ifdef DEBUG_ONLY 
	  numReplyBasedNgbrRevitalization++;
#endif
	} //bring this node up
#endif //BEACON_BASED_ESTIMATION 

	if (needToAdaptRoute) {
#ifdef USE_DETECTOR
	  if ((detectorState == STATE_SPECIAL_HANDLING && (tried != NULL || toTry !=NULL))
#ifdef OPP_TRACKER
                           || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
                          ) {
	    atomic {
	      tried = toTry = NULL;
#ifdef OPP_TRACKER
	      oppTrackingState = STATE_OPP_TRACKING_NO;
#endif
	    }
	    putInOrder(lof.ngbr_table_head, lof.ngbr_table_head);
	  }
#endif //USE_DETECTOR
	  putInOrder(lof.ngbr_table_head, ptr);
	  route_maintenance();
	}

	return;
      }
      else //search the next neighbor in the table
	ptr = ptr->next;
    }   //search the neighbor table

#ifndef BEACON_BASED_ESTIMATION
    //yet to perform initial link estimation
    rp = lof.initialEst_head;		
    while (rp != NULL) {
      if (rp->node_id == node_id) {			
#ifdef D_V_LOF
	atomic {
	  rp->ngbr_weight = ngbrWeight;
#ifdef MAXIMIZE_HOP_COUNT
	  rp->ngbr_numHops = ngbrHops;
#elif defined(MAXIMIZE_RELIABILITY)
	  rp->ngbr_pathReliability = ngbrReliability;
#endif
	}
#endif //D_V_LOF
	return;
      }
      else
	rp = rp->next;
    }
#endif //BEACON_BASED_ESTIMATION

    //a newly heard node
    handleANewlyHeardNgbr(node_id, ngbrWeight, thisSeqNo
#ifdef MAXIMIZE_HOP_COUNT
		              , ngbrHops
#elif defined(MAXIMIZE_RELIABILITY)
		              , ngbrReliability
#endif
                                                    );

  }//end of refresh_ngbr_table()


  //-------------------------------------------------------------------------------------------------------//
  /* 1) Adapt estimation of link properties;
   * 2) Detect significant change (i.e., bad => good) in network condition, and handle it accordingly.
   *
   * Used in: refresh_ngbr_table_beacon(), initialLinkEstimation(), process_tx(), process_txexc()
   *
   * @param isInitialLinkEst Whether this is for initial link estimation
   * 
   * Returns TRUE if need to adapt route, and FALSE otherwise.
   */
  bool link_estimation(ngbr_table_index_t * head, uint8_t status, uint32_t mac_delivery_latency, uint8_t tx_count, uint8_t isInitialLinkEst)
  {
    ngbr_table_t* ngbr;

#ifdef OPP_TRACKER
    ngbr_table_index_t* ptr;
#endif

#ifdef USE_AGE_FACTOR
    uint32_t seq_diff; 
#endif 

    uint32_t oldWeight = 0;

    uint16_t thisReliability = 0;

#ifdef MAXIMIZE_RELIABILITY
    uint16_t thisDataReliability = 0;
    uint16_t prevDataReliability;
    bool adaptedPathReliability = FALSE;

    uint32_t tp_sqrt;
#ifndef D_V_LOF
    uint16_t tp;
#endif
#endif

#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
    uint32_t theoreticalDev = 0;
#endif

#ifndef USE_NGBR_SWITCH
    uint32_t thisDevWeight;
#else
    uint16_t thisLogWeight, thisDevLogWeight;
#endif

    uint32_t thisLatency = 0;
    uint32_t thisWeight;

#ifdef USE_DETECTOR
    uint8_t scaleFactor;
    uint32_t oldDetectorLQI, thisDetectorLQI, thisDevDetectorLQI;
    uint8_t i;
#endif

#ifdef QUICK_BEACON
    uint32_t weightChange;
#endif

#if defined(USE_DATA_DRIVEN_ETX) || defined(USE_WINDOW_NT)
    bool needToAdaptRoute = FALSE;
#else
    bool needToAdaptRoute = TRUE;
#endif

#ifdef USE_PER_PHY_TX_PER_NADV
    uint8_t ind;
#endif

    //dbg(KEY_DEBUG, "enter link_estimation()\n");

    if (head == NULL)
      return FALSE;

    atomic {
      ngbr = &ngbrTable[head->index];

      oldWeight = ngbr->weight;

#ifdef USE_AGE_FACTOR
      seq_diff = lof.last_seqno - ngbr->seq_no; 
      ngbr->seq_no = lof.last_seqno; 
#endif

#ifdef MAXIMIZE_RELIABILITY
      /* local unicast reliability */
      ngbr->numDataSend++;
      if (status == 1)
	ngbr->numDataSucc++;
      if (ngbr->numDataSend < ngbr->numDataSucc)   //stabilizing 
	ngbr->numDataSend = ngbr->numDataSucc;

      if ((ngbr->numDataSend > UNICAST_RELIABILITY_CALC_BASE_NUM || //another window
             (isInitialLinkEst == 1 && 
               lof.initialEstNgbr!=NULL && ngbrTable[lof.initialEstNgbr->index].numInitSamplesToTake <= 0 &&
               ngbr->data_reliability <= 0
             ) //finished initial estimation yet not finished one est.
            ) && ngbr->numDataSend > 0
          ) { //adapt est. once, and start another calculation period
	thisDataReliability = (ngbr->numDataSucc << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)/ngbr->numDataSend;
	prevDataReliability = ngbr->data_reliability;
	//EWMA of link reliability
	if (ngbr->data_reliability > 0)
	  ngbr->data_reliability = (ngbr->data_reliability - (ngbr->data_reliability >> compPastW)) +
	                                               (thisDataReliability >> compPastW);
	else   //first calc
	  ngbr->data_reliability = thisDataReliability;

	//start another calculation period
	ngbr->numDataSend = ngbr->numDataSucc = 0;

	if (ngbr->data_reliability < MIN_RELIABILITY) {
	  ngbr->data_reliability = MIN_RELIABILITY;
	  ngbr->pathReliability = MIN_RELIABILITY;

	  adaptedPathReliability =TRUE;
	  goto done_data_reliability_est;
	}

	//adapt pathReliability
	if (prevDataReliability != ngbr->data_reliability) {
#ifdef D_V_LOF
	  ngbr->pathReliability = ((uint32_t)ngbr->ngbrPathReliability * (uint32_t)ngbr->data_reliability)
                                                                   >> LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT;
#else
	  ngbr->pathReliability = PERFECT_RELIABILITY;
	  for (tp=1; tp <= ngbr->estGeoHops && ngbr->pathReliability > 0 ; tp++) 
	    ngbr->pathReliability = ((uint32_t)ngbr->pathReliability * (uint32_t)ngbr->data_reliability) >> 
                                                                    LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT;
#endif
	  if (ngbr->pathReliability < MIN_RELIABILITY)
	    ngbr->pathReliability = MIN_RELIABILITY;

	  adaptedPathReliability =TRUE;
	}
      done_data_reliability_est:
	;
      }//adapt est. once ...
#endif //MAXIMIZE_RELIABILITY

#ifdef USE_PER_PHY_TX_PER_NADV
      /* link reliability or PER */
      //EWMA
      for (ind=1; ind < tx_count; ind++)
	ngbr->link_reliability = (ngbr->link_reliability - (ngbr->link_reliability >> compPastW)) + 
	                                            (1 << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT - compPastW));
      if (status == 1)
	ngbr->link_reliability = ngbr->link_reliability - (ngbr->link_reliability >> compPastW);
      else 
	ngbr->link_reliability = (ngbr->link_reliability - (ngbr->link_reliability >> compPastW)) + 
	                                            (1 << (LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT - compPastW));
      //
      if (ngbr->link_reliability == PERFECT_RELIABILITY)
	ngbr->link_reliability = PERFECT_RELIABILITY - 1;
#else

      //calculate the link reliability at this instance in the current period
#ifdef USE_DETECTOR
      if (detectorState == STATE_SPECIAL_HANDLING
#ifdef OPP_TRACKER
           || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
          ) {
	scaleFactor = LINK_RELIABILITY_CALC_BASE_NUM / tx_count + 
	                          ((LINK_RELIABILITY_CALC_BASE_NUM%tx_count == 0)? 0 : 1);
	ngbr->numSend += tx_count * scaleFactor; //scale up
	if (status == 1)
	  ngbr->numSucc += scaleFactor;

	goto inSH;
      }
#endif
      ngbr->numSend += tx_count; 
      if (status == 1) 
	ngbr->numSucc++;
      if (ngbr->numSend < ngbr->numSucc)   //stabilizing 
	ngbr->numSend = ngbr->numSucc;
#ifdef USE_DETECTOR
    inSH:
      ;
#endif

      if ((ngbr->numSend > LINK_RELIABILITY_CALC_BASE_NUM || //another window
             (ngbr->numSend >= CONSECUTIVE_SUCCESS_CALC_BASE && ngbr->numSend == ngbr->numSucc) || //very good link
             status != 1 || //has failed 
             (isInitialLinkEst == 1 && 
               lof.initialEstNgbr!=NULL && ngbrTable[lof.initialEstNgbr->index].numInitSamplesToTake <= 0 &&
               ngbr->link_reliability <= 0
             ) //finished initial estimation yet not finished one est.
#ifdef USE_DETECTOR
            || detectorState == STATE_SPECIAL_HANDLING
#ifdef OPP_TRACKER
            || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
#endif
            ) && ngbr->numSend > 0
          ) { //adapt est. once, and start another calculation period
#ifdef USE_WINDOW_NT
	if (ngbr->numSucc > 0) {
	  //thisLatency = (ngbr->numSend * NT_SCALE_FACTOR) / ngbr->numSucc;
	  thisLatency = ((ngbr->numSend + (ngbr->numSend << 1)) << 13) / ngbr->numSucc;
#endif

	  thisReliability = (ngbr->numSucc << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)/ngbr->numSend; 

	  //weighted average of link reliability
	  if (ngbr->link_reliability > 0) { //EWMA
#ifdef USE_DETECTOR
	    if (detectorState == STATE_SPECIAL_HANDLING
#ifdef OPP_TRACKER
                              || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
                            ) {
	      ngbr->link_reliability = (ngbr->link_reliability - (ngbr->link_reliability >> comp_detector_pastW)) + 
		(thisReliability >> comp_detector_pastW);
	      goto inSH3;
	    }
#endif
	    ngbr->link_reliability = (ngbr->link_reliability - (ngbr->link_reliability >> compPastW)) + 
	                                                (thisReliability >> compPastW);
	    //#endif
#ifdef USE_DETECTOR
	  inSH3:
	    ;
#endif
	  }
	  else  //first calc
	    ngbr->link_reliability = thisReliability;

	  if (ngbr->link_reliability < MIN_RELIABILITY)
	    ngbr->link_reliability = MIN_RELIABILITY;

#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
	  if (ngbr->n <= 0)
	    ngbr->n = NUM_INIT_SAMPLES;
	  if (ngbr->link_reliability > 0) { //use theoretical standard deviation sqrt(1-p)/p as the sample standard variation
	    tp_sqrt = LOF_sqrt((PERFECT_RELIABILITY - ngbr->link_reliability)
                                                              << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT
			);
	    theoreticalDev = ((tp_sqrt + (tp_sqrt << 1)) << 13) / ngbr->link_reliability;
	    /*
	    theoreticalDev = NT_SCALE_FACTOR
	                                 * LOF_sqrt((PERFECT_RELIABILITY - ngbr->link_reliability)
                                                                             << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT
			               )
	                                 /ngbr->link_reliability;
	    */
	  }
#ifdef D_V_LOF
	  //theoreticalDev *= SCALE_TEN;
	  theoreticalDev <<= 3;
#else
	  theoreticalDev *= ngbr->numHops;
#endif
#endif //MAXIMIZE_HOP_COUNT || MAXIMIZE_RELIABILITY

	  //start another calculation period
#ifdef USE_DETECTOR
	  if (detectorState == STATE_SPECIAL_HANDLING
#ifdef OPP_TRACKER
                            || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
                          ) {
	    if (ngbr->numSend > 150) {
	      ngbr->numSend = tx_count * scaleFactor;
	      ngbr->numSucc = scaleFactor;
	    }
	    goto inSH2;
	  }
#endif
	  ngbr->numSend = ngbr->numSucc = 0;
#ifdef USE_DETECTOR
	inSH2:
	  ;
#endif

#if defined(USE_DATA_DRIVEN_ETX) || defined(USE_WINDOW_NT)
	  needToAdaptRoute = TRUE;
#endif

#if defined(USE_LETX_OR_WNT_AS_DETECTOR) && !defined(USE_WINDOW_NT)
	  if (thisReliability < MIN_RELIABILITY)
	    thisReliability = MIN_RELIABILITY;
#endif

#ifdef USE_WINDOW_NT
	}
#endif

      } //adapt est. once ...

#endif //USE_PER_PHY_TX_PER_NADV
    } //atomic

    if (needToAdaptRoute == TRUE) {
      /* MAC delivery latency & weight*/
      //this latency 
#ifdef USE_DATA_DRIVEN_ETX
      if (ngbr->link_reliability > 0)
	thisLatency = (NT_SCALE_FACTOR << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)/ngbr->link_reliability;
      else 
	thisLatency = NT_SCALE_FACTOR;
#elif defined(USE_PER_PHY_TX_PER_NADV)
      thisLatency = (NT_SCALE_FACTOR << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / 
 	            (PERFECT_RELIABILITY - ngbr->link_reliability);
#elif !defined(USE_WINDOW_NT)

#ifdef FB_PENALIZE_UPON_FAILURE 
      if (status == 1)
	thisLatency = mac_delivery_latency;
      else {
	if (ngbr->link_reliability > 0)
	  thisLatency = mac_delivery_latency  + (NT_SCALE_FACTOR << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT)/ngbr->link_reliability;
	else
	  thisLatency = mac_delivery_latency * 2;
      }
#else
      thisLatency = mac_delivery_latency;
#endif 

#endif  //USE_DATA_DRIVEN_ETX


      //this weight and its log 
#ifdef D_V_LOF 
      //thisWeight = ngbr->ngbr_weight + thisLatency * SCALE_TEN;
      thisWeight = ngbr->ngbr_weight + (thisLatency << 3);
#else
      thisWeight = ngbr->numHops * thisLatency;
#endif
#ifdef USE_NGBR_SWITCH
      //thisLogWeight = LOF_log(thisWeight) * SCALE_HUNDRED;
      thisLogWeight = LOF_log(thisWeight) << 6;
#endif

      //EWMA of weight, logWeight
      if (ngbr->weight <= 0) { //the first sample 
	ngbr->weight = thisWeight; 
#ifndef USE_NGBR_SWITCH
	thisDevWeight = 0;
#else
	ngbr->logWeight = thisLogWeight; 
	ngbr->devLogWeight = 0; 
#endif
      }
      else { //not the first sample
#ifndef USE_NGBR_SWITCH
	thisDevWeight = thisWeight > ngbr->weight ? (thisWeight - ngbr->weight) : (ngbr->weight - thisWeight);
#else
	thisDevLogWeight = thisLogWeight > ngbr->logWeight ? (thisLogWeight - ngbr->logWeight) : (ngbr->logWeight - thisLogWeight);
#endif

#if defined(USE_DATA_DRIVEN_ETX) || defined(USE_PER_PHY_TX_PER_NADV)
	ngbr->weight = thisWeight; 
#ifdef USE_NGBR_SWITCH
	ngbr->logWeight = thisLogWeight; 
#endif
#else 
	//EWMA of weight & logWeight
#ifdef USE_AGE_FACTOR
	if (seq_diff <= 1) {
	  ngbr->weight = (ngbr->weight - (ngbr->weight >> compPastW)) + (thisWeight >> compPastW);
#ifdef USE_NGBR_SWITCH
	  ngbr->logWeight = (ngbr->logWeight - (ngbr->logWeight >> compPastW)) + (thisLogWeight >> compPastW);
#endif
	}
	else if(seq_diff > 31) {
	  ngbr->weight = (ngbr->weight * age_factor[29] + thisWeight * (100 - age_factor[29])) / 100;
#ifdef USE_NGBR_SWITCH
	  ngbr->logWeight = (ngbr->logWeight * age_factor[29] + thisLogWeight * (100 - age_factor[29])) / 100;
#endif
	}
	else {
	  ngbr->weight = (ngbr->weight * age_factor[seq_diff-2] + thisWeight * (100 - age_factor[seq_diff-2])) / 100;
#ifdef USE_NGBR_SWITCH
	  ngbr->logWeight = (ngbr->logWeight * age_factor[seq_diff-2] + thisLogWeight * (100 - age_factor[seq_diff-2])) / 100;
#endif
	}
#else //USE_AGE_FACTOR
	ngbr->weight = (ngbr->weight - (ngbr->weight >> compPastW)) + (thisWeight >> compPastW);
#ifdef USE_NGBR_SWITCH
	ngbr->logWeight = (ngbr->logWeight - (ngbr->logWeight >> compPastW)) + (thisLogWeight >> compPastW);
#endif
#endif  //USE_AGE_FACTOR

#endif //USE_DATA_DRIVEN_ETX

	//EWMA of devLogWeight/devWeight
#ifdef USE_AGE_FACTOR
	if(seq_diff < 10) {//calculate deviation only if the samples are more or less close to one another
#endif
#ifndef USE_NGBR_SWITCH
	  if (ngbr->devWeight <= 0) //first sample
	    ngbr->devWeight = thisDevWeight;
	  else //not the first sample
	    ngbr->devWeight = (ngbr->devWeight - (ngbr->devWeight >> compPastDevW)) + 
	                                          (thisDevWeight >> compPastDevW);
#else
	  if (ngbr->devLogWeight <= 0) //first sample
	    ngbr->devLogWeight = thisDevLogWeight;
	  else //not the first sample
	    ngbr->devLogWeight = (ngbr->devLogWeight - (ngbr->devLogWeight >> compPastDevW)) + 
	      (thisDevLogWeight >> compPastDevW);
#endif
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
	  ngbr->n++;
	  if (ngbr->n == 0)
	    ngbr->n = 10000;
	  //#ifndef USE_NGBR_SWITCH
	  //old: ngbr->CI_bound = ((ngbr->devWeight) << 1) / LOF_sqrt(ngbr->n); //for 95% CI
	  ngbr->CI_bound = (theoreticalDev << 1) / LOF_sqrt(ngbr->n);
	  //#else
	  //	  ngbr->CI_bound = ((ngbr->devLogWeight) << 1) / LOF_sqrt(ngbr->n); //for 95% CI
	  //#endif
	  if (ngbr->CI_bound > MAX_WEIGHT_CI_BOUND)
	    ngbr->CI_bound = MAX_WEIGHT_CI_BOUND;
#endif //MAXIMIZE_HOP_COUNT || MAXIMIZE_RELIABILITY

#ifdef USE_AGE_FACTOR
	}
#endif
      } //not the first sample

      dbg(KEY_DEBUG,
#ifdef MAXIMIZE_HOP_COUNT
               "ngbr: node-id=%d, this_mac_latency=%d, link_reliability=%d, weight=%d, thisReliability=%d, CI_bound=%d, numHops=%d \n"
#elif defined(MAXIMIZE_RELIABILITY)
               "ngbr: node-id=%d, this_mac_latency=%d, link_reliability=%d, weight=%d, thisReliability=%d, CI_bound=%d, data_reliability=%d, ngbrPathReliability=%d, pathReliability=%d, inInitEst=%d \n"
#else
               "ngbr: node-id=%d, this_mac_latency=%d, link_reliability=%d, weight=%d, thisReliability=%d \n"
#endif
#ifdef MAXIMIZE_HOP_COUNT
	  , ngbr->node_id, mac_delivery_latency, ngbr->link_reliability, ngbr->weight, thisReliability, ngbr->CI_bound, ngbr->numHops
#elif defined(MAXIMIZE_RELIABILITY)
	  , ngbr->node_id, mac_delivery_latency, ngbr->link_reliability, ngbr->weight, thisReliability, ngbr->CI_bound, ngbr->data_reliability  
#ifdef D_V_LOF
                      , ngbr->ngbrPathReliability
#else
	  , 0
#endif
	  , ngbr->pathReliability, isInitialLinkEst
#else
	  , ngbr->node_id, mac_delivery_latency, ngbr->link_reliability, ngbr->weight, thisReliability
#endif
       ); //dbg()

#ifdef QUICK_BEACON
      if (ngbr->weight < oldWeight) {
	weightChange = ngbr->weight > oldWeight? (ngbr->weight - oldWeight) : (oldWeight - ngbr->weight);
	if (
#ifdef KEEP_PARENT_EVEN_IF_DEAD
	      lof.ngbr_table_head != NULL && ngbrTable[lof.ngbr_table_head->index].liveness == IS_ALIVE &&
#endif
	      (weightChange << 7) > (oldWeight << QUICK_THRESHOLD)
	  )
	  atomic sendQuickBeacon = TRUE;
      }
#endif //QUICK_BEACON
    } //if (needToAdaptRoute == TRUE) ...

#ifdef USE_DETECTOR 
    if (isInitialLinkEst == 1) //no need for detection; detection is NOT for initial link estimation
      goto done;

    //adapt estimation of LQI in an agile manner (pastW is 3/4 instead of 15/16)
    atomic {
#ifdef USE_LETX_OR_WNT_AS_DETECTOR
#ifdef USE_WINDOW_NT
      if (thisLatency <= 0)
	thisDetectorLQI = ngbr->detectorLQI;
      else 
	thisDetectorLQI = thisLatency;
#else 
      if (thisReliability <= 0)
	thisDetectorLQI = ngbr->detectorLQI;
      else
	thisDetectorLQI = (NT_SCALE_FACTOR << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT) / 
                                                        thisReliability;
#endif

#else
      thisDetectorLQI = mac_delivery_latency;
#endif 

      oldDetectorLQI = ngbr->detectorLQI;
      if (thisDetectorLQI != oldDetectorLQI) { //need to adapt detectorLQI
	if (ngbr->detectorLQI == 0)
	  ngbr->detectorLQI = thisDetectorLQI;
	else if (seq_diff <= 1)
	  ngbr->detectorLQI = ngbr->detectorLQI - (ngbr->detectorLQI >> comp_detector_pastW) + 
	                                          (thisDetectorLQI >> comp_detector_pastW);
	else if (seq_diff > 11)
	  ngbr->detectorLQI = (ngbr->detectorLQI * age_factor_detector[9] + 
			    thisDetectorLQI * (100-age_factor_detector[9])
			  ) / 100;
	else
	  ngbr->detectorLQI = (ngbr->detectorLQI * age_factor_detector[seq_diff-2] + 
			       thisDetectorLQI * (100-age_factor_detector[seq_diff-2])
			  ) / 100;
	//devDetectorLQI 
	thisDevDetectorLQI = (ngbr->detectorLQI > oldDetectorLQI) ? ngbr->detectorLQI - oldDetectorLQI : 
                                                                                                                                             oldDetectorLQI - ngbr->detectorLQI;
	if (ngbr->devDetectorLQI == 0)
	  ngbr->devDetectorLQI = thisDevDetectorLQI;
	else	  
	  ngbr->devDetectorLQI = ngbr->devDetectorLQI - (ngbr->devDetectorLQI >> compPastW) + 
	                                                  (thisDevDetectorLQI >> compPastW);
      }
    } //atomic

    /* actions related to network condition detection  */
    if (lof.ngbr_table_head == NULL || lof.ngbr_table_head->next == NULL) //no need for detection 
      goto done;

    //dbg(KEY_DEBUG, "hello, world --- detector!\n");

    //regular detector
#ifdef OPP_TRACKER
    if (oppTrackingState == STATE_OPP_TRACKING_SH)
      goto processOppTracker;
#endif
    if (detectorState == STATE_NORMAL) { //at normal state
      if (seq_diff < 5 && oldDetectorLQI > ngbr->detectorLQI) {
	atomic {
	  detectorState = STATE_IN_DETECTION;
	  ngbr->last_level_of_entry = oldDetectorLQI;
	  ngbr->candidatePotential = 100 + improvedNetConditionHoldWindow;
	  probIntoSH = probIntoHoldWindow = FALSE;
	}
      }
    }
    else if (detectorState == STATE_IN_DETECTION) { //already in detection
      if (ngbr->detectorLQI > ngbr->last_level_of_entry) //go back to the normal state 
	atomic detectorState = STATE_NORMAL;
      else if ((ngbr->last_level_of_entry - ngbr->detectorLQI) < detector_threshold && //network condition is not good enough yet
                    (ngbr->devDetectorLQI == 0 || (ngbr->last_level_of_entry - ngbr->detectorLQI) < 3 * ngbr->devDetectorLQI) &&
                    (threshold_positive <= 0 || //not prob. into SH yet
                      (ngbr->last_level_of_entry - ngbr->detectorLQI) < threshold_positive ||
	  ((call Random.rand())%100) < 20
	) 
                  ) { //not to perform SH
	atomic ngbr->candidatePotential = 100 + improvedNetConditionHoldWindow;
      }
      else if (ngbr->candidatePotential > 100 &&
	(holdWindow_positive <= 0 ||
                      improvedNetConditionHoldWindow <= holdWindow_positive ||
                      ngbr->candidatePotential > (100 + (improvedNetConditionHoldWindow - holdWindow_positive)) ||
	  ((ngbr->last_level_of_entry - ngbr->detectorLQI) < detector_threshold &&
	    (ngbr->devDetectorLQI == 0 || (ngbr->last_level_of_entry - ngbr->detectorLQI) < 3 * ngbr->devDetectorLQI)
	  ) ||
                      ((call Random.rand())%100) < 20
	)
                  ) { //stability check before SH
	atomic ngbr->candidatePotential--;
      }
      else {
	atomic {
	  detectorState = STATE_SPECIAL_HANDLING;  //jump into the state of "special sampling"
#ifdef OPP_TRACKER
	  oppTrackingState = STATE_OPP_TRACKING_NO;
#endif

	  tried = NULL;
	  toTry = lof.ngbr_table_head->next;

	  if ((ngbr->last_level_of_entry - ngbr->detectorLQI) < detector_threshold &&
	       (ngbr->devDetectorLQI == 0 || (ngbr->last_level_of_entry - ngbr->detectorLQI) < 3 * ngbr->devDetectorLQI)
                          ) //on thresholding 
	    probIntoSH = TRUE;
	  else {
	    if (ngbr->devDetectorLQI != 0 && ngbr->devDetectorLQI < detector_threshold)
	      last_entry_threshold = ngbr->devDetectorLQI;
	    else 
	      last_entry_threshold = detector_threshold;
	  }
	  if (ngbr->candidatePotential > 100) //on temporal stability
	    probIntoHoldWindow = TRUE;

	  preSpecialHandlingParent = ngbrTable[lof.ngbr_table_head->index].node_id;
	  expectedNumPktsToSendAfterSH = 0;
	  detector_initiator = DETECTOR_INITIATOR_SELF;

	  for(i=0; i<MAX_NGBR_TABLE_SIZE; i++) {
	    //reset_neighbor_partially(&ngbrTable[i]);
#ifndef D_V_LOF
	    if (ngbr != &(ngbrTable[i]) && 
	        //ngbr->weight <= ngbrTable[i].numHops * NT_SCALE_FACTOR
	          ngbr->weight <= ((ngbrTable[i].numHops + (ngbrTable[i].numHops << 1)) << 13)
                                                           //if the ngbr will not be better than current parent even if NT = 1
                            ) 
	      ngbrTable[i].candidatePotential = 0; //no need to try out this neighbor
	    else 
#endif
	      ngbrTable[i].candidatePotential = STARTING_CANDIDATE_POTENTIAL;
	  }

#ifdef DEBUG_ONLY
	  numSH++;
	  dbg(KEY_DEBUG,"numSH = %d \n", numSH);
#endif
	}//atomic

#ifdef USE_DATA_DRIVEN_ETX
	needToAdaptRoute = TRUE;
#endif

#ifdef LOG_LOF_DETECTOR
	log_lof_detector(6);
#endif
      }
    } //STATE_IN_DETECTION 
    else if (detectorState == STATE_INFORMED_DETECTION) {//just being informed
      atomic {
	detectorState = STATE_SPECIAL_HANDLING;
#ifdef OPP_TRACKER
	oppTrackingState = STATE_OPP_TRACKING_NO;
#endif

	tried = NULL;
	toTry = lof.ngbr_table_head->next;

	last_entry_threshold = detector_threshold;

	preSpecialHandlingParent = ngbrTable[lof.ngbr_table_head->index].node_id; 
	expectedNumPktsToSendAfterSH = 0;
	if (ngbr->last_level_of_entry - ngbr->detectorLQI < detector_threshold)
	  detector_initiator = DETECTOR_INITIATOR_OTHERS_THRESHOLD_TOO_HIGH;
	else
	  detector_initiator = DETECTOR_INITIATOR_OTHERS;

	for(i=0; i<MAX_NGBR_TABLE_SIZE; i++) {
	  //reset_neighbor_partially(&ngbrTable[i]);
#ifndef D_V_LOF
	    if (ngbr != &(ngbrTable[i]) && 
	          //ngbr->weight <= ngbrTable[i].numHops * NT_SCALE_FACTOR
	          ngbr->weight <= ((ngbrTable[i].numHops + (ngbrTable[i].numHops << 1)) << 13)
                                                                          //if the ngbr will not be better than current parent even if NT = 1
                            ) 
	      ngbrTable[i].candidatePotential = 0; //no need to try out this neighbor
	    else 
#endif
	      ngbrTable[i].candidatePotential = STARTING_CANDIDATE_POTENTIAL_INFORMED;
	}
#ifdef DEBUG_ONLY
	numSH++;
#endif
      } //atomic

#ifdef USE_DATA_DRIVEN_ETX
      needToAdaptRoute = TRUE;
#endif

#ifdef LOG_LOF_DETECTOR
      log_lof_detector(10);
#endif
    }
    else if (detectorState == STATE_SPECIAL_HANDLING) {//in special handling 
      if (toTry != NULL) {
	atomic {
	  //for the neighbor just tried
	  tried = toTry; 
	  if (status !=1 && ngbrTable[tried->index].candidatePotential >0)
	    ngbrTable[tried->index].candidatePotential--;
	  if (oldWeight <= ngbr->weight && ngbrTable[tried->index].candidatePotential >0)
	    ngbrTable[tried->index].candidatePotential--;
	  //set next one to try 
	  toTry = tried->next; //if toTry is NULL, this will be handled in route_maintenance()
	}//atomic
      }

#ifdef USE_DATA_DRIVEN_ETX
      needToAdaptRoute = TRUE;
#endif
    }

    //opportunistic tracker
#ifdef OPP_TRACKER
  processOppTracker:
    if (detectorState == STATE_SPECIAL_HANDLING)
      goto done;

    //dbg(KEY_DEBUG, "hello, world --- opp-tracking!\n");

    if (oppTrackingState == STATE_OPP_TRACKING_NO && ngbr->oppTrackerLQI != 0) {
      if (ngbr->oppTrackerLQI >= ngbr->detectorLQI && ngbr->oppTrackerLQI < oldDetectorLQI) {
	atomic {
	  oppTrackingState = STATE_OPP_TRACKING_IN_DETECTION;
	  oppTrackingPotential = OPP_TRACKING_STABILITY_DETECTOR; 
	}
      }
    }
    else if (oppTrackingState == STATE_OPP_TRACKING_IN_DETECTION) {
      if (ngbr->oppTrackerLQI < ngbr->detectorLQI)
	atomic oppTrackingState = STATE_OPP_TRACKING_NO;
      else {
	if (oppTrackingPotential > 0)
	  oppTrackingPotential--;
	else {
	  //dbg(KEY_DEBUG, "perform opp-tracking\n");
#ifdef DEBUG_ONLY
	  numSH2++;
#endif
	  atomic {
	    oppTrackingState = STATE_OPP_TRACKING_SH;
	    detectorState = STATE_NORMAL;
	    //find the neighbor to try out opportunistically
	    ptr = lof.ngbr_table_head;
	    while (ptr != NULL)
	      if (ngbrTable[ptr->index].node_id == ngbr->oppNode)
		break;
	      else 
		ptr = ptr->next;
	  }
	  //
	  if (ptr == NULL || ngbrTable[ptr->index].liveness == IS_DEAD) {
	    atomic {
	      oppTrackingState = STATE_OPP_TRACKING_NO;
	      ngbr->oppTrackerLQI = 0;
	    }
	  }
	  else { //put the neighbor in the head
	    atomic {
	      thisWeight = ngbrTable[ptr->index].weight;
	      ngbrTable[ptr->index].weight = 0;
	    }
	    putInOrder(lof.ngbr_table_head, ptr);
	    atomic {
	      ngbrTable[ptr->index].weight = thisWeight;
	      ngbrTable[ptr->index].candidatePotential = OPP_TRACKING_STARTING_CANDIDATE_POTENTIAL;
	    }
	    //
	    needToAdaptRoute = TRUE;
	  }
	}
      }
    }
    else if (oppTrackingState == STATE_OPP_TRACKING_SH) {
      //update candidatePotential
      if (ngbr->candidatePotential > 0)
	ngbr->candidatePotential--;
      if (status !=1 && ngbr->candidatePotential >0)
	ngbr->candidatePotential--;
      if (oldWeight <= ngbr->weight && ngbr->candidatePotential >0)
	ngbr->candidatePotential--;
      //
      if (ngbr->candidatePotential > 0)
	needToAdaptRoute = FALSE;
      else 
	needToAdaptRoute = TRUE;
    }
#endif //OPP_TRACKER

  done:
    ;
#endif //USE_DETECTOR

#ifdef LOG_MEAN_VARIANCE
    if (isInitialLinkEst == 0)
      log_mean_var(oldWeight);
#endif

    return (needToAdaptRoute 
#ifdef MAXIMIZE_RELIABILITY
                   || adaptedPathReliability
#endif
                 );

  }//end of link_estimation()


  /*
   * To cancel the neighbor-switching and special handling, if any, that is yet to finish
   *
   * called in: initialLinkEstimation(...)
   */
  /* not used any more
     void cancel_ngbr_switch_and_SH()
     {
     ngbr_table_index_t * head = lof.ngbr_table_head;

     dbg(FLOW_DEBUG, "entering cancel_ngbr_switch_and_SH() \n");

     if (head == NULL)
     return;

     //first put the head into the right position
     head = putInOrder(head, head);
     if (head != lof.ngbr_table_head)
     lof.ngbr_table_head = head;
     //adapt next-hop, if need be
     route_maintenance();

     dbg(FLOW_DEBUG, "leaving cancel_ngbr_switch_and_SH() \n");
     } //end of cancel_ngbr_switch_and_SH(...)
  */


  /* Wrap up the initial estimation for one node
   *
   * Used in: initialLinkEstimation(), hb_timer_task()
   */ 
  void wrapUpOneInitialEst()
  {    
    ngbr_table_index_t * q;
    ngbr_table_index_t * ptr = lof.initialEstNgbr;

#if defined(USE_NGBR_SWITCH) || defined(USE_DETECTOR)
    ngbr_table_index_t * ngbr_head = lof.ngbr_table_head;
    bool tableInOrder = FALSE;
#endif

#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
    ngbr_table_t * ngbr;
    uint32_t theoreticalDev, tp_sqrt;
#endif

    atomic lof.initialEstNgbr = NULL; //NOTE: also serves as the mutual exclusion between 
                                                                    //               initialLinkestimation() and hb_timer_task()

    //If it is currently under neighbor exploration but has not been verified to be well (see route_maintenance()),
    //or if it is currently under special handling, first put the existing table in order before attaching the new neighbor
#ifdef USE_NGBR_SWITCH 
    if (ngbr_head != NULL && ngbr_head->next != NULL && lof.pktsBeforeSwitching == 0xffff) {
      putInOrder(ngbr_head, ngbr_head);
      tableInOrder = TRUE;
    }
#endif
#ifdef USE_DETECTOR
    if ((detectorState == STATE_SPECIAL_HANDLING && (tried != NULL || toTry !=NULL))
#ifdef OPP_TRACKER
          || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
        ) {
      atomic {
	tried = toTry = NULL;
#ifdef OPP_TRACKER
	oppTrackingState = STATE_OPP_TRACKING_NO;
#endif
      }
      if (!tableInOrder)
	putInOrder(ngbr_head, ngbr_head);
    }
#endif

    //find another node to perform initial est.
    move_to_initialEstNgbr(); 

    //put in initialized neighbor table
    atomic {
      ptr->next = NULL;
      if (lof.ngbr_table_head == NULL) //first time the forwarder is setup
	lof.ngbr_table_head = ptr;
      else  {
	q = lof.ngbr_table_head;
	while (q->next != NULL)
	  q = q->next;
	q->next=ptr; //attach to the tail
      }
      lof.ngbr_table_size++;
    }

    //dbg(KEY_DEBUG, "lof.ngbr_table_size = %d \n", lof.ngbr_table_size); 

#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
    ngbr = &(ngbrTable[ptr->index]);
    if (ngbr->n <= 0 && ngbr->link_reliability > 0) {//use theoretical standard deviation sqrt(1-p)/p as the sample standard variation
      ngbr->n = NUM_INIT_SAMPLES;

      tp_sqrt = LOF_sqrt((PERFECT_RELIABILITY - ngbr->link_reliability)
                                            << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT
		  );
      theoreticalDev = ((tp_sqrt + (tp_sqrt << 1)) << 13) / ngbr->link_reliability;
      /*
      theoreticalDev = NT_SCALE_FACTOR
	                 * LOF_sqrt((PERFECT_RELIABILITY - ngbr->link_reliability) 
                                                              << LINK_RELIABILITY_SCALE_FACTOR_LEFT_SHIFT 
		                   ) 
                                       /ngbr->link_reliability; 
      */
#ifdef D_V_LOF
      //theoreticalDev *= SCALE_TEN;
      theoreticalDev <<= 3;
#else
      theoreticalDev *= ngbr->numHops;
#endif

      //#ifndef USE_NGBR_SWITCH
      ngbr->CI_bound = (theoreticalDev << 1) / LOF_sqrt(ngbr->n); //for 95% CI
      //#else 
      //      ; //no good way to jump start the CI_bound due to the LOG operation
      //#endif
      if (ngbr->CI_bound > MAX_WEIGHT_CI_BOUND)
	ngbr->CI_bound = MAX_WEIGHT_CI_BOUND;
    } //CI_bound
#endif //MAXIMIZE_HOP_COUNT || MAXIMIZE_RELIABILITY

#ifdef USE_NGBR_SWITCH    //calculate new probability introduced by ptr
    new_pairprob(ptr);
#endif
 
    //put in the right place: sort by rank(liveness, weight, distance)
    if (lof.ngbr_table_head != ptr)
      putInOrder(lof.ngbr_table_head, ptr);
    //check whether need to change parent
    route_maintenance();

  }//end of wrapUpOneInitialEst()


  /* Peform initial link estimation
   *
   * Used in: fb_receive() 
   */
  void initialLinkEstimation( uint8_t s_status, uint32_t mac_delivery_latency, uint8_t tx_count)
  {
    ngbr_table_t *head;
    ngbr_table_index_t * ptr;

    ptr = lof.initialEstNgbr;
    if (ptr == NULL)
      return;

    head = &ngbrTable[ptr->index];

    /* liveness */	 
    if (s_status == 1) {
      atomic {
	head->liveness = IS_ALIVE;
	head->consecutive_tx_failure = 0;	    
      }
    }
    else {
      atomic head->consecutive_tx_failure++;
      //see if need to skip this neighbor
      if (head->consecutive_tx_failure >= DEAD_IF_FAIL_FIRST_CONSECUTIVE_INIT_ESTS) {
	atomic {
	  ptr->flag = 0;
	  lof.initialEstNgbr = NULL;
	}
	move_to_initialEstNgbr();
	return;
      }
    }

    /* adapt link property estimation */	    
    link_estimation(ptr, s_status, mac_delivery_latency, tx_count, 1);

    //check whether need to set the neighbor as dead
    if (s_status != 1 && head->liveness == IS_ALIVE && head->consecutive_tx_failure > MAX_CONSECUTIVE_FAIL)	  
      head->liveness = IS_DEAD;
    
    /* decide whether to wrap up initial link-estimation for this neighbor */ 
    if (head->numInitSamplesToTake <= 0) //finished initial link-estimation, put it in the active forwarder-candidate
      wrapUpOneInitialEst();
  }//end of initialLinkEstimation()


  /* Upon transmission success */
  void process_tx(uint32_t mac_delivery_latency, uint8_t tx_count)
  {
    //SUCCESS
    ngbr_table_t * head;
    ngbr_table_index_t* ptr = lof.ngbr_table_head;
    bool needToAdaptRoute;
		
    dbg(FLOW_DEBUG, "entering process_tx()\n");
		
    if(ptr == NULL) {
      dbg(FLOW_DEBUG, "lof.ngbr_table_head == NULL in process_tx()\n");
      return;
    }
    head = &ngbrTable[ptr->index];

    atomic {
      head->liveness = IS_ALIVE;
      head->consecutive_tx_failure = 0;
    }

    /* Hongwei
      lof.numRouteWithdrawalToSend = 0;
      lof.numProbeRequestToSend = 0;
    */

    //adapt estimation 
    needToAdaptRoute = link_estimation(ptr, 1, mac_delivery_latency, tx_count, 0);

    /*put in the right place in neighbor list, and adapt the next-hop accordingly */ 	  
    if (needToAdaptRoute == TRUE) {
      putInOrder(ptr, ptr);
      //adapt next-hop, if need be 
      route_maintenance();
    }

#ifdef USE_NGBR_SWITCH
    //prepare for or perform neighbor regeneration
    if (lof.pktsBeforeSwitching != 0xffff && lof.pktsBeforeSwitching > 0) {
      if (lof.initialEstNgbr == NULL 
#ifdef USE_DETECTOR 
           && detectorState != STATE_SPECIAL_HANDLING && detectorState !=  STATE_INFORMED_DETECTION
#ifdef OPP_TRACKER
           && oppTrackingState != STATE_OPP_TRACKING_SH
#endif
#endif
         ) //start exploration process only if there is no node at the stage of initial estimation, and not in detection state 
	lof.pktsBeforeSwitching--;
    }
    if (lof.pktsBeforeSwitching == 0) { //it is time to sample a neighbor probabilistically
      if (lof.initialEstNgbr  != NULL
#ifdef USE_DETECTOR
            || detectorState == STATE_SPECIAL_HANDLING || detectorState ==  STATE_INFORMED_DETECTION
#ifdef OPP_TRACKER
            || oppTrackingState == STATE_OPP_TRACKING_SH
#endif
#endif
           ) //do not try to regenerate any neighbor when the node is doing initial link estimation or SH; 
             //this is to guarantee the neghibor list are ordered by their priority. 
	lof.pktsBeforeSwitching = 1;
      else
	ngbr_switch();
    } //it is time to sample
#endif //USE_NGBR_SWITCH

    dbg(FLOW_DEBUG, "leave process_tx() \n");
  } //end of process_tx()


  /* Upon transmission failure */
  void process_txexc(int32_t mac_delivery_latency, uint8_t tx_count)
  {		
    ngbr_table_t * head;
    ngbr_table_index_t* ptr = lof.ngbr_table_head;
    bool needToAdaptRoute;
		
    dbg(FLOW_DEBUG, "entering process_txexc()\n");
		
    if(ptr == NULL) {
      dbg(FLOW_DEBUG, "lof.ngbr_table_head == NULL in process_txexc()\n");
      return;
    }
  	  
    head = &ngbrTable[ptr->index];
	 
    atomic head->consecutive_tx_failure++;

    /* Adapt l_link-reliability, mac-delivery-latency,  and weight, etc. */
    needToAdaptRoute = link_estimation(ptr, 0, mac_delivery_latency, tx_count, 0);
   
    /* decide whether this neighbor should be declared as dead */
    if (head->liveness == IS_ALIVE &&
         ((head->link_reliability > 0 && head->link_reliability < MIN_LIVING_NGBR_LINK_RELIABILITY)
           || head->consecutive_tx_failure > MAX_CONSECUTIVE_FAIL
         )
        ) {
      atomic head->liveness = IS_DEAD;
      needToAdaptRoute = TRUE;
    }

    /*put in the right place in neighbor list, and adapt the next-hop accordingly */	  
    if (needToAdaptRoute == TRUE) {
      putInOrder(ptr, ptr);
      //adapt next-hop, if need be
      route_maintenance();
    }

    dbg(FLOW_DEBUG, "leave process_txexc()\n");	  	  		
  }//end of process_txexc()


  /* Process MAC feedback */
  void fb_receive(uint8_t status, uint32_t delay, uint8_t tx_count, uint8_t destination, bool isInitEst)
  {
    if (TOS_LOCAL_ADDRESS == BASE_STATION_ID)
      return;

    if (isInitEst == TRUE)
      initialLinkEstimation(status, delay, tx_count);
#ifndef INIT_ESTIMATION_ONLY
    else if (
#ifdef CHECK_FB_ADDR
	     destination == lof.parent_id
#else
	     TRUE
#endif
	     ){
      if(status == 1)
	process_tx(delay, tx_count);
      else if(status == 0)
	process_txexc(delay, tx_count);
    }
#endif //INIT_ESTIMATION_ONLY
  }//end of fb_receive()



  //===============MAIN BODY==========================//
  /* Initialize state;
   *
   *Used in: Control.init() 
   */
  void init_state()
  {
    uint8_t i;
#ifdef USE_NGBR_SWITCH 
    uint8_t j, count; 
#endif

    atomic {
      lof.last_seqno = 0;
			
      if(TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
	lof.numRouteWithdrawalToSend = 0;
	lof.numProbeReplyToSend = NUM_PROBES_EACH_ROUND;	
      }
      else {
	lof.numRouteWithdrawalToSend = NUM_PROBES_EACH_ROUND;
	lof.numProbeReplyToSend = 0;	
      }
      lof.numProbeRequestToSend = 0;

      lof.ngbr_table_size = 0;
#ifdef USE_NGBR_SWITCH
      lof.pktsBeforeSwitching = 0xffff; 
#endif 
      lof.parent_id = TOS_LOCAL_ADDRESS;

      lof.ngbr_table_head = NULL;
      lof.base_loc.x = 0;
      lof.base_loc.y = 0;
		
      lof.initialEstNgbr = NULL;
      lof.initialEst_head = NULL;
      lof.initialEst_tail = NULL;
			
      hb_alternate=0; 
      ngbr_switch_count=0;
      num_tb_regeneration=0;

#ifdef USE_DETECTOR
#ifdef OPP_TRACKER
      oppTrackingState = STATE_OPP_TRACKING_NO;
#endif

      detectorState = STATE_NORMAL;
      tried = toTry = NULL;

      detector_threshold = MIN_THRESHOLD;
      threshold_positive = 0;
      probIntoSH = FALSE;

      preSpecialHandlingParent = 0xFFFF;
      expectedNumPktsToSendAfterSH = 0;
      improvedNetConditionHoldWindow = INIT_IMPROVED_NET_CONDITION_HOLD_WINDOW;
      holdWindow_positive = 0;
      probIntoHoldWindow = FALSE;

      whetherToPropagate = NOT_TO_PROPAGATE;
      detector_initiator = DETECTOR_INITIATOR_NO;
      beliefSeqNo = 0;
      timeToNextSeqNo = DETECTOR_SEQ_NO_INC_INTERVAL;
#endif

      for(i=0; i < MAX_NGBR_TABLE_SIZE; i++) {
	ngbrTableIndex[i].flag = 0; //not used
	ngbrTableIndex[i].index = NIL;
	ngbrTableIndex[i].next = NULL;

	reset_neighbor_partially(&ngbrTable[i]); //the actual neighbor table
      }

#ifndef BEACON_BASED_ESTIMATION
      for(i=0; i <= MAX_NGBR_TO_CONSIDER; i++) {
	yetToInitEstList[i].next=NULL;
	yetToInitEstList[i].node_id=i;
      }
#endif

#ifdef USE_NGBR_SWITCH 
      count=0;			
      for(i=0; i<MAX_NGBR_TABLE_SIZE; i++)
	for(j=i+1; j<MAX_NGBR_TABLE_SIZE; j++) {
	  ngbrPairProb[count].ptr1 = &ngbrTableIndex[i];
	  ngbrPairProb[count].ptr2 = &ngbrTableIndex[j];
	  ngbrPairProb[count].prob = 0;
	  count++;
	}
#endif

#ifdef D_V_LOF
      beaconCounter = 0;
#endif

#ifdef SUPPRESS_DUPLICATE_PKT
      for (i = 0; i < MAX_NUM_CHILDREN; i++) {
	dup_detector[i].child = 0xff;
	dup_detector[i].last_seqNo = 0xff;
      }
#endif

#if  defined(QUICK_BEACON) || defined(SNOOPING_BASED_RCVER_EST)
      sendQuickBeacon = FALSE;
#endif

#ifdef BEACON_BASED_ESTIMATION
      hb_seqNo = 0;
      to_send_index = 0;
#endif

#ifdef SNOOPING_BASED_RCVER_EST
      for (i=0; i < MAX_NUM_NODES_TO_SNOOP; i++) {
	snoop_link_estimator[i].src = FLAG_NOT_INITIALIZED;
	snoop_link_estimator[i].potentialChild = FALSE;
      }
      to_send_index = 0;
      round = 0;
#endif

    } //atomic

    hb_timer_interval = (call Random.rand())%MAX_PROBE_INTERVAL + 300;
    timeElapsedSinceLastTimerFiring = 0;

    set_loc(TOS_LOCAL_ADDRESS, &my_loc);
  }//end of init_state()


  command bool Control.init() {
#ifdef TOSSIM
    /* test correctness of numerical method */
    /*
      uint8_t i;
      for (i=0; i <= 100; i++)
      dbg(KEY_DEBUG, "sqrt(%d) ---> %d;  log(%d) ---> %d \n", i, LOF_sqrt(i), i * 10000, LOF_log(i * 10000)); 
      ;
     */
#endif 

    atomic toStartLOF = FALSE;
    init_state();

#ifdef DEBUG_ONLY 
    numProbeReplySent = numRouteWithdrawalSent = numReplyBasedNgbrRevitalization = 0;
    numSendButNoParent = numFwdButNoParent = 0; 
    numChildren = 0;
    numSH = 0;
    numSH2 = 0;
#endif

#ifndef IN_KANSEI
    call TimerControl.init();
#endif

    return call Random.init();
  }
  
  command bool Control.start() {

    atomic toStartLOF=TRUE;

#ifndef IN_KANSEI
    call TimerControl.start();
#endif

    return call Timer.start(TIMER_REPEAT, TIMER_INTERVAL);
  }
  
  command bool Control.stop() {
#ifndef IN_KANSEI
    return call TimerControl.stop();
#endif

    return SUCCESS;
  }
 

  /* Send initial-estimation packets
  *
  * Used in: hb_timer_task()
  */
  void send_initial_estimation_pkt()
  {
    ngbr_table_index_t  * head = lof.initialEstNgbr;
    lof_heartbeat_t * lof_hb_pkt = (lof_heartbeat_t *)(OutMsg.data+QUEUE_HEAD_LEN);

    if (head == NULL || ngbrTable[head->index].numInitSamplesToTake <= 0)
      return;
	  
    atomic {
      lof_hb_pkt->src = TOS_LOCAL_ADDRESS;
      lof_hb_pkt->type = PROBE_LINK_ESTIMATION;
    }

    if ((call CONTROL_Send.send(ngbrTable[head->index].node_id, LOF_HB_PKT_SIZE, &OutMsg)) == SUCCESS) {
      atomic {
	//ngbrTable[head->index].numSend++;
	ngbrTable[head->index].numInitSamplesToTake--;
	initEstFbDelay = MAX_INIT_EST_FB_DELAY;
      }

      //dbg(KEY_DEBUG, "send_initial_estimation_pkt, candidate_node=%d,numsend=%d,numsucc=%d\n", ngbrTable[head->index].node_id, ngbrTable[head->index].numSend, ngbrTable[head->index].numSucc);
    }

  }//end of send_initial_estimation_pkt()


#ifdef BEACON_BASED_ESTIMATION 
  void ngbr_liveness_checker()
  {
    ngbr_table_index_t *ptr, *tp;
    bool needToAdaptRoute = FALSE;

    ptr = lof.ngbr_table_head;
    while (ptr != NULL && ngbrTable[ptr->index].liveness != IS_DEAD) {
      ngbrTable[ptr->index].ttl--;
      if (ngbrTable[ptr->index].ttl <= 0) {
	dbg(KEY_DEBUG, "neighbor node <%d> is declared as dead\n", ngbrTable[ptr->index].node_id);
	tp = ptr->next;

	ngbrTable[ptr->index].liveness = IS_DEAD;
	ngbrTable[ptr->index].link_reliability -= (ngbrTable[ptr->index].link_reliability >> compPastW);
	putInOrder(lof.ngbr_table_head, ptr);

	needToAdaptRoute = TRUE;
	ptr = tp;
      }
      else 
	ptr = ptr->next;
    }

    if (needToAdaptRoute == TRUE)
      route_maintenance();

  } //end of ngbr_liveness_checker()
#endif


  /* Used in: Timer.fired() */
  uint32_t hb_timer_task(uint32_t interval)
  {
    uint8_t pkt_size;

#ifdef USE_DETECTOR
    lof_heartbeat_t * lof_hb_pkt = (lof_heartbeat_t *)(OutMsg.data+QUEUE_HEAD_LEN);
    lof_detector_t* detector_t;
#endif
 
#ifdef D_V_LOF
    beaconCounter++;
    if (beaconCounter%PROBE_INTERVALS_PER_BEACON == 0) { //another beacon interval
#ifdef BEACON_BASED_ESTIMATION 
      ngbr_liveness_checker();
#endif
      if (lof.numProbeReplyToSend == 0 && 
           (TOS_LOCAL_ADDRESS == BASE_STATION_ID || 
             (lof.parent_id != TOS_LOCAL_ADDRESS 
#ifdef KEEP_PARENT_EVEN_IF_DEAD
               && lof.ngbr_table_head != NULL && ngbrTable[lof.ngbr_table_head->index].liveness == IS_ALIVE
#endif
              )
          )
        ) //send beacon
      SendBeacon();
    }
#endif //D_V_LOF

#ifdef USE_DETECTOR
    if(whetherToPropagate == TO_PROPAGATE ||
       (whetherToPropagate == MAY_PROPAGATE && propHoldTime <= 0)
      ) {
      whetherToPropagate = NOT_TO_PROPAGATE;

      lof_hb_pkt->src = beliefSrc;
      lof_hb_pkt->type = PROBE_DETECTION;

      detector_t = (lof_detector_t*)(OutMsg.data + LOF_HB_PKT_SIZE);
      detector_t->seq = beliefSeqNo;
      detector_t->hops = propagatedHops;

#ifdef LOG_LOF_DETECTOR 
      log_lof_detector(8);
#endif
      call CONTROL_Send.send(TOS_BCAST_ADDR, sizeof(lof_detector_t)+LOF_HB_PKT_SIZE, &OutMsg);
    }
#endif //USE_DETECTOR

    /* Self-stabilization: try to wrap up initial estimation */
    if (lof.initialEstNgbr != NULL && ngbrTable[lof.initialEstNgbr->index].numInitSamplesToTake <= 0) {
      if (initEstFbDelay > 0)
	initEstFbDelay--;
      else {
	if (ngbrTable[lof.initialEstNgbr->index].link_reliability <= 0) {
	  atomic {
	    lof.initialEstNgbr->flag = 0;
	    lof.initialEstNgbr = NULL;
	  }
	  move_to_initialEstNgbr();
	}
	else 
	  wrapUpOneInitialEst();
      }
    }

    /* initial estimation, route withdrawal, probe reply, probe request */
    if (lof.initialEstNgbr != NULL && ngbrTable[lof.initialEstNgbr->index].numInitSamplesToTake > 0 && 
          lof.numRouteWithdrawalToSend > 0) {
      if (hb_alternate == 0) { //send initial estimation packet
	atomic hb_alternate = 1;
	send_initial_estimation_pkt();
      }
      else { //to send route-withdrawals, when need be
	atomic hb_alternate = 0;
	pkt_size = compose_probe_packet(PROBE_WITHDRAWAL, lof.numRouteWithdrawalToSend,OutMsg.data+QUEUE_HEAD_LEN);
	if ((call CONTROL_Send.send(TOS_BCAST_ADDR, pkt_size+QUEUE_HEAD_LEN, &OutMsg)) == SUCCESS)
	  lof.numRouteWithdrawalToSend--;
#ifdef DEBUG_ONLY 
	numRouteWithdrawalSent++;
#endif
      }
      goto hb_done;
    }

    if (lof.initialEstNgbr != NULL && ngbrTable[lof.initialEstNgbr->index].numInitSamplesToTake > 0 && 
          lof.numProbeReplyToSend > 0) {
      if (hb_alternate == 0) {
	atomic hb_alternate=1;
	send_initial_estimation_pkt();
      }
      else {
	atomic hb_alternate = 0;
	pkt_size = compose_probe_packet(PROBE_REPLY, lof.numProbeReplyToSend,OutMsg.data+QUEUE_HEAD_LEN);
	if ((call CONTROL_Send.send(TOS_BCAST_ADDR, pkt_size+QUEUE_HEAD_LEN, &OutMsg)) == SUCCESS)
	  lof.numProbeReplyToSend--;
#ifdef DEBUG_ONLY
	numProbeReplySent++;
#endif
      }
      goto hb_done;
    }

    //normal situation
    if (lof.initialEstNgbr != NULL && ngbrTable[lof.initialEstNgbr->index].numInitSamplesToTake > 0) //initial estimation only
      send_initial_estimation_pkt();
    else if (lof.numRouteWithdrawalToSend > 0) {//route-withdrawals only
      pkt_size = compose_probe_packet(PROBE_WITHDRAWAL, lof.numRouteWithdrawalToSend,OutMsg.data+QUEUE_HEAD_LEN);
      if ((call CONTROL_Send.send(TOS_BCAST_ADDR, pkt_size+QUEUE_HEAD_LEN, &OutMsg)) == SUCCESS)
	lof.numRouteWithdrawalToSend--;
#ifdef DEBUG_ONLY
      numRouteWithdrawalSent++;
#endif
    }
    else if (lof.numProbeReplyToSend > 0) { //probe reply only
      dbg(KEY_DEBUG,"send reply %d\n",lof.numProbeReplyToSend);
						    	
      pkt_size = compose_probe_packet(PROBE_REPLY, lof.numProbeReplyToSend,OutMsg.data+QUEUE_HEAD_LEN);
      if ((call CONTROL_Send.send(TOS_BCAST_ADDR, pkt_size+QUEUE_HEAD_LEN, &OutMsg)) == SUCCESS)
	lof.numProbeReplyToSend--;
      else {
	dbg(KEY_DEBUG,"send reply %d: failed => pkt_size=%, QUEUE_HEAD_LEN=%d\n",lof.numProbeReplyToSend, pkt_size, QUEUE_HEAD_LEN);
	;
      }
#ifdef DEBUG_ONLY
      numProbeReplySent++;
#endif
    }	    	  
    else if (lof.numProbeRequestToSend > 0) { //probe request only
      //dbg(KEY_DEBUG,"send request %d\n",lof.numProbeRequestToSend);

      pkt_size = compose_probe_packet(PROBE_REQUEST, lof.numProbeRequestToSend,OutMsg.data+QUEUE_HEAD_LEN);
      if ((call CONTROL_Send.send(TOS_BCAST_ADDR, pkt_size+QUEUE_HEAD_LEN, &OutMsg)) == SUCCESS)
	lof.numProbeRequestToSend--;
    }
  
  hb_done:
    return (call Random.rand())%MAX_PROBE_INTERVAL + 300; //milliseconds;	  
  } //end of hb_timer_task(...)


  /* Timer firing */
 event result_t Timer.fired() {

#ifdef SANITY_CHECK
   loc_t parent_loc, self_loc;
   uint8_t tp;

   //routing table
   ngbr_table_index_t * head = lof.ngbr_table_head;
   int numNgbrs = 0;
   while (head != NULL) {
     head = head->next;
     numNgbrs++;
   }
   if (numNgbrs != lof.ngbr_table_size)
     dbg(KEY_DEBUG, "inconsistent routing table\n");

   //loop freedom
   if (lof.parent_id != TOS_LOCAL_ADDRESS) {
     set_loc(lof.parent_id, &parent_loc);
     set_loc(TOS_LOCAL_ADDRESS, &self_loc);
     if (distanceCmp(&lof.base_loc, &parent_loc, &self_loc) <= 1) {
       dbg(KEY_DEBUG, "error(0): backward link is used: %d --->%d;   base<%d, %d>, me<%d, %d>, parent<%d, %d>\n", TOS_LOCAL_ADDRESS, lof.parent_id, lof.base_loc.x, lof.base_loc.y, my_loc.x, my_loc.y, parent_loc.x, parent_loc.y);
       tp++;
     }
     if (parent_loc.x >= self_loc.x && parent_loc.y >= self_loc.y) {
       dbg(KEY_DEBUG, "error(1): backward link is used: %d --->%d;   base<%d, %d>, me<%d, %d>, parent<%d, %d>\n", TOS_LOCAL_ADDRESS, lof.parent_id, lof.base_loc.x, lof.base_loc.y, my_loc.x, my_loc.y, parent_loc.x, parent_loc.y);
       tp++;
     }
   }
#endif //SANITY_CHECK

   /*control the virtual timer frequency */
   timeElapsedSinceLastTimerFiring += TIMER_INTERVAL;
   if (timeElapsedSinceLastTimerFiring < hb_timer_interval) {
#ifdef QUICK_BEACON
     if (sendQuickBeacon) {
       atomic sendQuickBeacon = FALSE;
       if (lof.ngbr_table_head != NULL)
	 SendBeacon();
     }
#endif

     goto done;
   }
   else 
     timeElapsedSinceLastTimerFiring = 0;

   /* Perform periodic task based on timer frequency */
#ifdef USE_DETECTOR
   //detectorSeqNo
   if (timeToNextSeqNo > hb_timer_interval)
     timeToNextSeqNo -= hb_timer_interval;
   else
     timeToNextSeqNo = 0;
   //prop. control
   if (propHoldTime > hb_timer_interval)
     propHoldTime -= hb_timer_interval;
   else
     propHoldTime = 0;
#endif
 
   hb_timer_interval = hb_timer_task(hb_timer_interval);

 done:
   return SUCCESS;
 } //end of Timer.fired()
  
  
  /* Send packets from upper layer */
 command result_t Routing.send(TOS_MsgPtr msg, uint8_t size)
 {
#ifdef SNOOPING_BASED_RCVER_EST
   uint8_t count;
#endif

   link_pkt_head* link_hd = (link_pkt_head*)msg->data;
   lof_pkt_head* lof_hd = (lof_pkt_head*)(msg->data+QUEUE_HEAD_LEN);

   /* return if parent is not available */  		  		
   if (lof.parent_id == TOS_LOCAL_ADDRESS) { 
     if (lof.numRouteWithdrawalToSend <= 0) { 
       lof.numRouteWithdrawalToSend = NUM_PROBES_EACH_ROUND; 
       lof.numProbeReplyToSend = lof.numProbeRequestToSend = 0; 
     }

#ifdef DEBUG_ONLY 
     numSendButNoParent++;
#endif

     msg->addr=0xff;
     //Hongwei: signal Routing.sendDone(msg, FAIL);
     return FAIL;
   }
   else { //has parent
     //if table head is "dead" (due to false positive in blacklisting)
     if (lof.ngbr_table_head == NULL || //for stabilization
           ngbrTable[lof.ngbr_table_head->index].liveness == IS_DEAD
         ) {
       if (lof.numRouteWithdrawalToSend <= 0) {
	 lof.numRouteWithdrawalToSend = NUM_PROBES_EACH_ROUND; 
	 lof.numProbeReplyToSend = lof.numProbeRequestToSend = 0;
       }

#ifdef DEBUG_ONLY 
       numSendButNoParent++;
#endif

#ifdef KEEP_PARENT_EVEN_IF_DEAD
       if (lof.ngbr_table_head == NULL) {
#endif
	 msg->addr=0xff;
	 //Hongwei: signal Routing.sendDone(msg, FAIL);
	 return FAIL;
#ifdef KEEP_PARENT_EVEN_IF_DEAD
       }
#endif
     } //table head is "dead"

     //send the packet
     atomic {
       lof.last_seqno++;

#ifdef SUPPRESS_DUPLICATE_PKT 
       ngbrTable[lof.ngbr_table_head->index].seqNo++;
#endif  

#ifdef USE_DETECTOR
       //for temporal feedback
       if (detectorState == STATE_SPECIAL_HANDLING) 
	 expectedNumPktsToSendAfterSH++;
       else if (expectedNumPktsToSendAfterSH > 0) {
	 expectedNumPktsToSendAfterSH--;
	 if (expectedNumPktsToSendAfterSH == 0) { //positive temporal feedback
	   if (holdWindow_positive <= 0)
	     holdWindow_positive = improvedNetConditionHoldWindow;
	   else if (!probIntoHoldWindow)
	     holdWindow_positive = holdWindow_positive - (holdWindow_positive>>compPastW) + (improvedNetConditionHoldWindow>>compPastW);

	   if (probIntoHoldWindow) { //improvedNetConditionHoldWindow (decrease)
	     atomic probIntoHoldWindow = FALSE;
	     improvedNetConditionHoldWindow = improvedNetConditionHoldWindow - (improvedNetConditionHoldWindow>>compPastW) + (holdWindow_positive>>compPastW);
	   }
	 }
       }
#endif
     } //atomic
   }

   atomic {
     link_hd->tx_count = 0;
     link_hd->delay = 0;
     link_hd->source = TOS_LOCAL_ADDRESS;
 
     lof_hd->src = TOS_LOCAL_ADDRESS;
#ifdef NON_GEO_DV
     lof_hd->generator = TOS_LOCAL_ADDRESS;
#endif
#ifdef SUPPRESS_DUPLICATE_PKT 
     lof_hd->seqNo = ngbrTable[lof.ngbr_table_head->index].seqNo;
#endif

#if defined(D_V_SNOOPING) || defined(SNOOPING_BASED_RCVER_EST)
     lof_hd->weight = ngbrTable[lof.ngbr_table_head->index].weight;
#ifdef MAXIMIZE_HOP_COUNT
     lof_hd->numHops = (uint8_t)(ngbrTable[lof.ngbr_table_head->index].numHops + 1);
#elif defined(MAXIMIZE_RELIABILITY)
     lof_hd->pathReliability = ngbrTable[lof.ngbr_table_head->index].pathReliability;
#endif
#endif //D_V_SNOOPING || SNOOPING_BASED_RCVER_EST

#ifdef SNOOPING_BASED_RCVER_EST
#ifndef PIGGYBACK_RCVER_EST_ONTO_DATA
     if (sendQuickBeacon != TRUE) {
       lof_hd->rcver_est_src = lof_hd->rcver_est_linkReliability = 0;
       goto no_piggyback_rcver_est;
     }
#endif
     count = 0;
     while (count < MAX_NUM_NODES_TO_SNOOP ) {//have not finished searching
       if (snoop_link_estimator[to_send_index].src == FLAG_NOT_INITIALIZED || //not containing info
             snoop_link_estimator[to_send_index].linkReliability <= 0 ||
             snoop_link_estimator[to_send_index].potentialChild != TRUE
           ) {
	 count++;
	 to_send_index = (to_send_index + 1)%MAX_NUM_NODES_TO_SNOOP;
       }
       else
	 break;
     }
     //
     if (count < MAX_NUM_NODES_TO_SNOOP) {
       lof_hd->rcver_est_src = (uint8_t)snoop_link_estimator[to_send_index].src;
       lof_hd->rcver_est_linkReliability = snoop_link_estimator[to_send_index].linkReliability;
     }
     else 
       lof_hd->rcver_est_src = lof_hd->rcver_est_linkReliability = 0;
#ifndef PIGGYBACK_RCVER_EST_ONTO_DATA
   no_piggyback_rcver_est:
     ;
#endif
#endif //SNOOPING_BASED_RCVER_EST

#if defined(QUICK_BEACON) || defined(SNOOPING_BASED_RCVER_EST)
     if (sendQuickBeacon == TRUE) 
       sendQuickBeacon = FALSE; //to avoid extra contention introduced to data transmissions by quick-beacons
#endif

   } //atomic

   if ((call DATA_Send.send(lof.parent_id, size, msg)) == SUCCESS) {
     //atomic ngbrTable[lof.ngbr_table_head->index].numSend++;

     dbg(FLOW_DEBUG, "Routing.send, node_id=%d, numsend=%d\n",ngbrTable[lof.ngbr_table_head->index].node_id,ngbrTable[lof.ngbr_table_head->index].numSend);

     return SUCCESS;
   } 
   else 
     return FAIL; 
 } //end of Routing.send()


 /*
  * Used in: CONTROL_Send.sendDone(), DATA_Send.sendDone()
  */
 result_t processSendDone(TOS_MsgPtr msg, bool success, bool isForData)
 {
   uint32_t delay;
   uint8_t status;

   link_pkt_head* link_hd = (link_pkt_head*)msg->data;

   lof_heartbeat_t* lof_hb_pkt = (lof_heartbeat_t*)(msg->data+QUEUE_HEAD_LEN);
    
#ifdef BEACON_BASED_ESTIMATION //i.e., not using data-driven estimation
   return success;
#endif

   //not for data or initial link estimation (i.e., is for other control packets such as heartbeat packet)
   if (!(isForData == TRUE) && !(lof_hb_pkt->type == PROBE_LINK_ESTIMATION))
     return success;

#ifdef USE_TX_COUNT
   //delay = link_hd->tx_count * NT_SCALE_FACTOR; /*scale*/
   delay = (link_hd->tx_count + (link_hd->tx_count << 1)) << 13;
#else
   delay = link_hd->delay;
#endif 

   if (delay <= 0) {//due to reasons such as change_destination() at QueuedSendM, 
                               //which is in turn due to some previous transmission failures 
     return success;     
     //delay = SCALE_THOUSAND;
   }

   status = 1;
   if (success != SUCCESS || msg->ack == 0)
     status = 0;

   if (isForData == TRUE)
     fb_receive(status, delay, link_hd->tx_count, msg->addr, FALSE);
   else
     fb_receive(status, delay, link_hd->tx_count, msg->addr, TRUE);
        
   return SUCCESS;
 } //end of processSendDone()


 event result_t CONTROL_Send.sendDone(TOS_MsgPtr msg, bool success) {
   processSendDone(msg, success, FALSE);
   return success;
 }


 event result_t DATA_Send.sendDone(TOS_MsgPtr msg, bool success) {
   processSendDone(msg, success, TRUE); 

   return signal Routing.sendDone(msg, success); 
 }


#ifdef D_V_LOF
 /* When receive a periodic beacon packet
  *
  * Used in: hb_receive() 
  */
  void hb_receive_beacon(uint8_t* data, uint8_t size)
  {
#ifndef NON_GEO_DV
    loc_t ngbr_loc;
#endif

    lof_heartbeat_t * lof_hb_pkt = (lof_heartbeat_t*)(data+QUEUE_HEAD_LEN);

    dbg(FLOW_DEBUG,"in hb_receive_beacon \n");

    /*
    //??????
    if (TOS_LOCAL_ADDRESS == BASE_STATION_ID)
      return;
    */

#ifdef SNOOPING_BASED_RCVER_EST
    if (lof_hb_pkt->src < MAX_NUM_NODES_TO_SNOOP && //sanity check
          lof_hb_pkt->parent_id != TOS_LOCAL_ADDRESS && //not my child
          (TOS_LOCAL_ADDRESS == BASE_STATION_ID || lof.parent_id != TOS_LOCAL_ADDRESS) && //self is connected
          snoop_link_estimator[lof_hb_pkt->src].src != FLAG_NOT_INITIALIZED && //rcv-est has been initialized
          snoop_link_estimator[lof_hb_pkt->src].linkReliability >= MIN_RELIABILITY //at least one round of rcv-est is done
        )
      amIBetter(TRUE, lof_hb_pkt->src, lof_hb_pkt->weight
#ifdef MAXIMIZE_HOP_COUNT
	        , lof_hb_pkt->numHops
#elif defined(MAXIMIZE_RELIABILITY)
	        , lof_hb_pkt->pathReliability
#endif
	     );
#endif //SNOOPING_BASED_RCVER_EST

    if (TOS_LOCAL_ADDRESS == BASE_STATION_ID)
      return;

#ifndef NON_GEO_DV	
    set_loc(lof_hb_pkt->src, &ngbr_loc);
    if (distanceCmp(&lof.base_loc, &ngbr_loc, &my_loc) <= 1)
      return;		//not closer than self
#endif

    refresh_ngbr_table(lof_hb_pkt->src, lof_hb_pkt->weight, 
#ifdef BEACON_BASED_ESTIMATION
		lof_hb_pkt->hb_seqNo, lof_hb_pkt->lrList, FALSE, 0, 0
#elif defined(SNOOPING_BASED_RCVER_EST)
 	                    0, lof_hb_pkt->lrList, FALSE, 0, 0
#else
	                    0, NULL, FALSE, 0, 0 //dummy
#endif
#ifdef MAXIMIZE_HOP_COUNT
		, lof_hb_pkt->numHops
#elif defined(MAXIMIZE_RELIABILITY)
		, lof_hb_pkt->pathReliability
#endif
		);
  
    dbg(FLOW_DEBUG,"leave hb_receive_beacon \n");
  }//end of hb_receive_beacon()
#endif


 /* process received heartbeat packets 
 *
 * Used in: processRcvdHbPkt(...) 
 */
  void hb_receive(uint8_t* data, uint8_t size) 
  {
#ifndef NON_GEO_DV
    loc_t ngbr_loc;
    uint8_t temp; 
#endif

    lof_heartbeat_t * lof_hb_pkt = (lof_heartbeat_t*)(data+QUEUE_HEAD_LEN);

    dbg(FLOW_DEBUG, "entering hb_receive(...)\n");

#ifndef NON_GEO_DV
    set_loc(lof_hb_pkt->src, &ngbr_loc);
#endif

#ifdef D_V_LOF
    if (lof_hb_pkt->type == PROBE_BEACON) { //distance-vector routing 
      hb_receive_beacon(data, size);

      return;
    }
#endif

    //check if the sender is withdrawing its route
    if (lof_hb_pkt->type == PROBE_WITHDRAWAL) {
      //dbg(KEY_DEBUG, "received a PROBE_WITHDRAWAL packet from %d\n", lof_hb_pkt->src);

      //clear the "already served its request", if any
      //      served_list[lof_heartbeat_pkt->node_id] = NOT_SERVED_FLAG;

      //sends probe-reply if self can serve as a potential parent 
#ifndef NON_GEO_DV
      temp = distanceCmp(&lof.base_loc, &ngbr_loc, &my_loc);
#endif
      if (TOS_LOCAL_ADDRESS == BASE_STATION_ID || 
           (lof.parent_id != TOS_LOCAL_ADDRESS 
             && ngbrTable[lof.ngbr_table_head->index].liveness == IS_ALIVE 
#ifdef NON_GEO_DV
             && lof.parent_id != lof_hb_pkt->src //to avoid short loops
#else
             && temp < 1
#endif
           )
         )
	lof.numProbeReplyToSend = NUM_PROBES_EACH_ROUND;
      else if (TOS_LOCAL_ADDRESS != BASE_STATION_ID 
#ifndef NON_GEO_DV
                    && temp >= 1
#endif
                  )
	blacklist_if_exists(lof_hb_pkt->src);

      return;
    } //PROBE_WITHDRAWAL 
    else if (lof_hb_pkt->type == PROBE_REQUEST) {//receives a probe-request 
      //dbg(KEY_DEBUG, "received a PROBE_REQUEST packet\n");

      if (TOS_LOCAL_ADDRESS == BASE_STATION_ID || 
            (lof.parent_id != TOS_LOCAL_ADDRESS 
              && ngbrTable[lof.ngbr_table_head->index].liveness == IS_ALIVE 
#ifdef NON_GEO_DV
              && lof.parent_id != lof_hb_pkt->src //to avoid short loops
#else
              && distanceCmp(&lof.base_loc, &ngbr_loc, &my_loc) < 1
#endif
            )
          ) {//reply only when self is connected and is closer to the base station	
	/*
	  if(served_list[lof_heartbeat_pkt->node_id] != NOT_SERVED_FLAG) {//check whether has served request for this node
	  if(lof_heartbeat_pkt->num_probe_to_send == NUM_PROBES_EACH_ROUND) //the requested node must have 
	  //been reset
	  lof.numProbeReplyToSend = NUM_PROBES_EACH_ROUND;
	  }
	  else {//not yet served
	  served_list[lof_heartbeat_pkt->node_id] = SERVED_FLAG;	      	
	  lof.numProbeReplyToSend = NUM_PROBES_EACH_ROUND;
	  }
	*/
	lof.numProbeReplyToSend = NUM_PROBES_EACH_ROUND;
      } //if
      return;
    }//PROBE_REQUEST
    else if (lof_hb_pkt->type == PROBE_REPLY) {//receives a probe-reply
      //dbg(KEY_DEBUG, "received a PROBE_REPLY packet from %d\n", lof_hb_pkt->src);

      //maintain neighbor table for neighbors that are closer to the parent than itself or incurs lower cost
      if (TOS_LOCAL_ADDRESS != BASE_STATION_ID 
#ifndef NON_GEO_DV
            && (distanceCmp(&lof.base_loc, &ngbr_loc, &my_loc) > 1)
#endif
          )
	refresh_ngbr_table(lof_hb_pkt->src, 
#ifdef D_V_LOF
		                 lof_hb_pkt->weight,
#else
		                0,  //as a dummy; not used
#endif
#ifdef BEACON_BASED_ESTIMATION
		                lof_hb_pkt->hb_seqNo, lof_hb_pkt->lrList, FALSE, 0, 0
#else
		               0, NULL, FALSE, 0, 0  //as a dummy; not used
#endif
#ifdef MAXIMIZE_HOP_COUNT
		               , lof_hb_pkt->numHops
#elif defined(MAXIMIZE_RELIABILITY)
		               , lof_hb_pkt->pathReliability
#endif
		               );  //update the neighbor list
    }
    
    //dbg(FLOW_DEBUG, "exiting hb_receive(...)\n");
  }//end of hb_receive()



#ifdef SUPPRESS_DUPLICATE_PKT
  /* To determine whether a received packet is a duplicate
   *
   * Used in: data_receive()
   */ 
  bool isDuplicatePkt(TOS_MsgPtr rcvdPtr)
  {
    uint8_t i;
    uint8_t tp0, tp1; 
    lof_pkt_head* lof_hd = (lof_pkt_head*)(rcvdPtr->data+QUEUE_HEAD_LEN);

    i = 0;
    while (i < MAX_NUM_CHILDREN && dup_detector[i].child != 0xff) {
      if (dup_detector[i].child == lof_hd->src) {//has received packet from this child before
	//advance this element one position so as to increase search efficiency
	if (i > 0) {
	  atomic {
	    tp0 = dup_detector[i-1].child;
	    tp1 = dup_detector[i-1].last_seqNo;
	    //switch position
	    dup_detector[i-1].child = dup_detector[i].child;
	    dup_detector[i-1].last_seqNo = dup_detector[i].last_seqNo;
	    dup_detector[i].child = tp0;
	    dup_detector[i].last_seqNo = tp1;
	    //
	    i--;
	  } //atomic 
	}
	//check whether it is a duplicate
	if (dup_detector[i].last_seqNo != lof_hd->seqNo) {
	  atomic dup_detector[i].last_seqNo = lof_hd->seqNo;
	  return FALSE;
	}
	else
	  return TRUE;
      }
      else //next item
	i++;
    }//while()

    //first time heard packet from this child recently 
    if (i >= MAX_NUM_CHILDREN) {//children-table has been used up
      dup_detector[MAX_NUM_CHILDREN-1].child = lof_hd->src;
      dup_detector[MAX_NUM_CHILDREN-1].last_seqNo = lof_hd->seqNo;
#ifdef DEBUG_ONLY
      numChildren = 0xff;
#endif
    }
    else {
      dup_detector[i].child = lof_hd->src;
      dup_detector[i].last_seqNo = lof_hd->seqNo;
#ifdef DEBUG_ONLY
      numChildren = i+1;
#endif
    }

    return FALSE;
  } //end of isDuplicatePkt()
#endif //SUPPRESS_DUPLICATE_PKT 


  /* Process received data packets
   *
   * Used in: DATA_Receive.receive()
   */
  void data_receive(TOS_MsgPtr RcvdPtr) 
  {
#ifdef SNOOPING_BASED_RCVER_EST
    uint8_t count;
#endif

    link_pkt_head* link_hd = (link_pkt_head*)RcvdPtr->data;
    lof_pkt_head* lof_hd = (lof_pkt_head*)(RcvdPtr->data+QUEUE_HEAD_LEN);

#if defined(QUICK_BEACON) || defined(SNOOPING_BASED_RCVER_EST)
    if (sendQuickBeacon == TRUE)
      sendQuickBeacon = FALSE; //to avoid extra contention introduced to data transmissions by quick-beacons
#endif

#ifdef SNOOPING_BASED_RCVER_EST
    receiver_side_link_estimator(link_hd->source, link_hd->tx_count, FALSE, lof_hd->weight
#ifdef MAXIMIZE_HOP_COUNT
			, lof_hd->numHops
#elif defined(MAXIMIZE_RELIABILITY)
			, lof_hd->pathReliability
#endif
                                                          );
#endif

    //If I am the base station then send the packet to the application
    if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
#if defined(SUPPRESS_DUPLICATE_PKT) && !defined(EXPT_ONLY) 
      if (!isDuplicatePkt(RcvdPtr))
#endif
	signal Routing.receive(RcvdPtr);
    } 
    else { //I am a non-base node 
#ifdef EXPT_ONLY 
      signal Routing.receive(RcvdPtr); //for debug and statistical analysis 
#endif
      //check self has a parent
      if (lof.parent_id != TOS_LOCAL_ADDRESS) { //has parent, thus forward the packet to my parent.
#ifdef NON_GEO_DV
	//detect potential loop
	if (lof_hd->generator == TOS_LOCAL_ADDRESS) {//there must exist a loop
	  dbg(KEY_DEBUG, "Warning: loop detected: %d -> %d", TOS_LOCAL_ADDRESS, lof.parent_id);
	  blacklist_if_exists(lof.parent_id); 
	  return;
	}
#endif

#ifdef SUPPRESS_DUPLICATE_PKT
	if (isDuplicatePkt(RcvdPtr))
	  return;
#endif

	atomic {
	  lof.last_seqno++; 

#ifdef SUPPRESS_DUPLICATE_PKT 
	  ngbrTable[lof.ngbr_table_head->index].seqNo++;
#endif

#ifdef USE_DETECTOR
	  //for temporal feedback
	  if (detectorState == STATE_SPECIAL_HANDLING)
	    expectedNumPktsToSendAfterSH++;
	  else if (expectedNumPktsToSendAfterSH > 0){
	    expectedNumPktsToSendAfterSH--;
	    if (expectedNumPktsToSendAfterSH == 0) { //positive temporal feedback
	      if (holdWindow_positive <= 0)
		holdWindow_positive = improvedNetConditionHoldWindow;
	      else if (!probIntoHoldWindow) 
		holdWindow_positive = holdWindow_positive - (holdWindow_positive>>compPastW) + (improvedNetConditionHoldWindow>>compPastW);

	      if (probIntoHoldWindow) { //improvedNetConditionHoldWindow (decrease)
		atomic probIntoHoldWindow = FALSE;
		improvedNetConditionHoldWindow = improvedNetConditionHoldWindow - (improvedNetConditionHoldWindow>>compPastW) + (holdWindow_positive>>compPastW);
	      }
	    }
	  }
#endif
	} //atomic

	  //compose header fields
	atomic {
	  link_hd->tx_count = 0;
	  link_hd->delay = 0;
	  link_hd->source = lof_hd->src; //used by QueuedSendM.nc to log packet dropping stats
 
	  lof_hd->src = TOS_LOCAL_ADDRESS;
#ifdef SUPPRESS_DUPLICATE_PKT 
	  lof_hd->seqNo = ngbrTable[lof.ngbr_table_head->index].seqNo;
#endif

#if defined(D_V_SNOOPING) || defined(SNOOPING_BASED_RCVER_EST)
	  lof_hd->weight = ngbrTable[lof.ngbr_table_head->index].weight; 
#ifdef MAXIMIZE_HOP_COUNT
	  lof_hd->numHops = (uint8_t)(ngbrTable[lof.ngbr_table_head->index].numHops + 1);
#elif defined(MAXIMIZE_RELIABILITY)
	  lof_hd->pathReliability = ngbrTable[lof.ngbr_table_head->index].pathReliability;
#endif
#endif //D_V_SNOOPING || SNOOPING_BASED_RCVER_EST

#ifdef SNOOPING_BASED_RCVER_EST
#ifndef PIGGYBACK_RCVER_EST_ONTO_DATA
	  if (sendQuickBeacon != TRUE) {
	    lof_hd->rcver_est_src = lof_hd->rcver_est_linkReliability = 0;
	    goto no_piggyback_rcver_est_2;
	  }
#endif
	  count = 0;
	  while (count < MAX_NUM_NODES_TO_SNOOP ) {//have not finished searching
	    if (snoop_link_estimator[to_send_index].src == FLAG_NOT_INITIALIZED || //not containing info
                              snoop_link_estimator[to_send_index].linkReliability <= 0 ||
                              snoop_link_estimator[to_send_index].potentialChild != TRUE
                            ) {
	      count++;
	      to_send_index = (to_send_index + 1)%MAX_NUM_NODES_TO_SNOOP;
	    }
	    else
	      break;
	  }
	  //
	  if (count < MAX_NUM_NODES_TO_SNOOP) {
	    lof_hd->rcver_est_src = (uint8_t)snoop_link_estimator[to_send_index].src;
	    lof_hd->rcver_est_linkReliability = snoop_link_estimator[to_send_index].linkReliability;
	  }
	  else 
	    lof_hd->rcver_est_src = lof_hd->rcver_est_linkReliability = 0;
#ifndef PIGGYBACK_RCVER_EST_ONTO_DATA
	no_piggyback_rcver_est_2:
	  ;
#endif
#endif //SNOOPING_BASED_RCVER_EST
	} //atomic

	dbg(DATA_DEBUG, "%d ---fwd---> %d,  pkt-src = %d :  %d-th packet sent/forwarded \n", 
                                                         TOS_LOCAL_ADDRESS, lof.parent_id, lof_hd->src, lof.last_seqno
	       );

	if ((call DATA_Send.send(lof.parent_id, RcvdPtr->length, RcvdPtr)) == SUCCESS)
	  ;//atomic ngbrTable[lof.ngbr_table_head->index].numSend++;
	  		
      }//end of "I am non-base node, but has a parent"
      else { //I am a non-base node, and does not have a parent; I should notify my chilren of this fact
	if (lof.numRouteWithdrawalToSend <= 0) {
	  atomic {
	    lof.numRouteWithdrawalToSend = NUM_PROBES_EACH_ROUND; //notify children who must have missed
	                                                                                                                                        //the withdrawals sent earlier
	    lof.numProbeReplyToSend = lof.numProbeRequestToSend = 0;
	  }
	}

#ifdef DEBUG_ONLY 
	numFwdButNoParent++;
#endif
      }
    } //end "I am a non-base node" 
  }//end of data_receive() 


 /* Process received control packets
  *
  * Used in: CONTROL_Receive.receive()
  */
 TOS_MsgPtr processRcvdHbPkt(TOS_MsgPtr rcvd_packet)
 {
   lof_heartbeat_t * lof_hb_pkt = (lof_heartbeat_t *)(rcvd_packet->data+QUEUE_HEAD_LEN);

   if (!toStartLOF)
     return rcvd_packet;
    
   //call Leds.greenToggle();

   //dbg(FLOW_DEBUG, "entering receive()\n");
        	    
   if (lof_hb_pkt->src <= MAX_NGBR_TO_CONSIDER &&
        (lof_hb_pkt->type == PROBE_BEACON ||
         lof_hb_pkt->type == PROBE_REQUEST || lof_hb_pkt->type == PROBE_REPLY ||
         lof_hb_pkt->type == PROBE_WITHDRAWAL
       )
     ) //regular control packets
     hb_receive(rcvd_packet->data, rcvd_packet->length);
   else if (lof_hb_pkt->type == PROBE_LINK_ESTIMATION) //initial-estimation packet
     ;
#ifdef USE_DETECTOR
   else if (lof_hb_pkt->type == PROBE_DETECTION) {//detection packet: propagation of detector's belief 
     lof_detector_t* detector_t = (lof_detector_t*)(rcvd_packet->data+LOF_HB_PKT_SIZE);

     if (detector_t->seq > beliefSeqNo && detector_t->hops < MAX_PROP_HOPS) {//received notification of a new detection event
       beliefSrc = lof_hb_pkt->src;
       beliefSeqNo = detector_t->seq;
       timeToNextSeqNo = DETECTOR_SEQ_NO_INC_INTERVAL;
       propagatedHops = detector_t->hops + 1;

       whetherToPropagate = MAY_PROPAGATE;
       propHoldTime = (call Random.rand())%PROP_HOLDING_TIME + 128;
       numPktsRcvdDuringHoldTime = 0;

       //#ifdef LOG_LOF_DETECTOR
       //       log_lof_detector(5);
       //#endif

       if (lof.ngbr_table_head != NULL && lof.ngbr_table_head->next != NULL &&
            detectorState == STATE_IN_DETECTION &&
            ngbrTable[lof.ngbr_table_head->index].last_level_of_entry > ngbrTable[lof.ngbr_table_head->index].detectorLQI &&
            ngbrTable[lof.ngbr_table_head->index].last_level_of_entry - ngbrTable[lof.ngbr_table_head->index].detectorLQI > MIN_THRESHOLD//detector_threshold
           )
	 detectorState = STATE_INFORMED_DETECTION;
     }
     else if (whetherToPropagate == MAY_PROPAGATE &&
                   detector_t->seq == beliefSeqNo &&
                   detector_t->hops >= propagatedHops
                ) {//propagation of the same event by siblings or descendants
       numPktsRcvdDuringHoldTime++;
       if (numPktsRcvdDuringHoldTime >= NUM_OTHER_PROGS_TO_COVER_SELF)
	 whetherToPropagate = NOT_TO_PROPAGATE;
     }
   } //PROBE_DETECTION
#endif //USE_DETECTOR

   return rcvd_packet;
 } //end of processRcvdHbPkt()


 event TOS_MsgPtr CONTROL_Receive.receive(TOS_MsgPtr rcvd_packet) {
   return processRcvdHbPkt(rcvd_packet);
 } 
  
  
 event TOS_MsgPtr DATA_Receive.receive(TOS_MsgPtr rcvd_packet) {
   lof_pkt_head * lof_hd = (lof_pkt_head *)(rcvd_packet->data+QUEUE_HEAD_LEN);

   if (lof_hd->src <= MAX_NGBR_TO_CONSIDER)
     data_receive(rcvd_packet);

   return rcvd_packet;   
 }

#if defined(D_V_SNOOPING) || defined(SNOOPING_BASED_RCVER_EST)
 /* process snooped packets: to fetch D-V routing information */
 event TOS_MsgPtr snoop.receive(TOS_MsgPtr packet)
{
#ifdef SNOOPING_BASED_RCVER_EST
  link_pkt_head* link_hd;
#endif
#ifndef NON_GEO_DV
  loc_t ngbr_loc;
#endif

  lof_pkt_head * lof_hd = (lof_pkt_head*)(packet->data+QUEUE_HEAD_LEN); //snooped data
  lof_heartbeat_t * lof_hb_pkt = (lof_heartbeat_t*)(packet->data+QUEUE_HEAD_LEN); //snooped initEst pkts

   //check whether this packet can be used for routing decision making
   if (packet->crc != 1 || //comment out this line so as not to check crcs 
        packet->group != TOS_AM_GROUP ||
        packet->addr == TOS_BCAST_ADDR || packet->addr == TOS_LOCAL_ADDRESS ||
       (packet->type != AM_APP_DATA_CHANNEL &&
         !(packet->type == AM_LOF_CONTROL && lof_hb_pkt->type == PROBE_LINK_ESTIMATION)
       )
      ) 
     return packet;

#ifdef SNOOPING_BASED_RCVER_EST
  link_hd = (link_pkt_head*)packet->data;

  if (packet->type == AM_LOF_CONTROL) {//overhear initEst packets
    receiver_side_link_estimator(link_hd->source, link_hd->tx_count, TRUE, 0 
#if defined(MAXIMIZE_HOP_COUNT) || defined(MAXIMIZE_RELIABILITY)
		                    , 0
#endif
		                  );
    return packet;
  }
  else //overhear data
    receiver_side_link_estimator(link_hd->source, link_hd->tx_count, TRUE, lof_hd->weight
#ifdef MAXIMIZE_HOP_COUNT
		                    , lof_hd->numHops
#elif defined(MAXIMIZE_RELIABILITY)
		                    , lof_hd->pathReliability
#endif
                                                          );
#endif //SNOOPING_BASED_RCVER_EST

#if defined(D_V_SNOOPING) || defined(SNOOPING_BASED_RCVER_EST)
   //Fetch routing information and process it accordingly
   //Very similar to hb_receive_beacon(...)
#ifndef NON_GEO_DV
   set_loc(lof_hd->src, &ngbr_loc);
   if (distanceCmp(&lof.base_loc, &ngbr_loc, &my_loc) <= 1)  //not closer to destination than self
     return packet;
   else
#endif
#if !defined(PIGGYBACK_RCVER_EST_ONTO_DATA) && !defined(D_V_SNOOPING)
     if (lof_hd->rcver_est_src == TOS_LOCAL_ADDRESS)
#endif
       refresh_ngbr_table(lof_hd->src, lof_hd->weight, 
#ifdef BEACON_BASED_ESTIMATION
	  	   lof_hd->hb_seqNo, 
		   NULL, TRUE, 0, 0 //dummy
#elif defined(SNOOPING_BASED_RCVER_EST)
		   0, NULL, //dummy
                                          TRUE, lof_hd->rcver_est_src, lof_hd->rcver_est_linkReliability
#else
		   0, NULL, TRUE, 0, 0 //dummy
#endif
#ifdef MAXIMIZE_HOP_COUNT
		   , lof_hd->numHops
#elif defined(MAXIMIZE_RELIABILITY)
		   , lof_hd->pathReliability
#endif
                                          );
#endif  //D_V_SNOOPING || PIGGYBACK_RCVER_EST_ONTO_DATA

   return packet;

} //end of snoop.receive()
#endif


 default event TOS_MsgPtr Routing.receive(TOS_MsgPtr msg) 
{
   return msg;
 }


 //--------------------- STATE REFLECTION ------------------------//
 command uint8_t Routing.getNumofTableRegeneration()
 {
   return num_tb_regeneration;
 }


 command uint8_t Routing.getCurrentParent()
 {
   return lof.parent_id;
 }
  

 command uint8_t Routing.getNumofDeadNeighbors()
 {
   uint8_t count=0;
   ngbr_table_index_t *tail = lof.ngbr_table_head; 
	
   while (tail != NULL) {
     if (tail->index < MAX_NGBR_TABLE_SIZE && //stabilization 
           ngbrTable[tail->index].liveness == IS_DEAD)
       count++;

     tail=tail->next;
   }
		
   return count;	
 }


 command uint8_t Routing.getCurrentBufferSize()
 {
   return call QueueControl.getOccupancy();
 }


 command uint8_t Routing.getCurrentTableSize()
 {
   return lof.ngbr_table_size;
 }
  

 command uint8_t Routing.getSwitchCount()
 {
   return ngbr_switch_count;
 }
  

 command uint8_t Routing.getPktSwitchBefore()
 {
#ifdef USE_NGBR_SWITCH
   return lof.pktsBeforeSwitching;
#else
   return 0;
#endif 
 } 

 default event result_t Routing.sendDone(TOS_MsgPtr msg, result_t success) {
   return SUCCESS; 
 }


#ifdef DEBUG_ONLY 
  command uint16_t Routing.getNumProbeReplySent()
  {
    return numProbeReplySent;
  }

  command uint16_t Routing.getNumRouteWithdrawalSent()
  {
    return numRouteWithdrawalSent;
  }

  command uint16_t Routing.getNumReplyBasedNgbrRevitalization()
  {
    return numReplyBasedNgbrRevitalization;
  }

  command uint16_t Routing.getNumSendButNoParent()
  {
    return numSendButNoParent;
  }

  command uint16_t Routing.getNumFwdButNoParent()
  {
    return numFwdButNoParent;
  }

  command uint8_t Routing.getNumChildren()
  {
    //return numChildren;
       return lof.ngbr_table_size; //to record the ngbr-table-size instead :-)
  }

  command uint8_t Routing.getNumSH()
  {
    return numSH;
  }

  command uint8_t Routing.getNumSH2()
  {
    return numSH2;
  }
#endif

}
