/***********************************************************************************
    Copyright (C) <2005>  <Hongwei Zhang>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    Author:
    =======
    Hongwei Zhang
    Department of Computer Science and Engineering
    The Ohio State University, USA

    E-mail: zhangho@cse.ohio-state.edu

***********************************************************************************/


 /*
 * linxy.h: type defintion for linxy.c
 *
 * Hongwei Zhang (zhangho@cse.ohio-state.edu)
 * Date: October 25, 2004 - 
 *
 */

#include "linxy.h"


/*
 * To check whether ARP cache has an entry for "destination", and is alive
 *
 * Return value: 1: yes              0: no          -1: error reading the ARP table
 *
 * used in:
 */
#define ARP_CACHE "/proc/net/arp"
int in_arp_cache_alive(if_id_t destination)
{
  char buf[256];
  char ip_addr[256];
  int hw_type, flag;
  //char * bp;

  /* open the file to read ARP cache data*/
  FILE * f = fopen(ARP_CACHE, "r");
  if (f==NULL)
    return -1;

  /* check whether destination is in arp cache alive */
  //skip the title line
  fgets(buf, 255, f);
  //read data lines of format: IP address   HW-type   flags   HW-address   mask   device
  while (fgets(buf,255,f)){
    //elog(LOG_NOTICE, "%s", buf);
    sscanf(buf, "%s %x %x", ip_addr, &hw_type, &flag); //parse out the ip-address and flag (up --> ==2;  or down)
    //elog(LOG_NOTICE, "ip_addr = %s        hw_type = %d    flag = %d", ip_addr, hw_type, flag);

    //check if this entry is for the "destination"
    if (destination == ntohl(inet_addr(ip_addr))) {
      fclose(f);
      if (flag == ATF_COM || flag == ATF_PERM || flag == ATF_PUBL) { //is up
	elog(LOG_NOTICE, "%s is UP", ip_addr);
	return 1;
      }
      else { //is down
	elog(LOG_NOTICE, "%s is DOWN", ip_addr);
	return 0;
      }
    }
  } //end of while(...)

  fclose(f);
  return 0;
} //end of in_arp_cache_alive(...)


/* 
 * To allocate a buffer for MAC feedback
 *
 * used in: lx_send(...), lx_send_buffered_data_pkts(...)
 */
 lx_buf_t * allocate_mac_fb_buf( lx_state_t * r, link_pkt_type_t pkt_type, if_id_t dest_id)
{
  elog(LOG_DEBUG(2), "entering allocate_mac_fb_buf()");

  /* for MAC feedback */
#ifdef SIMULATION_ONLY
  //compose the feedback packet to forward to the upper layer
  link_pkt_t link_hdr_fb ={
    type: PKT_TYPE_LX_FB,
  };
  buf_t * up_pkt = buf_new();
  //construct the link header
  bufcpy(up_pkt, &link_hdr_fb, sizeof(link_pkt_t));
  //creat and copy the feedback data
  linxy_fb_pkt_t linxy_fb;
  linxy_fb.client_pkt_type = pkt_type;
  linxy_fb.status = 1;  //1: success   -1: failure
  linxy_fb.fb_delay = 1000 + rand()%1000; //microseconds
  gettimeofday(&(linxy_fb.rcv_time), NULL);
  memcpy(linxy_fb.mac_addr, "00:00:00:00:00:00     ", MAC_ADDR_DISPLAY_LEN);
  bufcpy(up_pkt, &linxy_fb, sizeof(linxy_fb_pkt_t));
  //send the data to application
  lp_receive(r->lp, (link_pkt_t *)up_pkt->buf, up_pkt->len - sizeof(link_pkt_t));
  buf_free(up_pkt);

  elog(LOG_DEBUG(2), "exiting allocate_mac_fb_buf()");

  return NULL;
#else   //real demo 
  /* maintaining the packets whose feedback is yet to be generated */
  lx_buf_t * buf = NULL;
  //create the buffer
  if ((buf = (lx_buf_t *) malloc(sizeof(lx_buf_t))) < 0) {
    elog(LOG_CRIT, "cannot allocate memory to buffer the fb for  a packet yet to be acked");
    //lx_memory_shortage_fb(r);
    errno = EAGAIN;
    exit(1);
    //return -errno;
  }
  buf->pkt_type = pkt_type;
  buf->dest_id = dest_id;
  gettimeofday(&(buf->send_time), NULL);
  buf->up_pkt = NULL;
  buf->next = NULL;
  //attach to the list
  if (r->buf_head == NULL)
    r->buf_head = r->buf_tail = buf;
  else {
    r->buf_tail->next = buf;
    r->buf_tail = buf;
  }
  r->outstanding_fb_wait++;
  r->queue_len++;

  elog(LOG_DEBUG(2), "exiting allocate_mac_fb_buf()");

  return buf;
#endif //end of SIMULATION_ONLY
} //end of allocate_mac_fb_buf(...)


//#ifndef SIMULATION_ONLY
/*
 * generate a feeback to signify the the mac-fb has been lost (because destination is down)
 *
 * used in: lx_stabilization_timer_task(...) 
 */
void handle_fb_loss(lx_state_t * r)
{
  char buf[DEFAULT_LX_MTU+sizeof(linxy_iwtx_pkt_t)+sizeof(link_pkt_t)];

  elog(LOG_DEBUG(2), "entering handle_fb_loss()");

  linxy_iwtx_pkt_t * fb_pkt = (linxy_iwtx_pkt_t *) buf;
  if_id_t down_dest = r->buf_head->dest_id;
  fb_pkt->status = -2; 
  /*the following fields are meaningless
    fb_pkt->fb_delay = MAX_MAC_FB_LATENCY*1000;
    fb_pkt->bitrate = 1000000; //bit/sec
    gettimeofday(&(fb_pkt->rcv_time), NULL);
    //MAC-addr is meaningless in this case:  memcpy(linxy_fb.mac_addr, fb_pkt->mac_addr, MAC_ADDR_DISPLAY_LEN);
    */
  lx_process_ack(r, fb_pkt, r->buf_head);

  //in the same manner, process the other neighboring ack-buffers related to the same destination
  while (r->buf_head != NULL)
    if (r->buf_head->dest_id == down_dest)
      lx_process_ack(r, fb_pkt, r->buf_head);  //will update r->buf_head
    else 
      break;
  if (r->buf_head != NULL) { //deal with other lx_buf that are not contiguous
    lx_buf_t * ptr = r->buf_head->next;
    while (ptr != NULL) {
      if (ptr->dest_id == down_dest)
	lx_process_ack(r, fb_pkt, ptr);
      ptr = ptr->next;
    }
  }

  /* has been taken care of in lx_timer_task
  //check to see if there are additional fb-losses
  struct timeval current_time;
  gettimeofday(&current_time, NULL);
  while (r->buf_head != NULL)
    if ((misc_tv_usec_l(&current_time) - misc_tv_usec_l(&r->buf_head->send_time)) > MAX_MAC_FB_LATENCY * 1000) 
      lx_process_ack(r, fb_pkt);
    else 
      break;
  */

  elog(LOG_DEBUG(2), "exiting handle_fb_loss()");
} //end of handle_fb_loss(r)


/* Precess a MAC feedback: send it up, or enqueue/mark it
 *
 * Used in: lx_iwtx_feedback(...), handle_fb_loss(...), lx_memory_shortage_fb
 */
void lx_process_ack(lx_state_t * r, linxy_iwtx_pkt_t * fb_pkt,  lx_buf_t * fb_buf_ptr)
{
  lx_buf_t * ptr, * ptr_prev;

  elog(LOG_DEBUG(2), "entering lx_process_ack()");

  if (r->buf_head == NULL) { //extra feedback
    elog(LOG_ERR, "extra feedback");
    elog(LOG_DEBUG(2), "exiting lx_process_ack()");
    return;
  }

  if (fb_pkt->status != 1 && fb_pkt->status != -1) {//locally generated by linxy itself: -2, -3, or -4
    ptr = fb_buf_ptr;
    goto valid;
  }
  else 
    ptr = r->buf_head;

  //r->time_waited = 0;

  //feedback validity check: type
  while (ptr != NULL)
    if ((ptr->dest_id == LINK_BROADCAST && strstr(fb_pkt->mac_addr, INVALID_FB_MAC_ADDR_TRAIL) != NULL) ||
          (ptr->dest_id != LINK_BROADCAST && strstr(fb_pkt->mac_addr, INVALID_FB_MAC_ADDR_TRAIL) == NULL)
       ) //is the one
      break;
    else {
      ptr_prev = ptr;
      ptr = ptr->next;
    }

  if (ptr == NULL) { //incomplete feedback: see hostap_hw.c "driver feedback" ;
                                      //(since I always allow driver to send feedback, possibly with incomplete information except for 1 or -1 status)
    /*
    if (r->buf_head->dest_id == LINK_BROADCAST) //CAREFUL: may introduce error in logic, especially for unicast 
                                                                                                     //(thus we check whether head if unicast or broadcast)
      ptr = r->buf_head; 
    */
    elog(LOG_ERR, "invalid feedback: no matching item");
    elog(LOG_DEBUG(2), "exiting lx_process_ack()");
    return;
  }
  /* NOT necessarily, if the mac feedback is not FIFO
  else if (ptr != r->buf_head) {//feedbacks for the items between r->buf_head and ptr must have been lost 
    lx_buf_t * ptr_lost = r->buf_head; 
    while (ptr_lost != ptr && ptr_lost != NULL) { 
      ptr_lost = ptr_lost->next;  
      handle_fb_loss(r);  
    } 
  }  
  */
 
#ifdef LX_DEBUG
      elog(LOG_NOTICE, "VERIFIED:   num_sent = %f        num_fb = %f", (double)r->num_sent, (double)r->num_fb);
#endif

  //compose the feedback packet to forward to the upper layer
  link_pkt_t  link_hdr;
 valid:
  link_hdr.type =PKT_TYPE_LX_FB;
#ifdef READ_MAC_TX_TIME
  get_last_mac_tx_time(&(link_hdr.if_rcv_time.stamp));
#endif
  buf_t * up_pkt = buf_new();
  //construct the link header
  bufcpy(up_pkt, &link_hdr, sizeof(link_pkt_t));
  //creat and copy the feedback data
  linxy_fb_pkt_t linxy_fb;
  if (ptr == NULL)
    linxy_fb.client_pkt_type = PKT_TYPE_EXTRA_TX_FB;
  else 
    linxy_fb.client_pkt_type = ptr->pkt_type;
  if (fb_pkt->status == -1 && ptr != NULL && ptr->dest_id == LINK_BROADCAST) //deal with incomplete feedback
    linxy_fb.status = 1; //always 1 for broadcast
  else 
    linxy_fb.status = fb_pkt->status;
  linxy_fb.fb_delay = fb_pkt->fb_delay;
  linxy_fb.bitrate = fb_pkt->bitrate;
  linxy_fb.rcv_time = fb_pkt->rcv_time;
#ifdef LOG_LINK_STATS
  linxy_fb.tx_unicasts = r->tx_unicasts;
  linxy_fb.tx_multicasts = r->tx_multicasts;
  linxy_fb.rx_unicasts = r->rx_unicasts;
  linxy_fb.rx_multicasts = r->rx_multicasts;
#endif
  memcpy(linxy_fb.mac_addr, fb_pkt->mac_addr, MAC_ADDR_DISPLAY_LEN);
  bufcpy(up_pkt, &linxy_fb, sizeof(linxy_fb_pkt_t));
  if (ptr == NULL) { //send up immediately
    lp_receive(r->lp, (link_pkt_t *)up_pkt->buf, up_pkt->len - sizeof(link_pkt_t));
    buf_free(up_pkt);
  }
  else //buffer it to be pushed up later after checking
    ptr->up_pkt = up_pkt;

  /* sending up fb packets when ready */
  while (r->buf_head != NULL && r->buf_head->up_pkt != NULL) {
    up_pkt = r->buf_head->up_pkt;
    lp_receive(r->lp, (link_pkt_t *)up_pkt->buf, up_pkt->len - sizeof(link_pkt_t));
    buf_free(up_pkt);
    // update the fb-buffer list
    lx_buf_t * tp = r->buf_head;
    r->buf_head = r->buf_head->next;
    if (r->buf_head == NULL)
      r->buf_tail = NULL;
    //free buffer space
    free(tp);
    r->outstanding_fb_wait--;
    r->queue_len--;
  }

  if (r->data_buf_head == NULL && r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT)
    lp_unblock(r->lp, 0);

  elog(LOG_DEBUG(2), "exiting lx_process_ack()");
} //end of lx_process_ack(...)
//#endif //SIMULATION_ONLY

/* 
 * To send a feedback to upper layers, signifying udp-sender-error
 *             
 * used in lx_receive(...), receipt_timeout(...), lx_stabilization_timer_task(...) 
 */
void lx_send_error_fb(lx_state_t * r)
{
  elog(LOG_DEBUG(2), "entering lx_send_error_fb()");

  char buf[DEFAULT_LX_MTU+sizeof(linxy_iwtx_pkt_t)+sizeof(link_pkt_t)];
  linxy_iwtx_pkt_t * fb_pkt = (linxy_iwtx_pkt_t *) buf;

  if_id_t down_dest = r->buf_head->dest_id; //after avon park

  fb_pkt->status = -3; //UDP send error
  lx_process_ack(r, fb_pkt, r->buf_head);

  //After avon park: in the same manner, process the other neighboring ack-buffers related to the same destination
  while (r->buf_head != NULL)
    if (r->buf_head->dest_id == down_dest)
      lx_process_ack(r, fb_pkt, r->buf_head);  //will update r->buf_head
    else
      break;
  if (r->buf_head != NULL) { //deal with other lx_buf that are not contiguous
    lx_buf_t * ptr = r->buf_head->next;
    while (ptr != NULL) {
      if (ptr->dest_id == down_dest)
	lx_process_ack(r, fb_pkt, ptr);
      ptr = ptr->next;
    }
  }

  elog(LOG_DEBUG(2), "exiting lx_send_error_fb()");
} //end of lx_send_error_fb(r)



/* 
 * To send a feedback to upper layers, signifying no-mac-feedback because of 
 *  not sending the packet which is in turn due to memory shortage
 *
 * used in lx_send(...); not used now because it violate FIFO style feedback
 */
void lx_memory_shortage_fb(lx_state_t * r, lx_buf_t * fb_buf_ptr) 
{
  elog(LOG_DEBUG(2), "entering lx_memory_shortage_fb()");

  char buf[DEFAULT_LX_MTU+sizeof(linxy_iwtx_pkt_t)+sizeof(link_pkt_t)];

  linxy_iwtx_pkt_t * fb_pkt = (linxy_iwtx_pkt_t *) buf;
  fb_pkt->status = -4; //memory shortage

  lx_process_ack(r, fb_pkt, fb_buf_ptr);

  elog(LOG_DEBUG(2), "exiting lx_memory_shortage_fb()");
} //end of lx_memory_shortage_fb(r)


/*Handle the timer for receipt_control*/
int receipt_timeout(void *data, int interval, g_event_t *ev)
{

  elog(LOG_DEBUG(2), "entering receipt_timeout()");

  lx_state_t * r = (lx_state_t *) data;

  //error in UDP send operation; to generate a feedback and let higher-layer deal with it
  lx_send_error_fb(r);
  //receipt control
  r->receipt_waiting = 0;
  //send buffer packet, if any
  if (r->data_buf_head != NULL &&
        r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT &&
        r->writable == 1
      ) {
    elog(LOG_DEBUG(2), "to send BUFFERED packet");
    lx_send_buffered_data_pkts(r);
  }

  elog(LOG_DEBUG(2), "exiting receipt_timeout()");
  return TIMER_DONE;
} //end of receipt_timeout(...)


/*
 * To send a packet in the data buffer queue
 *
 * called in: receipt_timeout(...), lx_receive(...), lx_stabilization_timer_task(...), handle_writable(...) 
 */
int lx_send_buffered_data_pkts(lx_state_t * r)
{
  lx_data_buf_t * head;
  buf_t * pkt;
  int retval;

  elog(LOG_DEBUG(2), "entering lx_send_buffered_data_pkts(...)");

  //try sending packets
  head = r->data_buf_head;
  pkt = head->packet;
  //request a receipt
  link_receipt_request((link_pkt_t *)pkt->buf, NULL);
  //send the packet
  retval = lu_send(r->link, (link_pkt_t *)pkt->buf, pkt->len - sizeof(link_pkt_t));
  //check sending status
  if (retval >= 0) {//success
    //receipt control
    r->receipt_waiting = 1;
    g_timer_add(RECEIPT_TIMEOUT, receipt_timeout, r, NULL, &(r->receipt_timer));
    //prepare mac-fb buffer
    allocate_mac_fb_buf(r, head->pkt_type, ((link_pkt_t *)pkt->buf)->dst.id);
    gettimeofday(&(r->last_send), NULL);
    //move to next
    r->data_buf_head = head->next;
    if (r->data_buf_head == NULL)
      r->data_buf_tail = NULL;
    //free memory
    buf_free(head->packet);
    free(head);
    //log
    r->data_queue_len--;
    elog(LOG_NOTICE, "node %d: data_queue_len = %d", my_node_id, r->data_queue_len);
#ifdef LX_DEBUG
    r->num_sent++;
#endif
    //break;//debug only
  }
  else if (errno == EAGAIN) {//buffer full
    elog(LOG_WARNING, "linxy failed to send a packet: %m");
    lu_writable_cb_set_enable(r->link, 1); //enable it
    r->writable = 0;
    //break;
  }
  else {
    elog(LOG_CRIT, "got fatal error when sending a packet: %m");
    exit(1);
  }

  if (r->data_buf_head == NULL && r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT)
    lp_unblock(r->lp, 0);

  elog(LOG_DEBUG(2), "exiting lx_send_buffered_data_pkts(...)");

  return EVENT_RENEW;
} //end of lx_send_buffered_data_pkts(...)


/*
 * Callback function for writable; 
 * to send a packet in the data buffer queue, if any
 *
 */
int lx_handle_writable(lu_context_t * link)
{
  lx_state_t * r = (lx_state_t *) lu_data(link);

  //first, disable writable callback
  r->writable = 1;
  lu_writable_cb_set_enable(r->link, 0);

  //try sending packets
  if (r->data_buf_head != NULL &&
       r->receipt_waiting == 0 &&
       r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT
      ) {
    elog(LOG_DEBUG(2), "to send  BUFFERED packet");
    lx_send_buffered_data_pkts(r);
  }

  return EVENT_RENEW;
} //end of handle_writable(...)
