/***********************************************************************************
    Copyright (C) <2005>  <Hongwei Zhang>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    Author:
    =======
    Hongwei Zhang
    Department of Computer Science and Engineering
    The Ohio State University, USA

    E-mail: zhangho@cse.ohio-state.edu

***********************************************************************************/


/*
 * linxy.c: as a proxy to link layer, providing feedback on send success or failure 
 *
 * Hongwei Zhang (zhangho@cis.ohio-state.edu)
 * Date: July 25, 2004 - 
 *
 */

#include "linxy.h"

#ifndef SIMULATION_ONLY
/* Callback when a packet comes up via the (UDP) network from iwtxstatus */
static gint lx_iwtx_feedback(gpointer data, gint fd, int fusd_condition, g_event_t *ev)
{

  elog(LOG_DEBUG(2), "entering lx_iwtx_feedback()");

  lx_state_t * r = (lx_state_t *) data;
  char buf[DEFAULT_LX_MTU+sizeof(linxy_iwtx_pkt_t)+sizeof(link_pkt_t)];

  int offset = sizeof(link_pkt_t)-sizeof(linxy_iwtx_pkt_t);
  linxy_iwtx_pkt_t * fb_pkt = (linxy_iwtx_pkt_t *)(buf+offset);
  int data_len = sizeof(buf) - offset;
  
  struct sockaddr_in addr;
  struct timeval rcv_time;
  uint16_t this_fb_delay; //milliseconds

  /* read the incoming UDP datagram and its timestamp; read all the packets if there are many */
  while (udp_read_with_timestamp(fd, (char *)fb_pkt, &data_len, &addr, &rcv_time) >= 0) {
    //make sure it has a sane address
    if (addr.sin_family != AF_INET) {
      elog(LOG_WARNING, "got packet with a non-Internet return address - confusing!");
      continue;
    }
 
    //make sure it's big enough! 
    if (data_len < sizeof(linxy_iwtx_pkt_t)) {
      elog(LOG_WARNING, "got short packet (%d bytes) from iwtxstatus", data_len);
      continue;
    }
 
    /* notify the upper layer by sending it up via the pd (packetdev) device */
    /* for debug only 
    //print out the timestamp of the received tx event, as well as the mac address
    int s = (fb_pkt->rcv_time.tv_sec) % 86400;
    elog(LOG_NOTICE, "%02d:%02d:%02d.%06u   %d   %s", s / 3600, (s % 3600) / 60, s % 60, (u_int32_t) fb_pkt->rcv_time.tv_usec, fb_pkt->status, fb_pkt->mac_addr);
    */

#ifdef LX_DEBUG
    r->num_fb++;
    elog(LOG_NOTICE, "unverified:   num_sent = %f        num_fb = %f", (double)r->num_sent, (double)r->num_fb);
#endif

    //adjust max feedback delay
    if (r->buf_head != NULL) {
      this_fb_delay = (misc_tv_usec_l(&rcv_time)  - misc_tv_usec_l(&(r->last_send))) / 1000; //milliseconds
      elog(LOG_NOTICE, "this_fb_delay = %d    avg_fb_delay = %d milliseconds", this_fb_delay, r->avg_fb_delay);

      if (this_fb_delay > MAX_MAC_FB_DELAY)
	this_fb_delay = MAX_MAC_FB_DELAY;
      r->avg_fb_delay = (this_fb_delay + r->avg_fb_delay) >> 1;
    }

    //process the feedback 
    lx_process_ack(r, fb_pkt, r->buf_head);
  } //while(...)

  elog(LOG_DEBUG(2), "exiting lx_iwtx_feedback()");

  return EVENT_RENEW;
} //end of lx_iwtx_feedback(...) 


/*Handle the timer for stabilization and data buffer management*/
int lx_stabilization_timer_task(void *data, int interval, g_event_t *ev)
{

  elog(LOG_DEBUG(2), "entering lx_stabilization_timer_task()");

  lx_state_t * r = (lx_state_t *) data;

  //stabilization for MAC feedback
  if (r->buf_head != NULL) {
    struct timeval current_time;
    gettimeofday(&current_time, NULL);
    long long time_waited = (misc_tv_usec_l(&current_time) - misc_tv_usec_l(&(r->last_send))) / 1000; //milliseconds
    if (time_waited > MAX_ARP_RESOLUTION_DELAY + MAX_MAC_FB_LATENCY_RATIO * r->avg_fb_delay) {
      elog(LOG_CRIT, "warning: no MAC feedback; please check configuration.  queue_len = %d     data_queue_len = %d. ", 
                                        r->queue_len, r->data_queue_len);
      int dest_state = in_arp_cache_alive(r->buf_head->dest_id);
      if (dest_state == 0 || r->buf_head->dest_id == LINK_BROADCAST) //destination is "down"
	handle_fb_loss(r);
      else if (dest_state == 1 && time_waited > INTRA_NODE_ERROR_THRESHOLD_WAIT) //destination is up 
	                                                                                                                                                              //but no feedback; intra-node issue
	lx_send_error_fb(r);
    }
  }

  //send data if necessary
  if (r->data_buf_head != NULL &&
       r->receipt_waiting == 0 &&
       r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT &&
       r->writable == 1 //lower-layer is ready
      ) {
    elog(LOG_DEBUG(2), "to send BUFFERED packet");
    lx_send_buffered_data_pkts(r);
  }

  //stabilization for receipt control
  if (r->data_buf_head == NULL && r->receipt_waiting == 1 && !g_timer_isset(r->receipt_timer))
    r->receipt_waiting = 0;

  //stabilization for fb-control
  if (r->buf_head == NULL && r->outstanding_fb_wait != 0)
    r->outstanding_fb_wait = 0;

  //stabilization for data buffer management
  if (//r->data_buf_head != NULL && 
        //r->receipt_waiting == 0 &&
        //r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT &&
        r->writable == 0 &&
        lu_writable_cb_get_enable(r->link) == 0) {
    lu_writable_cb_set_enable(r->link, 1); //enable it
    //lx_send_buffered_data_pkts(r); //debug only
  }

  elog(LOG_DEBUG(2), "exiting lx_stabilization_timer_task()");

  return TIMER_RENEW;
} //end of lx_stabilization_timer_task(...)
#endif //SIMULATION_ONLY


/* 
 * Called when application sends a packet.
 */
static int lx_enqueue(lp_context_t *lp, link_pkt_t *packet, int data_len)
{
  //Can put code to check whether packet is valid

  return 0; //okay to receive the packet
}


/*
 * Called after packets are enqueued. Constructs the link and lx header. Sends it to the link.
 */
static int lx_send(lp_context_t * lp, link_pkt_t * link_pkt_orig, int data_len, int loop_needed)
{

  elog(LOG_DEBUG(2), "entering lx_send()");

  lx_state_t * r = (lx_state_t *) lp_data(lp);

  //elog(LOG_NOTICE, "entering lx_send(...)");

  //check payload length
  if (data_len > r->status.MTU) {
    elog(LOG_WARNING, "payload exceeds MTU");
    errno = EMSGSIZE;
    elog(LOG_DEBUG(2), "exiting lx_send(...)");
    return -errno;
  }

  //declaration for outgoing packet
  buf_t * pkt_out = buf_new();
  // create outgoing link header
  link_pkt_t link_hdr = {
    dst: {
      id: link_pkt_orig->dst.id,
    },
    type: PKT_TYPE_LX,
  };
  // create outgoing lx header
  lx_pkt_t lx_hdr = {
    client_pkt_type: link_pkt_orig->type, 
  };
  //construct the outgoing packet
  bufcpy(pkt_out, &link_hdr, sizeof(link_hdr));
  bufcpy(pkt_out, &lx_hdr, sizeof(lx_hdr));
  bufcpy(pkt_out, link_pkt_orig->data, data_len);

  //firt, check to see whether need to send the receveid packet from upper layer immediately
  if (r->data_buf_head == NULL && 
       r->receipt_waiting == 0 &&
       r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT &&
       r->writable == 1
      ) {//okay to send
    int thisRetVal;
    //request a receipt
    link_receipt_request((link_pkt_t *)pkt_out->buf, NULL);
    //send
    thisRetVal = lu_send(r->link, (link_pkt_t *)pkt_out->buf, pkt_out->len - sizeof(link_pkt_t));
    if (thisRetVal >= 0) { //success
      buf_free(pkt_out);
      allocate_mac_fb_buf(r, link_pkt_orig->type, link_pkt_orig->dst.id);
      gettimeofday(&(r->last_send), NULL);
      //receipt control
      r->receipt_waiting = 1;
      g_timer_add(RECEIPT_TIMEOUT, receipt_timeout, r, NULL, &(r->receipt_timer));
#ifdef LX_DEBUG
      r->num_sent++;
#endif
      goto done;
    }
    else if (errno == EAGAIN) {//buffer full
      elog(LOG_WARNING, "linxy failed to send a packet: %m");
      //enable writable callback
      r->writable = 0;
      lu_writable_cb_set_enable(r->link, 1);
    }
    else {
      elog(LOG_CRIT, "got fatal error when sending a packet: %m");
      exit(1);
    }
  }
  //put this packet in queue, when not able to send
  lx_data_buf_t * data_buf = (lx_data_buf_t *)malloc(sizeof(lx_data_buf_t));
  if (data_buf == NULL) {
    elog(LOG_CRIT, "cannot allocate memory to store the packet to be sent");
    //compose the mac feedback to show memory shortage
    lx_buf_t * fb_buf_ptr = allocate_mac_fb_buf(r, link_pkt_orig->type, link_pkt_orig->dst.id);
    lx_memory_shortage_fb(r, fb_buf_ptr);
    errno = EAGAIN;
    //free memory
    buf_free(pkt_out);
    exit(1);
    //return -errno;
  }
  else {//put in the data buffer queue
    data_buf->pkt_type = link_pkt_orig->type;
    data_buf->packet = pkt_out;
    data_buf->next = NULL;
    //add to the data buffer queue
    if (r->data_buf_head == NULL)
      r->data_buf_head = r->data_buf_tail = data_buf;
    else {
      r->data_buf_tail->next = data_buf;
      r->data_buf_tail = data_buf;
    }
    //log
    r->data_queue_len++;
    elog(LOG_NOTICE, "node %d: data_queue_len = %d", my_node_id, r->data_queue_len);
  }

 done:
#ifdef LOG_LINK_STATS
  if (link_pkt_orig->dst.id == LINK_BROADCAST)
    r->tx_multicasts++;
  else 
    r->tx_unicasts++;
#endif
  //elog(LOG_NOTICE, "exiting lx_send(...)");
  if (r->data_buf_head == NULL && r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT) {
    elog(LOG_DEBUG(2), "exiting lx_send()");
    return 0; //remains unblocked
  }
  else {
    elog(LOG_DEBUG(2), "exiting lx_send()");
    return LP_BLOCKED; //==1
  }
} //end of lx_send(...)


/* The callback is called when a data packet arrives on the link previously opened. */
int lx_receive(lu_context_t * link, link_pkt_t * link_pkt, ssize_t data_len)
{
  elog(LOG_DEBUG(2), "entering lx_receive()");

  lx_pkt_t * lx_pkt = (lx_pkt_t *) link_pkt->data;

  // Get out state pointer
  lx_state_t * r = (lx_state_t *) lu_data(link);

  //check for receipt, and make sure the packet is for linxy
  if(link_pkt->type != PKT_TYPE_LX) {
    if (link_pkt->type == PKT_TYPE_MAC_CTRL &&
          link_pkt->ext_type == MAC_CTRL_RECEIPT &&
          link_receipt_for_us(link_pkt)) {//ok, we get a receipt
      //handle receipt
      if (link_pkt->retval != 0) //error in UDP send operation; to generate a feedback and let higher-layer deal with it
	lx_send_error_fb(r);
      //receipt control
      r->receipt_waiting = 0;
      if (g_timer_isset(r->receipt_timer)) 
	g_event_destroy(r->receipt_timer);
      //send buffer packet, if any
      if (r->data_buf_head != NULL &&
            r->outstanding_fb_wait <= MAX_OUTSTANDING_FB_WAIT &&
            r->writable == 1
          ) {
	elog(LOG_DEBUG(2), "to send BUFFERED packet");
	lx_send_buffered_data_pkts(r);
      }
    }
    else 
      elog(LOG_NOTICE, "got a packet not meant for linxy - fliter failed!");

    goto done;
  }

  //compose the packet to forward to the upper layer
  link_pkt_t link_hdr ={
    dst: {id: link_pkt->dst.id},
    src: {id: link_pkt->src.id},
    type: lx_pkt->client_pkt_type,
    ext_type: link_pkt->ext_type,
    flags: link_pkt->flags,
    seqno: link_pkt->seqno,
    rcv_time: link_pkt->rcv_time,
#ifdef SEND_RCV_TIMESTAMP_PROC_FILE
    if_rcv_time: link_pkt->if_rcv_time,
#endif
  };
 
  buf_t * up_pkt = buf_new();

  //construct the link header
  bufcpy(up_pkt, &link_hdr, sizeof(link_pkt_t));
  //copy the client-layer data
  bufcpy(up_pkt, lx_pkt->data, data_len - sizeof(lx_pkt_t));
  //send the data to application
  lp_receive(r->lp, (link_pkt_t *)up_pkt->buf, up_pkt->len - sizeof(link_pkt_t));

  buf_free(up_pkt);

 done: 
#ifdef LOG_LINK_STATS
  if (link_pkt->dst.id == LINK_BROADCAST)
    r->rx_multicasts++;
  else 
    r->rx_unicasts++;
#endif
  free(link_pkt);

  elog(LOG_DEBUG(2), "exiting lx_receive()");

  return EVENT_RENEW;
} //end of lx_receive(...)


/*
 * Processes the commands for the link.
 */
static int lx_ioctl(lp_context_t *lp, int cmd, void *arg)
{
  lx_state_t *r = (lx_state_t *)lp_data(lp);

  elog(LOG_NOTICE,"Link-proxy Ioctl.");
  switch(cmd) {
  case LINK_GET_STATUS:
    memmove((link_status_t *)arg, &(r->status), sizeof(link_status_t));
    return 0;
  case LINK_RESET:
    elog(LOG_DEBUG(1),"'resetting link test0'");
    return 0;
  case LINK_SET_ACTIVE:
    r->status.active = *((int *) arg);
    elog(LOG_DEBUG(1), "app set link test0 to be %s", r->status.active ? "active" : "inactive");
    lp_push_status(r->lp, &(r->status));
    return 0;
  }
  return 0;
} //end of lx_ioctl(...)


/*
 * Maintains the status of the link.
 */
int lx_status_request(lp_context_t *lp)
{
  lx_state_t *r = (lx_state_t *) lp_data(lp);
	
  lp_push_status(lp, &(r->status));

  return EVENT_RENEW;
}


/* Callback activated when we are asked to shut down by erun. */
void lx_shutdown (void * data)
{
  lx_state_t * r = (lx_state_t *)data;
  //kill the child process
  if (kill(r->child_process_id, SIGINT) >= 0)
    elog(LOG_NOTICE, "killed the iwtxstatus process");
else
    elog(LOG_CRIT, "Failed to kill the iwtxstatus process; please kill it manually");

  elog(LOG_NOTICE,"shutting down linxy");
  exit(0);
}


void usage(char *name)
{
  misc_print_usage
    (name, " -U <device> -P <device>",
     "  --uses <device>: Specify device to use\n"
     "  --provides <device>: Specify device to provide\n"
     );
  exit(1);
}

int main(int argc, char *argv[])
{
  lx_state_t lx_state;
  char *uses = NULL;
  char *provides = NULL;

  misc_init(&argc, argv , CVSTAG);

  emrun_opts_t emrun_opts = {
    shutdown: lx_shutdown,
    data: &lx_state
  };

  /* Parse command-line arguments */
  uses = link_parse_uses(&argc, argv , NULL);
  if(uses == NULL) {
    elog(LOG_CRIT, "Please specify a link to use!");
    usage(argv[0]);
  }
	
  provides = link_parse_provides(&argc, argv, NULL);
  if(provides == NULL) {
    elog(LOG_CRIT, "Please specify a link to provide!");
    usage(argv[0]);
  }

  if(misc_args_remain(&argc, argv)) {
    elog(LOG_CRIT, "Additional unparsed arguments!");
    usage(argv[0]);
  }

  /* Initializes the state. */
  memset(&lx_state, 0 , sizeof(lx_state));
  lx_state.avg_fb_delay = DEFAULT_MAC_FB_DELAY;
  //receipt
  lx_state.receipt_timer = NULL;
  lx_state.receipt_waiting = 0;
  lx_state.dev_name = provides;
  //mac feedback
  lx_state.buf_head = lx_state.buf_tail = NULL;
  lx_state.outstanding_fb_wait = 0;
  lx_state.queue_len = 0;
  //lx_state.is_waiting_ack = -1;
  //lx_state.time_waited = 0;
  //data buffer
  lx_state.data_buf_head = lx_state.data_buf_tail = NULL; 
  lx_state.data_queue_len = 0;
  lx_state.writable = 1;
#ifdef LX_DEBUG
  lx_state.num_sent = lx_state.num_fb = 0;
#endif
#ifdef LOG_LINK_STATS
  lx_state.tx_unicasts = lx_state.tx_multicasts = lx_state.rx_unicasts = lx_state.rx_multicasts = 0;
#endif

  /* open the link to proxy-send packet */
  lu_opts_t lu_opts = {
    opts:	{
      name: uses,
      data: &lx_state,
      //also for receipt packet: pkt_type: PKT_TYPE_LX,
      q_opts: {
	inq_len: 400,
	outq_len: 80,
      }
    },
    receive: lx_receive,
    writable: lx_handle_writable,
  };
 
  if (lu_open(&lu_opts, &(lx_state.link)) < 0) {
    elog(LOG_CRIT,"can't open %s: %m", link_name(&lu_opts.opts, NULL));
    exit(-1);
  }

  //-----------------------------------------------------------//
  lu_get_if_id(lx_state.link, &(lx_state.status.if_id));
  lx_state.status.active = 1;
  lx_state.status.POT = 0;
  lu_get_mtu(lx_state.link, &(lx_state.status.MTU));
  /* Create the link for the higher layer */
  lp_opts_t opts = {
    description: "Interface to the higher layer",
    send: lx_send,
    enqueue: lx_enqueue,
    status_request: lx_status_request,
    ioctl: lx_ioctl,
    opts: {
      name: lx_state.dev_name,
      data: &lx_state
    },
    we_are_root: 1
  };
 
  lx_state.status.active = 1;
  uint16_t mtu;
  if (lu_get_mtu(lx_state.link, &mtu) >= 0)
    lx_state.status.MTU = mtu - sizeof(lx_pkt_t);
  else 
    lx_state.status.MTU = DEFAULT_LX_MTU - sizeof(lx_pkt_t);

  //Register the link
  if(lp_register(&opts, &(lx_state.lp)) < 0) {
    elog(LOG_CRIT,"can't create link dev %s: %m", lx_state.dev_name);
    exit(-1);
  }
		
  lx_status_request(lx_state.lp);

#ifndef SIMULATION_ONLY
  /* Set up a local udp socket to hear event reports from iwtxstatus */
  struct in_addr linxy_iwtx_addr = {
    s_addr: INADDR_ANY
  };
  //open the socket
  while ((lx_state.sock_fd = udp_listen_if(MAC_FEEDBACK_UDP_PORT, &linxy_iwtx_addr)) < 0) {
    elog_g(LOG_CRIT, "can't listen to UDP port %d at the interface %X", MAC_FEEDBACK_UDP_PORT, INADDR_ANY);
    //exit(-1);
    sleep(10);
  }
  elog_g(LOG_CRIT, "is listening for MAC feedback at UDP port %d at the interface %X", MAC_FEEDBACK_UDP_PORT, INADDR_ANY);
  //set the udp receive buffer size
  long rcv_buf_size = UDP_RCV_BUF_SIZE; 
  if (setsockopt(lx_state.sock_fd, SOL_SOCKET, SO_RCVBUF, &rcv_buf_size, sizeof(rcv_buf_size)) < 0) 
    fprintf(stderr, "cannot set the UPD receive buffer size to %f: %m\n", (double)rcv_buf_size);
  //set nonblocking
  set_nonblock(lx_state.sock_fd, 1);
  //create an event that is activated when a UDP packet arrives
  if (g_event_add(lx_state.sock_fd, FUSD_NOTIFY_INPUT, lx_iwtx_feedback, &lx_state, NULL, NULL) < 0) {
    elog_g(LOG_CRIT, "couldn't create UDP reading event: %m");
    exit(-1);
  }
  /*Fork a new process, and then start iwtxstatus */
  pid_t pid; //process id of the child process
  pid = fork();
  if (pid == -1) {
    elog(LOG_CRIT, "Failed to fork a process to run iwtxstatus");
    exit(-2);
  }
  else if (pid == 0) { //I am the child process 
    char * args[2];
    args[0] = "iwtxstatus";
    args[1] = NULL;//0 argument

    elog(LOG_CRIT, "Begin to run iwtxstatus");
    execvp(args[0], args); 
    //if execvp returns, it must have failed
    elog(LOG_CRIT, "Failed to execute iwtxstatus");
    exit(-3);
  }
  else { //I am the parent process
    lx_state.child_process_id = pid; 
    //set up the timer for stabilization and data buffer management
    g_timer_add(LX_TIMER_INTERVAL, lx_stabilization_timer_task, &lx_state, NULL, NULL);
  }
#endif //end of SIMULATION_ONLY

  /* Start the event loop running - it should never exit. */
  emrun_init(&emrun_opts);
  g_main();
  elog_g(LOG_CRIT,"event loop terminated abnormally!");
  return 1;
}
